#!/usr/local/bin/perl

########
#AutoSim.pl is used to automatically kick off the simple scalar simulation
#process from compiling to simulation. A simulation spec file must be supplied
#to AutoSim with all the relevant infomation.
#Developer : Xun Dong from Computer Science Department of York
#Version   : 1.0
#Platform  : Linux
########

use strict;

########
#Other modules used in this scripts
########
use Getopt::Long;

my $version        = 1.0;
my $sim_root_dir   = "";

my $source_dir = "";
my $sim_config_dir = "";

my $check_ss_version = 'true';
my $auto_spec_file   = "";

my @test_cases;
my @make_cmds;

my %ss_version = qw(
			arm 	3.0
			pisa	3.0
					);

my $log_file = 'autoSim.log';

my $test_accouts = 0;
my $test_failed  = 0;
my @failed_tests;

######## End of the Global Scalar Variables declearation

######## 
#Below is the main procedure of the AutoSim. It would first parse the command line
#arguments and spec files, and create a list of required simulation test case. 
#It will iterate each test case one by one. For each test case it will finish the compile 
#and simulation in one turn, and then move onto the next test case.
########

open (LOG, ">$log_file");

print_header();

parse();

#The autoRun will iterate through all the test case one by one. In each test case 
#autoRun will issue the make command in the order specified by the spec file.
#All the simulation behaviour is out of autoRun's control, the user must make sure 
#all relevant information is provided in the make file correctly and sufficiently.
autoRun();

close (LOG);

print "There are $test_accouts benchmarks run\n";
print "$test_failed of them failed to execute\n";
if($test_failed >0){
	print "These failed are : \n@failed_tests";
}

print "Benchmarks automated run completed\n";

########
#Utility procedures starts from here
########

########
#Function : print_header()
#Other function invoked: None
#Parameter : None
#Description : The procedure will gather environment information on the
#              running machine and print out the information. It will
#              terminate if the required program is not installed
#return : None
########
sub print_header{
	
	my $standard_header = "AutoSim : Automated tool for running simplescalar. Version 1.0\nCopyright (c) by Xun Dong from Computer Science Department of York.\nAll rights reserved. This is for academic use only.\n";
	print LOG $standard_header;
	system ("date");
	print LOG "\nChecking the simulation enviroments\n";
	
}


########
#Function : parse()
#Other function invoked: process_spec; verify_ss_inst;
#Parameter : @ARGV(array of arguments given on the command line)
#Description : This procedure will parse the arguments given on the 
#              commandline and assign relevant value to the scalar variables.
#return : None
########
sub parse{
	my $tests;
	my $commands;
	my $rroot = "";
	
	GetOptions('help|h||?',     =>       sub {print <DATA>; exit ;},
               'version|ver|v', =>       sub {print "AutoSim\n";
		                                    print "AutoSim version is : $version\n";
		                                    exit;
                                            },
               'level',         =>       sub {print "The expected simplescalar for PISA is version : $ss_version{'pisa'} \n";
                                            print "The expected simplescalar for ARM is version : $ss_version{'arm'} \n"; 
				 							exit;
				 							},
			   'initial:s'        =>     \$rroot,
			   'spec:s'         =>	   	 \$auto_spec_file,
               'root=s',        =>       \$sim_root_dir,
		       'src=s',         =>       \$source_dir,
			   'chkss:s',		=> 		 \$check_ss_version,
			   'tests:s'		=>       \$tests,
			   'make:s'		    =>       \$commands									
		      ) or die "can not parse the option correctly, please try inst -h\n";

	if(!($rroot eq "")){
		print "Creating the recommened file system structure for using the AutoSim\n";
		chdir ($rroot) or die "Can not switch to the root directory\n";
		mkdir ("AutoSimulation") or die "Failed to create the AutoSimulation directory or it is already existed\n";
		chdir ("AutoSimulation");
		mkdir ("benchmarks") or die "Failed to create the benchmark directory or it is already existed\n";
		mkdir ("config") or die "Failed to create the config directory or it is already existed\n";
		mkdir ("results") or die "Failed to create the sresult directory or it is already existed\n";
		exit;
	}
	
	#must be no space in the tests attributes										
	@test_cases = split (';', $tests);										
				
	@make_cmds  = split (';', $commands);
											
	#If both command line argument and config file exist. The command line arguments
  	#will have priority.
	if(-e $auto_spec_file ){
		print "Processing the AutoSim Spec file\n";
		process_spec($auto_spec_file);
 	}											
	else {
		die "Error :: No spec file provided or is not exist.\n";	
	}

	(-d $sim_root_dir) or die "Error :: Root directory for the running of the simulation must be provided\n$sim_root_dir is not a directory or not exist\n";
	chdir($sim_root_dir);	
  	
	#verify the arguments provided by the user.
  	$source_dir = $sim_root_dir."\/".$source_dir; 
  	(-d $source_dir) or die "Error :: $source_dir is not a directory or not exist\n";
 	
	if(@test_cases == 0){
		print "AutoSim will execute all the benchmarks\n";
		@test_cases = glob("$source_dir\/*");
		print "@test_cases";		
	}
  	
	if($check_ss_version == 'true'){
		
		#check the version of the simplescalar installed is meet the requirements
		verify_ss_inst() or die "Error : the required simplescalar version is not installed\n";
  	
	}
  	
	#!(-d $sim_config_dir) or die "Error :: $sim_config_dir is not a directory or not exist\n";
  	
	print LOG "Simulation envrionment validated.\n";
	
}

########
#Function : verify_ss_inst()
#Other function invoked: none
#Parameter :  $arm_version, $pisa_version
#Description : This procedure take the string arguments as the simulater's installation parent 
#              directory path. The remaining arguments are the version required.
#return: return 1 if matched, otherwise return 0
########
sub verify_ss_inst{
	#set the variables;
	my @outputs;
	my $arm_ok = 0;
	my $pisa_ok = 0;
	my $rt=1;
	

	#check the arm version simulator
	system("arm-sim-safe -redir:sim sim-safe.help");
	unless (open (CHECK, "sim-safe.help")) {
		die ("can not open sim-safe.help\n");	
	};
		
	#break out of the loop once the version of the simplescalar is found
	while(<CHECK>){
			
		if(/version $ss_version{'arm'}/){
			$arm_ok = 1;
			print "Simplescalar for Arm architecture has the right version\n";
			last;
		}
	}
		
	close(CHECK);
	unlink('sim-safe.help');
		
	#check the pisa version simulator
	system("pisa-sim-safe -redir:sim sim-safe.help");
		
	unless (open (CHECK, "sim-safe.help")) {
		die ("can not open sim-safe.help\n");	
	} ;
		
	#break out of the loop once the version of the simplescalar is found
	while(<CHECK>){
		if(/version $ss_version{'pisa'}/){
			$pisa_ok = 1;
			print "Simplescalar for Pisa architecture has the right version\n";
			last;
		}
	}
		
	close(CHECK);
	unlink('sim-safe.help');
	
	if($arm_ok == 0 || $pisa_ok ==0){
		print "arm: $arm_ok \npisa : $pisa_ok";
		$rt = 0;
		print "The version of the simplescalar is not supported\nVerification failed\n";		
	}

	return $rt;
}

########
#Function : process_spec()
#Other function invoked: none
#Parameter : $spec_file
#Description : This procedure take the string arguments as the config file's path. 
#              It will get the content in the config file and process it according 
#              to its content. The symbol '#' denote the start of the comment. Each 
#              line in the config file is in the format of att = val1;val2...;valN.  
#              It will also create a list of test case the user want
#              to test. The default is to create a permutation of 
#              architecture and all config files. The default is to run every
#              kind of simulator.
#return: syntax error will cause the program to terminate immediately;
########
sub process_spec{
	
	(my $spec_file) = @_;
	
	unless (open(SPEC, $spec_file)) {
		die ("can not open $spec_file \n");	
	};
	
	#process the config file;
	while(<SPEC>){
		#process the config file line by line;
		if(/^\s*#.*/){
			next;
		}elsif(/^\s*$/){ #empty line
			next;
		}elsif(/^\s*root\s*=.+$/i){ #root argument
			my @entry;
			@entry = split('=', $_);
			#remove the space
			$entry[1] =~ s/\s//g;
			$sim_root_dir = $entry[1];
			print "The simulation root directory is $sim_root_dir\n";			
			next;			
		}elsif(/^\s*src\s*=.+$/){ # src argument
			my @entry;
			@entry = split('=', $_);
			#remove the space
			$entry[1] =~ s/\s//g;
			$source_dir = $entry[1];
			print "The source_dir is $source_dir\n";
			next;			
		}elsif(/^\s*chkss\s*=.+$/){ # check simplescalar version
			my @entry;
			@entry = split('=', $_);
			#remove the space
			$entry[1] =~ s/\s//g;
			if (($entry[1] =~ /true/i) || $entry[1] > 0){
				$check_ss_version = 'true';
			}else{
				$check_ss_version = 'false';	
			}
			print "Check the SS version : $check_ss_version\n";
			next;
		}elsif(/^\s*default_ss_pisa_v\s*=.+$/i){ #root argument
			my @entry;
			@entry = split('=', $_);
			#remove the space
			$entry[1] =~ s/\s//g;
			$ss_version{'pisa'} = $entry[1];
			print "The required version of the SS for pisa architecture is : $ss_version{'pisa'}\n";
			next;
		}elsif(/^\s*default_ss_arm_v\s*=.+$/i){ #root argument
			my @entry;
			@entry= split('=', $_);
			#remove the space
			$entry[1] =~ s/\s//g;
			$ss_version{'arm'} = $entry[1];	
			print "The required version of the SS for arm architecture is : $ss_version{'arm'}\n";
			next;
		}
		elsif(/^\s*tests\s*=.+$/i){ #process the test case list
			my @entry;
			my $index = 0 ;
			@entry = split('=', $_);
			
			#extracting the test case. the symbol to seperate is ";"
			@test_cases = split(';', $entry[1]);
			while($index < @test_cases){
				$test_cases[$index] = "$sim_root_dir\/$source_dir\/$test_cases[$index]";
				$test_cases[$index] =~ s/\s//g;
				print "test case : $test_cases[$index]\n";
				$index +=1;
			}
			next;
		}
		elsif(/^\s*make\s*=.+$/i){ #process the make option
			my @entry;
			my $index = 0;
			@entry = split('=', $_);
			$entry[1] =~ s/\n//;
			#extracting the make option. the symbol to seperate is ";"
			@make_cmds = split(';', $entry[1]);
			
			#process the options
			while($index < @make_cmds){
				$make_cmds[$index] = $make_cmds[$index];
				print "make command $index : $make_cmds[$index]\n";
				$index +=1;
			}
			next;
		}
		
	}
	
	close(SPEC);
}


########
#Function : autoRun()
#Other function invoked: none
#Parameter : none but will use the global variables: @make_cmds & @test_cases
#Description :The autoRun will iterate through all the test case one by one. 
#             In each test case autoRun will issue the make command in the 
#             order specified by the spec file. All the simulation behaviour 
#             is out of autoRun's control, the user must make sure all relevant 
#             information is provided in the make file correctly and sufficiently. 
#return: none
########
sub autoRun{
	my $failed = 0;
	my $index  = 0;
	foreach my $test (@test_cases){
			
		if(-d $test){
			$test_accouts++;
			print LOG "Start the simulation at $test\n";
			#change to the working directory
			chdir ($test);
		
			system ("pwd");
			#issue the make command in order
			foreach my $command (@make_cmds){
				#invest how to add error detection
				system ($command) && $failed++;
				if($failed >0){
					$test_failed++;
					$failed_tests[$index] = $test."\n";
					$index++;
					$failed = 0;
					last;
				}
			}
						
			print LOG "The test for : $test has finished\n";	
		}			
	}
}

########
# The following section defines the info printed out if the user requests
# help.
########
__DATA__
"autotest" perl script

This perl scripts will automatically compile the benchmarks and run the
simplescalar simulator against the complied benchmarks according to the
configurations provided. The result will be stored in a certain way as the
user specified.

Synopsis:
autotest.pl mandatory_arguments [option ...]

General
=======
-h|help||?              Display this help and exit
-v|ver|version          Display the program version and exit

Simple Scalar Version used
==================

--level                 Return the version of the simple scalar used in this
                        test.

Test result directory
====================

--root                  Specify the root directory used for the simulation.
--src                   Specify the benchmark directory's name under $root/src
--chkss                 The option to check the version of the simplescalar before 
                        running the simulation test.
--inst_dir_arm          Specify the directory in which the arm simplescalar is installed.
                        eg. /usr/local/pkg
--inst_dir_pisa         Specify the directory in which the pisa simplescalar is installed.
                        eg. /usr/local/pkg
--make                  Give the command which used to call the GNU make. It can
                        be multiple commands, but must use ';' to seperate.
--spec                  Specify the path to get the automated simulation 
                        specification file.
--tests                 Specify the benchmarks to simulate
