/*
 * include/asm-arm/arch-ebsa285/irq.h
 *
 * Copyright (C) 1996-1998 Russell King
 *
 * Changelog:
 *  22-Aug-1998	RMK	Restructured IRQ routines
 *  03-Sep-1998	PJB	Merged CATS support
 *  20-Jan-1998	RMK	Started merge of EBSA286, CATS and NetWinder
 *  26-Jan-1999	PJB	Don't use IACK on CATS
 *  01-Feb-1999 PJB	ISA IRQs start at 0 not 16
 */
#include <asm/hardware.h>
#include <asm/dec21285.h>
#include <asm/irq.h>

/*
 * Footbridge IRQ translation table
 *  Converts from our IRQ numbers into FootBridge masks
 */
static int dc21285_irq_mask[] = {
	IRQ_MASK_UART_RX,	/*  0 */
	IRQ_MASK_UART_TX,	/*  1 */
	IRQ_MASK_TIMER1,	/*  2 */
	IRQ_MASK_TIMER2,	/*  3 */
	IRQ_MASK_TIMER3,	/*  4 */
	IRQ_MASK_IN0,		/*  5 */
	IRQ_MASK_IN1,		/*  6 */
	IRQ_MASK_IN2,		/*  7 */
	IRQ_MASK_IN3,		/*  8 */
	IRQ_MASK_DOORBELLHOST,	/*  9 */
	IRQ_MASK_DMA1,		/* 10 */
	IRQ_MASK_DMA2,		/* 11 */
	IRQ_MASK_PCI,		/* 12 */
	IRQ_MASK_SDRAMPARITY,	/* 13 */
	IRQ_MASK_I2OINPOST,	/* 14 */
	IRQ_MASK_PCI_ERR	/* 15 */
};

static int isa_irq;

static inline int fixup_irq(unsigned int irq)
{
	if (irq == isa_irq) {
		if (machine_is_cats()) {
			unsigned int status = inb(0x20) | (inb(0xa0) << 8);
			status &= ~(1<<2);
			for (irq = 0; irq < 16; irq++) {
				if (status & 1)
					break;
				status >>= 1;
			}
		} else
			irq = *(unsigned char *)PCI_IACK;
	}

	return irq;
}

static void dc21285_mask_irq(unsigned int irq)
{
	*CSR_IRQ_DISABLE = dc21285_irq_mask[irq & 15];
}

static void dc21285_unmask_irq(unsigned int irq)
{
	*CSR_IRQ_ENABLE = dc21285_irq_mask[irq & 15];
}

static void isa_mask_pic_lo_irq(unsigned int irq)
{
	unsigned int mask = 1 << (irq & 7);

	outb(inb(PIC_MASK_LO) | mask, PIC_MASK_LO);
}

static void isa_mask_ack_pic_lo_irq(unsigned int irq)
{
	unsigned int mask = 1 << (irq & 7);

	outb(inb(PIC_MASK_LO) | mask, PIC_MASK_LO);
	outb(0x20, PIC_LO);
}

static void isa_unmask_pic_lo_irq(unsigned int irq)
{
	unsigned int mask = 1 << (irq & 7);

	outb(inb(PIC_MASK_LO) & ~mask, PIC_MASK_LO);
}

static void isa_mask_pic_hi_irq(unsigned int irq)
{
	unsigned int mask = 1 << (irq & 7);

	outb(inb(PIC_MASK_HI) | mask, PIC_MASK_HI);
}

static void isa_mask_ack_pic_hi_irq(unsigned int irq)
{
	unsigned int mask = 1 << (irq & 7);

	outb(inb(PIC_MASK_HI) | mask, PIC_MASK_HI);
	outb(0x62, PIC_LO);
	outb(0x20, PIC_HI);
}

static void isa_unmask_pic_hi_irq(unsigned int irq)
{
	unsigned int mask = 1 << (irq & 7);

	outb(inb(PIC_MASK_HI) & ~mask, PIC_MASK_HI);
}

static void no_action(int cpl, void *dev_id, struct pt_regs *regs)
{
}

static struct irqaction irq_cascade = { no_action, 0, 0, "cascade", NULL, NULL };

static __inline__ void irq_init_irq(void)
{
	int irq, first_irq;

	*CSR_IRQ_DISABLE = -1;
	*CSR_FIQ_DISABLE = -1;

	switch (machine_type) {
	case MACH_TYPE_EBSA285:
		isa_irq = -1;
		first_irq = 16;
		break;

	case MACH_TYPE_CATS:
		outb(0x11, PIC_LO);
		outb(0, PIC_MASK_LO);		/* IRQ IACK not used */
		outb(0x04, PIC_MASK_LO);
		outb(0x01, PIC_MASK_LO);
		outb(0xff, PIC_MASK_LO);
		outb(0x68, PIC_LO);
		outb(0x0a, PIC_LO);

		outb(0x11, PIC_HI);
		outb(0, PIC_MASK_HI);		/* IRQ IACK not used */
		outb(0x02, PIC_MASK_HI);
		outb(0x01, PIC_MASK_HI);
		outb(0xff, PIC_MASK_HI);
		outb(0x68, PIC_HI);
		outb(0x0a, PIC_HI);

		isa_irq = IRQ_IN2;
		first_irq = 0;
		break;

	case MACH_TYPE_NETWINDER:
		outb(0x11, PIC_LO);
		outb(0x00, PIC_MASK_LO);	/* IRQ_IACK starts at 0 */
		outb(0x04, PIC_MASK_LO);
		outb(0x01, PIC_MASK_LO);
		outb(0xff, PIC_MASK_LO);

		outb(0x11, PIC_HI);
		outb(0x08, PIC_MASK_HI);	/* IRQ_IACK starts at 8 */
		outb(0x02, PIC_MASK_HI);
		outb(0x01, PIC_MASK_HI);
		outb(0xff, PIC_MASK_HI);

		isa_irq = IRQ_IN3;
		first_irq = 0;
		break;
	}

	for (irq = first_irq; irq < NR_IRQS; irq++) {
		irq_desc[irq].valid	= 1;
		irq_desc[irq].probe_ok	= 1;
		if (irq < 8) {
			irq_desc[irq].mask_ack	= isa_mask_ack_pic_lo_irq;
			irq_desc[irq].mask	= isa_mask_pic_lo_irq;
			irq_desc[irq].unmask	= isa_unmask_pic_lo_irq;
		} else if (irq < 16) {
			irq_desc[irq].mask_ack	= isa_mask_ack_pic_hi_irq;
			irq_desc[irq].mask	= isa_mask_pic_hi_irq;
			irq_desc[irq].unmask	= isa_unmask_pic_hi_irq;
		} else {
			irq_desc[irq].mask_ack	= dc21285_mask_irq;
			irq_desc[irq].mask	= dc21285_mask_irq;
			irq_desc[irq].unmask	= dc21285_unmask_irq;
		}
	}

	if (isa_irq != -1) {
		request_region(PIC_LO, 2, "pic1");
		request_region(PIC_HI, 2, "pic2");
		setup_arm_irq(IRQ_ISA_CASCADE, &irq_cascade);
		setup_arm_irq(isa_irq, &irq_cascade);
	}
}
