/*
 * linux/include/asm-arm/arch-clps7110/hardware.h
 *
 * Written 1998,1999 by Werner Almesberger
 *
 * This file contains Linux-specific hardware definitions for the CL-PS7110
 * architecture and #includes hardware definitions of the CL-PS7110 chip,
 * plus architecture-specific hardware definitions.
 */

#ifndef __ASM_ARCH_HARDWARE_H
#define __ASM_ARCH_HARDWARE_H

#include <linux/config.h>

#ifdef CONFIG_ARCH_PSION
#include <asm/arch/hw_psion.h>
#elif CONFIG_ARCH_GEOFOX
#include <asm/arch/hw_geofox.h>
#else
#error Unknown architecture
#endif

/*
 * Our memory model (generalized):
 *
 * Physical             Size    Type                    Virtual
 * (_START)             (_SIZE)                         (_BASE)
 * 00000000-7fffffff	   2GB	don't care		n/a
 * 40000000-4ffffff	 256MB	CompactFlash (Psion)	f0000000
 * 70000000-7fffffff     256MB  PCMCIA (Geofox)		f0000000
 * 80000000-80000fff       4kB  internal registers      d0000000
 * 80001000-bfffffff    1020kB  not used                n/a
 * c0000000-ffffffff	1024MB	RAM, non-contiguous	c0000000 (1)
 *
 * (1) Psion: contiguous in virtual memory; Geofox: not yet contiguous, but
 *     migration to RPC-like approach planned
 */

/*
 * Start conventions for the kernel:
 *
 * R0 = 0	head-armv.S sanity check
 * R1 = 8	CL-PS7110
 */

/*
 * Memory map when kernel is started:
 *
 * Physical location  Offset in   Size	 Description
 * start    end	      boot file
 * -------- --------  ----------  -----  --------------------------------------
 * c0000000-c001ffff		  128kB  Video RAM (1)
 * c0020000-c00207ff		    2kB  frag_p2v_lookup_table
 * c0020800-c00209ff		  0.5kB  frag_v2p_lookup_table
 * c0020a00-c0020fff		  1.5kB	 currently unused (2)
 * c0021000-c0022fff		    8kB  currently unused (3)
 * c0023000-c00235ff   00000800	  1.5kB	 parameters (asm/setup.h:param_struct)
 * c0023600-c00237ff   00000e00	  0.5kB	 list of available memory areas (4)
 * c0023800-c0023fff		    2kB  currently unused
 * c0024000-c0033fff		   64kB	 page tables (5)
 * c0034000-c0037fff		   16kB	 swapper_pg_dir (5)
 * c0038000-...	       00002000		 start of text segment, then initrd
 *
 * (1) 0-12bff for Psion (640x240), 0-18fff for Geofox (640x320). Unused pages
 *     can be given to VM.
 * (2) Architectures without KMEM_IS_VIRTUAL can use page c002 for VM.
 * (3) Can be used for VM.
 * (4) Either contained in boot file or provided by relocator.
 * (5) Initialized by head-armv.S
 *
 * Note: Boot file locations 0x0000-0x07ff contain the relocator, locations
 *	 0x1000-0x1fff contain a scratch page which is only used during
 *	 relocation. The physical location of either depends on where EPOC32
 *	 loads them.
 */

/*
 * IO address mapping
 */

#define IO_BASE			0xd0000000	/* virtual */
#define IO_START		0x80000000	/* physical */
#define IO_SIZE			0x00100000	/* actually it's only 0x1000,
						   but this way we can use a
						   section to map it and don't
						   waste an entire page table*/

/*
 * PCMCIA address mapping
 *
 * Seems that all CL-PS7110 based devices have only a single PCMCIA/CF slot,
 * although the architecture seems to allow for up to eight of them.
 *
 * PCMCIA_START depends on the machine architecture.
 */

#define PCMCIA_BASE		0xf0000000	/* virtual */
#define PCMCIA_SIZE		0x10000000	/* 4 areas of 64MB each */

/*
 * RAM definitions
 */

#define RAM_START		0xc0000000	/* physical */

/*
 * KERNTOPHYS does *not* translate from kernel addresses to physical addresses.
 * Instead, it translates from addresses of kernel objects (code, data, etc.)
 * to addresses of memory, both in kernel address space. In our case, this is a
 * no-op.
 *
 * This KERNTOPHYS is only used in arch/arm/mm/init.c
 */

#define KERNTOPHYS(x)		((unsigned long) &(x))

#define KERNEL_BASE		0xc0038000
#define PARAMS_BASE		0xc0023000
/*#define Z_PARAMS_BASE		@@@*/
#define FLUSH_BASE_PHYS         0	/* ROM @@@ ?*/


/*
 * LCD display definitions.
 *
 * LCD_X_SIZE and LCD_Y_SIZE depend on the machine architecture.
 */

#define LCD_BASE		0xc0000000
#define LCD_MEM_SIZE		(LCD_X_SIZE*LCD_Y_SIZE/2)

/*
 * Register definitions
 */

#include <asm/arch-clps7110/clps7110.h>

/*
 * Allow access to IO registers as if they were normal variables
 */

#define __IOB(x)	(*(volatile unsigned char *) (IO_BASE+(x)))
#define __IOL(x)	(*(volatile unsigned long *) (IO_BASE+(x)))

#define IO_PADR		__IOB(PADR)
#define IO_PBDR		__IOB(PBDR)
#define IO_PCDR		__IOB(PCDR)
#define IO_PDDR		__IOB(PDDR)
#define IO_PADDR	__IOB(PADDR)
#define IO_PBDDR	__IOB(PBDDR)
#define IO_PCDDR	__IOB(PCDDR)
#define IO_PDDDR	__IOB(PDDDR)
#define IO_PEDR		__IOB(PEDR)
#define IO_PEDDR	__IOB(PEDDR)
#define IO_SYSCON	__IOL(SYSCON)
#define	IO_SYSFLG	__IOL(SYSFLG)
#define	IO_MEMCFG1	__IOL(MEMCFG1)
#define	IO_MEMCFG2	__IOL(MEMCFG2)
/* DRFPR */
#define IO_INTSR	__IOL(INTSR)
#define IO_INTMR	__IOL(INTMR)
#define	IO_LCDCON	__IOL(LCDCON)
#define IO_TC1D		__IOL(TC1D)
#define IO_TC2D		__IOL(TC2D)
#define IO_RTCDR	__IOL(RTCDR)
#define IO_RTCMR	__IOL(RTCMR)
/* PMPCON */
#define IO_CODR 	__IOB(CODR)
#define IO_UARTDR	__IOL(UARTDR)
#define IO_UBRLCR	__IOL(UBRLCR)
#define IO_SYNCIO	__IOL(SYNCIO)
#define	IO_PALLSW	__IOL(PALLSW)
#define	IO_PALMSW	__IOL(PALMSW)
#define IO_STFCLR	__IOL(STFCLR)
#define IO_BLEOI	__IOL(BLEOI)
#define IO_MCEOI	__IOL(MCEOI)
#define IO_TEOI		__IOL(TEOI)
#define IO_TC1EOI	__IOL(TC1EOI)
#define IO_TC2EOI	__IOL(TC2EOI)
#define IO_RTCEOI	__IOL(RTCEOI)
#define IO_UMSEOI	__IOL(UMSEOI)
#define IO_COEOI	__IOL(COEOI)
#define IO_HALT		__IOL(HALT)
#define IO_STDBY	__IOL(STDBY)

#endif
