/*
 * linux/include/asm-arm/proc-armo/uaccess.h
 *
 * Copyright (C) 1996 Russell King
 * Copyright (C) 1998 Philip Blundell
 */

#include <asm/io.h>

/*
 * The fs functions are implemented on the ARM2 and ARM3 architectures
 * manually.  Accesses in USER_DS are allowed to fault; we do them with ldrt
 * instructions so that page permissions are respected.  Accesses in KERNEL_DS
 * are not expected to fault and the kernel will oops if this ever happens.
 * We do these with plain ldr instructions and you can access any memory this
 * way.
 */

/*
 * These are the values used to represent the user `fs' and the kernel `ds'
 */
#define KERNEL_DS	0x03000000
#define USER_DS   	0x02000000

extern __inline__ void set_fs (mm_segment_t fs)
{
	current->addr_limit = fs;
}

#define get_ds()	(KERNEL_DS)
#define get_fs()	(current->addr_limit)
#define segment_eq(a,b)	((a) == (b))

#define __range_ok(addr,size) ({					\
	unsigned long flag, sum;					\
	__asm__ __volatile__("subs %1, %0, %3; cmpcs %1, %2; movcs %0, #0" \
		: "=&r" (flag), "=&r" (sum)				\
		: "r" (addr), "Ir" (size), "0" (current->addr_limit)	\
		: "cc");						\
	flag; })

#define __addr_ok(addr) ({						\
	unsigned long flag;						\
	__asm__ __volatile__("cmp %2, %0; movlo %0, #0"			\
		: "=&r" (flag)						\
		: "0" (current->addr_limit), "r" (addr)			\
		: "cc");						\
	(flag == 0); })

#define access_ok(type,addr,size) (__range_ok(addr,size) == 0)

/* Emit the exception fixup information for an ARM3 user access. 
   "bad" is the instruction that may fault.  "fix" is the place to go
   after catching the exception, "res" is the result register that we
   set to zero and "err" is the register to load -EFAULT into.  If either
   "err" or "res" is set to r14 then effectively nothing happens (they're
   still loaded, but r14 contains nothing useful after an abort anyway). */
#define EXCEPT(bad,fix,err,res)						\
	".section __ex_table,\"a\"\n"					\
	"	.align 3\n"						\
	"	.word	" bad "\n"					\
	"	ldr	" err ", [ " res ", $(" fix " - " bad ")/4]\n"	\
	".previous\n"

/* Do user accesses.  If the address is above 0x02000000 then it's in
   physical RAM and only allowed in kernel context, so we can bypass the
   checks.  If it's in logical RAM we might be trying to do the access
   in user context so we must use translation; MEMC will trap bad 
   accesses.  These macros get inlined so we want to keep them small.  */
#define __put_user_asm_byte(x,addr,err)					\
	__asm__ __volatile__(						\
		"tst	%2, $0x02000000		@ put_user_asm_byte\n"	\
	"	strneb	%1, [%2]\n"					\
	"1:	streqbt	%1, [%2]\n"					\
	"2:\n"								\
	EXCEPT("1b", "2b", "%0", "lr")					\
	: "=r" (err)							\
	: "r" (x), "r" (addr), "0" (err) : "lr", "cc")

#define __put_user_asm_half(x,addr,err)					\
	do {								\
	register unsigned int _tmp, _tmp2;				\
	__asm__ __volatile__(						\
		"tst	%4, $0x02000000		@ put_user_asm_half\n"	\
	"	mov	%2, %3, lsr #8\n"				\
	"	strneb	%3, [%4]\n"					\
	"	strneb	%2, [%4, #1]\n"					\
	"	addeq	%1, %4, #1\n"					\
	"1:	streqbt	%3, [%4]\n"					\
	"2:	streqbt	%2, [%1]\n"					\
	"3:\n"								\
	EXCEPT("1b", "3b", "%0", "lr")					\
	EXCEPT("2b", "3b", "%0", "lr")					\
	: "=r" (err), "=&r" (_tmp), "=&r" (_tmp2)			\
	: "r" (x), "r" (addr), "0" (err) : "lr", "cc");			\
	} while (0)

#define __put_user_asm_word(x,addr,err)					\
	__asm__ __volatile__(						\
		"tst	%2, $0x02000000		@ put_user_asm_word\n"	\
	"	strne	%1, [%2]\n"					\
	"1:	streqt	%1, [%2]\n"					\
	"2:\n"								\
	EXCEPT("1b", "2b", "%0", "lr")					\
	: "=r" (err)							\
	: "r" (x), "r" (addr), "0" (err) : "lr", "cc")

#define __get_user_asm_byte(x,addr,err)					\
	__asm__ __volatile__(						\
		"tst	%2, $0x02000000		@ get_user_asm_byte\n"	\
	"	ldrneb	%1, [%2]\n"					\
	"1:	ldreqbt	%1, [%2]\n"					\
	"2:\n"								\
	EXCEPT("1b", "2b", "%0", "%1")					\
	: "=r" (err), "=r" (x)						\
	: "r" (addr), "0" (err) : "lr", "cc")

#define __get_user_asm_half(x,addr,err)					\
	do {								\
	register unsigned int _tmp, _tmp2;				\
	__asm__ __volatile__(						\
		"tst	%4, $0x02000000		@ get_user_asm_half\n"	\
	"	ldrneb	%1, [%4]\n"					\
	"	ldrneb	%2, [%4, #1]\n"					\
	"	addeq	%3, %4, #1\n"					\
	"1:	ldreqbt %1, [%4]\n"					\
	"2:	ldreqbt %2, [%3]\n"					\
	"	orr	%1, %1, %2, lsl #8\n"				\
	"3:\n"								\
	EXCEPT("1b", "3b", "%0", "%1")					\
	EXCEPT("2b", "3b", "%0", "%1")					\
	: "=r" (err), "=r" (x), "=&r" (_tmp), "=&r" (_tmp2)		\
	: "r" (addr), "0" (err) : "lr", "cc");				\
	} while (0)

#define __get_user_asm_word(x,addr,err)					\
	__asm__ __volatile__(						\
		"tst	%2, $0x02000000		@ get_user_asm_word\n"	\
	"	ldrne	%1, [%2]\n"					\
	"1:	ldreqt	%1, [%2]\n"					\
	"2:\n"								\
	EXCEPT("1b", "2b", "%0", "%1")					\
	: "=r" (err), "=r" (x)						\
	: "r" (addr), "0" (err) : "lr", "cc")

#define __get_user3_asm_word(x,addr,where)				\
	__asm__ __volatile__(						\
		"tst	%1, $0x02000000		@ get_user3_asm_word\n"	\
	"	ldrne	%0, [%1]\n"					\
	"1:	ldreqt	%0, [%1]\n"					\
	"2:\n"								\
	EXCEPT("1b", "%2", "lr", "%0")					\
	: "=r" (x)							\
	: "r" (addr), "m" (where) : "lr", "cc")

/* More complicated operations.  These require more code and so we don't
   inline them.  We still want them to be fast though. :-)  */
extern unsigned long __arch_copy_from_user(void *to, const void *from, unsigned long n);
#define __do_copy_from_user(to,from,n)				\
	(n) = __arch_copy_from_user(to, from, n)

extern unsigned long __arch_copy_to_user(void *to, const void *from, unsigned long n);
#define __do_copy_to_user(to,from,n)				\
	(n) = __arch_copy_to_user(to, from, n)

extern unsigned long __arch_clear_user(void *addr, unsigned long n);
#define __do_clear_user(addr,sz)				\
	(sz) = __arch_clear_user(addr, sz)

extern unsigned long __arch_strncpy_from_user(char *to, const char *from, unsigned long count);
#define __do_strncpy_from_user(dst,src,count,res)		\
	(res) = __arch_strncpy_from_user(dst, src, count)

extern unsigned long __arch_strlen_user(const char *s);
#define __do_strlen_user(s,res)					\
	(res) = __arch_strlen_user(s)
