package org.zeyda.clawcircus.utils;

import org.zeyda.clawcircus.Data.Diagram.Block;
import org.zeyda.clawcircus.Data.Diagram.PortBlock;
import org.zeyda.clawcircus.Data.Diagram.SubSystem;
import org.zeyda.clawcircus.Data.Diagram.ResolveMethod;
import org.zeyda.clawcircus.Data.Diagram.Port;
import org.zeyda.clawcircus.Data.Diagram.PortDir;

import org.zeyda.clawcircus.Data.ClaSP.BlockWiring;
import org.zeyda.clawcircus.Data.ClaSP.Flow;
import org.zeyda.clawcircus.Data.ClaSP.Enabled;
import org.zeyda.clawcircus.Data.ClaSP.Signal;

import org.zeyda.clawcircus.Toolbox.ClaSPLibrary;

import org.zeyda.clawcircus.IO.ClaSP.jaxb.*;

import org.zeyda.clawcircus.collections.*;
import org.zeyda.clawcircus.collections.impl.*;

import javax.xml.bind.JAXBElement;

import java.util.List;

/* There is an open issue in this class with regards to raising semantic
 * errors if signal instances cannot be constructed due to actual Ports in
 * the diagram not being connected. Currently this results in null elements
 * being inserted into inps and outs, this is crucial in order to maintain
 * the significance of element positions in these lists as port numbers
 * (in rinps and pouts unconnected ports are simply ignored). */

public class ClaSPUtils {
   public static BlockWiring calcBlockWiringFromLib(Block block) {
      BlockWiringXml block_wiring_xml =
         ClaSPLibrary.getBlockWiring(block.getBlockType());
      if (block_wiring_xml == null) {
         throw new IllegalArgumentException(
            "Couldn't generate BlockWiring for Block: " + block.toString());
      }
      SignalList inps = generateInps(block_wiring_xml, block);
      SignalList outs = generateOuts(block_wiring_xml, block);
      FlowSet flows = generateFlows(block_wiring_xml, block);
      return new BlockWiring(inps, outs, flows);
   }

   private static SignalList generateInps(BlockWiringXml block_wiring_xml,
      Block block) {
      SignalList result = new SignalListImpl();
      String inps = block_wiring_xml.getInps();
      if (inps != null) {
         inps = inps.trim().toLowerCase();
         assert
            inps.equals("varlength") || MiscUtils.canBeParsedAsInt(inps);
         int inps_num;
         if (inps.equals("varlength")) {
            /* Infer signal list dynamically from block. */
            inps_num = block.getInputPortsNum();
         }
         else {
            /* Infer signal list statically from library. */
            inps_num = Integer.parseInt(inps);
         }
         /* Consistency check: library signature vs. instantiated block. */
         if (block.getInputPortsNum() != inps_num) {
            throw new AssertionError(
               "Inconsistency in number of input ports: ClaSP block " +
               "library vs. actual inputs of instantiated Block.");
         }
         for (int index = 1; index <= inps_num; index++) {
            if (block.getInputPort(index).isConnected()) {
               result.add(block.getInputPort(index).getLink().getSignal());
            }
            else {
               result.add(null); /* Raise a semantic error here?! */
            }
         }
      }
      return result;
   }

   private static SignalList generateOuts(BlockWiringXml block_wiring_xml,
      Block block) {
      SignalList result = new SignalListImpl();
      String outs = block_wiring_xml.getOuts();
      if (outs != null) {
         outs = outs.trim().toLowerCase();
         assert
            outs.equals("varlength") || MiscUtils.canBeParsedAsInt(outs);
         int outs_num;
         if (outs.equals("varlength")) {
            /* Infer signal list dynamically from block. */
            outs_num = block.getOutputPortsNum();
         }
         else {
            /* Infer signal list statically from library. */
            outs_num = Integer.parseInt(outs);
         }
         /* Consistency check: library signature vs. instantiated block. */
         if (block.getOutputPortsNum() != outs_num) {
            System.out.println(block.toString());
            throw new AssertionError(
               "Inconsistency in number of output ports: ClaSP block " +
               "library vs. actual outputs of instantiated Block.");
         }
         for (int index = 1; index <= outs_num; index++) {
            if (block.getOutputPort(index).isConnected()) {
               result.add(block.getOutputPort(index).getLink().getSignal());
            }
            else {
               result.add(null); /* Raise a semantic error here?! */
            }
         }
      }
      return result;
   }

   private static FlowSet generateFlows(BlockWiringXml block_wiring_xml,
      Block block) {
      FlowSet result = new FlowSetImpl();
      for(FlowXml flow : block_wiring_xml.getFlows().getFlow()) {
         result.add(generateFlow(flow, block));
      }
      return result;
   }

   private static Flow generateFlow(FlowXml flow, Block block) {
      Enabled enabled = getEnabled(flow, block);
      boolean ordered = getOrdered(flow, block);
      SignalSet rinps = getRInps(flow, block);
      SignalSet pouts = getPOuts(flow, block);
      return new Flow(enabled, ordered, rinps, pouts);
   }

   private static Enabled getEnabled(FlowXml flow, Block block) {
      if (flow.getEnabled().isAlways()) {
         return Enabled.ALWAYS;
      }
      else {
         return
            new Enabled(getSignals(flow.getEnabled(), block, PortDir.INPUT));
      }
   }

   private static boolean getOrdered(FlowXml flow, Block block) {
      return flow.isOrdered();
   }

   private static SignalSet getRInps(FlowXml flow, Block block) {
      return getSignals(flow.getRinps(), block, PortDir.INPUT);
   }

   private static SignalSet getPOuts(FlowXml flow, Block block) {
      return getSignals(flow.getPouts(), block, PortDir.OUTPUT);
   }

   private static SignalSet getSignals(PortSpecXml port_spec, Block block,
      PortDir dir) {
      SignalSet result = new SignalSetImpl();
      for (int port_num : port_spec.getPort()) {
         Port port = getPort(block, port_num, dir);
         if (port.isConnected()) {
            result.add(port.getLink().getSignal());
         }
         else {
            /* Raise some semantic error here?! */
         }
      }
      for (JAXBElement<List<Integer>> jaxb_elem : port_spec.getPortList()) {
         List<Integer> port_list = jaxb_elem.getValue();
         for (int port_num : port_list) {
            Port port = getPort(block, port_num, dir);
            if (port.isConnected()) {
               result.add(port.getLink().getSignal());
            }
            else {
               /* Raise some semantic error here?! */
            }
         }
      }
      for (PortRangeXml port_range : port_spec.getPortRange()) {
         Integer from = port_range.getFrom();
         if (from == null) {
            from = 1;
         }
         Integer to = port_range.getTo();
         if (to == null) {
            switch (dir) {
            case INPUT:
               to = block.getInputPortsNum();
               break;

            case OUTPUT:
               to = block.getOutputPortsNum();
               break;
            }
         }
         for (int port_num = from; port_num <= to; port_num++) {
            Port port = getPort(block, port_num, dir);
            if (port.isConnected()) {
               result.add(port.getLink().getSignal());
            }
            else {
               /* Raise some semantic error here?! */
            }
         }
      }
      for (String port_pattern : port_spec.getPortPattern()) {
         throw new AssertionError(
            "Processing of <portPattern>..</portPattern> not supported yet.");
      }
      return result;
   }

   private static Port getPort(Block block, int port_num, PortDir dir) {
      switch (dir) {
      case INPUT:
         return block.getInputPort(port_num);

      case OUTPUT:
         return block.getOutputPort(port_num);

      default:
         throw new AssertionError();
      }
   }

   public static Signal getSignalFromPort(Port port, ResolveMethod method) {
      Signal signal = null;
      switch(method) {
         case INTERNAL:
            assert port.getBlock().isSubSystem();
            PortBlock port_block =
               ((SubSystem) port.getBlock()).resolveIntoBlock(port);
            if (port_block != null) {
               signal = port_block.getSignal();
            }
            break;

         case EXTERNAL:
            if (port.isConnected()) {
               signal = port.getLink().getSignal();
            }
            break;

         default:
            throw new AssertionError();
      }
      return signal;
   }

   public static void printClaSPModel(SubSystem system) {
      System.out.println("[ClaSP Model]");
      System.out.println(" SubSystem: " + system.getPath());
      for (Block block : system) {
         if (block.getClass().getName().startsWith(
            "org.zeyda.clawcircus.Data.Diagram.BlockTypesGen")) {
            System.out.println("---");
            System.out.print("Block: " + StringUtils.quote(block.getName()));
            System.out.print(" (" + block.getBlockType() + ")");
            System.out.println();
            BlockWiring block_wiring = block.getBlockWiring();
            if (block_wiring == null) {
               throw new AssertionError("BlockType " + block.getBlockType()
                  + " not in ClaSP library as it should.");
            }
            System.out.print(block_wiring.toString());
         }
      }
      for (Block block : system) {
         if (block.isSubSystem()) {
            System.out.println();
            printClaSPModel((SubSystem) block);
         }
      }
   }
}
