package org.zeyda.clawcircus.Toolbox;

import org.zeyda.clawcircus.Data.Simulink.*;

import java.util.List;
import java.util.LinkedList;

public class MdlClawZAdaptor {
   private static boolean verbose_output;

   static {
      disableVerboseOutput();
   }

   public static void invoke(MdlBlock model) {
      discardProblematicBlocks(model);
      populateBlockParameterDefaults(model);
   }

   private static void discardProblematicBlocks(MdlBlock model) {
      List<MdlBlock> culprits = new LinkedList<MdlBlock>();
      for(MdlElement element : model) {
         if(element.isBlock()) {
            MdlBlock block = (MdlBlock) element;
            if (checkForProblematicSymbols(block)) {
               culprits.add(block);
            }
         }
      }
      /* We have to do the removal in a separate step as we cannot modify the
       * underlying collection while still iterating over it. */
      for(MdlBlock block : culprits) {
         /* Note the following doesn't actually remove the block from culprits
          * but from its context in the model. */
         block.remove();
         printVerbose("Removed Simulink block " + block.getName() +
            " (contains $ character(s))");
      }
   }

   private static boolean checkForProblematicSymbols(MdlBlock block) {
      return block.serialise().contains("$");
   }

   private static void populateBlockParameterDefaults(MdlBlock model) {
      MdlBlock bpd = model.getBlock("BlockParameterDefaults");
      if (bpd == null) {
         printVerbose("BlockParameterDefaults block not present, " +
            "no propagation was performed.");
         return ;
      }
      MdlBlock system = model.getBlock("System");
      if (system == null) {
         System.out.println("System block not present, no propagation was " +
            "performed.");
         return ;
      }
      for(MdlElement element : bpd) {
         if (element.isBlock()) {
            MdlBlock block = (MdlBlock) element;
            if(block.getName().equals("Block")) {
               MdlAttribute block_type = block.getAttribute("BlockType");
               if (block_type == null) {
                  continue; /* Maybe display a warning message here. */
               }
               String block_type_value = block_type.getValue();
               if (verbose_output) {
                  printVerbose("Populating default attributes for BlockType "
                     + block_type_value);
               }
               for(MdlElement attr : block) {
                  if (attr.isAttribute() && attr != block_type) {
                     infuseAttribute(
                        system, block_type_value, (MdlAttribute) attr);
                  }
               }
            }
         }
      }
   }

   private static void infuseAttribute(MdlBlock block, String block_type,
      MdlAttribute attr) {
      if (block.getName().equals("Block")) {
         if (block.hasAttribute("BlockType") &&
            block.getAttributeValue("BlockType").equals(block_type)) {
            if (!block.hasAttribute(attr.getName())) {
               block.add((MdlAttribute) attr.clone()); /* Clone here! */
            }
         }
      }
      /* Infusing of attributes is performed recursively. */
      for(MdlElement element : block) {
         if(element.isBlock()) {
            infuseAttribute((MdlBlock) element, block_type, attr);
         }
      }
   }

   private static void printVerbose(String text) {
      if (verbose_output) {
         System.out.println(text);
      }
   }

   public static void enableVerboseOutput() {
      verbose_output = true;
   }

   public static void disableVerboseOutput() {
      verbose_output = false;
   }
}
