package org.zeyda.clawcircus.Parser;

import org.zeyda.clawcircus.Data.Simulink.*;

import org.zeyda.clawcircus.utils.StringUtils;

import org.zeyda.clawcircus.exceptions.ParserException;

import java.util.List;
import java.util.ArrayList;

public class MdlParser {
   protected MdlLexer lexer;

   public MdlParser(MdlLexer lexer) {
      assert lexer != null;
      this.lexer = lexer;
      lexer.reset();
   }

   public MdlBlock parseMdl() throws ParserException {
      lexer.reset();
      Token name = lexer.current();
      if (!(name.isValidName() && name.getText().equals("Model"))) {
         throw new ParserException("\"Model\" expected here", name);
      }
      return parseBlock();
   }

   public List<MdlBlock> parseLmf() throws ParserException {
      List<MdlBlock> block_spec_list = new ArrayList<MdlBlock>();
      lexer.reset();
      while(!lexer.eof()) {
         Token name = lexer.current();
         if (!(name.isValidName() &&
            name.getText().equals("BlockSpecification"))) {
            throw new ParserException(
               "\"BlockSpecification\" expected here", name);
         }
         block_spec_list.add(parseBlock());
      }
      return block_spec_list;
   }

   protected MdlBlock parseBlock() throws ParserException {
      Token name = lexer.consume();
      if (!name.isValidName()) {
         throw new ParserException("valid block name expected here", name);
      }
      if (!lexer.consume().isLCURLY()) {
         throw new ParserException("{ expected here", lexer.last());
      }
      MdlBlock block = new MdlBlock(name.toString());
      while (!lexer.current().isRCURLY()) {
         name = lexer.current();
         if (!name.isValidName()) {
            throw new ParserException(
               "valid attribute / block name or } expected here", name);
         }
         if (lexer.next().isLCURLY()) {
            block.add(parseBlock());
         }
         else {
            block.add(parseAttribute());
         }
      }
      lexer.consume(); /* Consume closing curly bracket. */
      return block;
   }

   protected MdlAttribute parseAttribute() throws ParserException {
      Token name = lexer.consume();
      if (!name.isValidName()) {
         throw new ParserException("valid attribute name expected here", name);
      }
      Token value = lexer.consume();
      if (!value.isValidValue()) {
         throw
            new ParserException("valid attribute value expected here", value);
      }
      String value_str = value.toString();
      if (value.isQuotedText()) {
         while(lexer.current().isQuotedText()) {
            /*value_str += "\n" + lexer.consume().toString();*/
            value_str =
               StringUtils.joinQuoted(value_str, lexer.consume().toString());
         }
      }
      return new MdlAttribute(name.toString(), value_str);
   }
}
