package org.zeyda.clawcircus.Parser;

import org.zeyda.clawcircus.utils.StringUtils;

import java.io.StringReader;
import java.io.StreamTokenizer;
import java.io.IOException;

import java.util.List;
import java.util.LinkedList;

/* We implement MdlLexer relying on the StreamTokenizer class at the moment.
 * This is no optimal since StreamTokenizer interprets escape sequences in
 * quoted text, currently we compensate for this by reversing the process
 * through a call to StringUtils.quote(String). Maybe re-implement this class
 * at some time no using StreamTokenizer. */

public class MdlLexer {
   private StreamTokenizer tokeniser;
   private Token[] tokens;

   private int index;

   public MdlLexer(String text) {
      tokeniser = new StreamTokenizer(new StringReader(text));
      tokens = new Token[0];
      configureSyntax();
      try {
         generateTokens();
      }
      catch (IOException e) {
         System.out.println(e);
         System.exit(-1);
      }
      reset();
   }

   private void configureSyntax() {
      tokeniser.resetSyntax();
      tokeniser.wordChars('\u0000', '\u00ff');
      tokeniser.quoteChar('\"');
      tokeniser.quoteChar('[');
      tokeniser.ordinaryChar('{');
      tokeniser.ordinaryChar('}');
      tokeniser.whitespaceChars(' ', ' ');
      tokeniser.whitespaceChars('\t', '\t');
      tokeniser.whitespaceChars('\n', '\n');
      tokeniser.whitespaceChars('\r', '\r');
      tokeniser.eolIsSignificant(false);
      tokeniser.slashSlashComments(false);
      tokeniser.slashStarComments(false);
   }

   private void generateTokens() throws IOException {
      List<Token> token_list = new LinkedList<Token>();
      Token token;
      do {
         tokeniser.nextToken();
         int line = tokeniser.lineno();
         switch(tokeniser.ttype) {
            case StreamTokenizer.TT_WORD:
               assert tokeniser.sval != null;
               token = new Token(TokenType.TEXT, tokeniser.sval, line);
               break;

            case '"':
               assert tokeniser.sval != null;
               tokeniser.sval = StringUtils.quote(tokeniser.sval); /* (!) */
               token = new Token(TokenType.QUOTED_TEXT, tokeniser.sval, line);
               break;

            /* Note that here the opening [ is matched against the end of the
             * line rather than the closing ]. This is a limitation imposed by
             * the StreamTokenizer class which requires opening and closing
             * quotes to be similar characters. */
            case '[':
               assert tokeniser.sval != null;
               token =
                  new Token(TokenType.TEXT, "[" + tokeniser.sval, line);
               break;

            case '{':
               token = new Token(TokenType.LCURLY, line);
               break;

            case '}':
               token = new Token(TokenType.RCURLY, line);
               break;

            case StreamTokenizer.TT_EOF:
               token = new Token(TokenType.EOF, line);
               break;

            default:
               throw new AssertionError();
         }
         token_list.add(token);
      } while (!token.isEOF());
      tokens = token_list.toArray(tokens);
   }


   public void dumpTokens() { /* For debugging purposes. */
      for(Token token : tokens) {
         System.out.println("[" + token + "]");
      }
   }

   public void reset() {
      index = 0;
   }

   /* Note that the lookahead parameter may moreover be negative. */

   public Token LL(int lookahead) {
      assert index + lookahead >= 0;
      if (index + lookahead < tokens.length) {
         return tokens[index + lookahead];
      }
      else {
         if (tokens.length != 0 &&
            tokens[tokens.length-1].getType() == TokenType.EOF) {
            return tokens[tokens.length-1];
         }
         else {
            return Token.EOF_TOKEN; /* Return a generic EOF token here. */
         }
      }
   }

   public Token consume() {
      Token result = current();
      if (index < tokens.length) {
         index++;
      }
      return result;
   }

   public boolean eof() {
      return current().isEOF();
   }

   public Token next() {
      return LL(1);
   }

   public Token last() {
      return LL(-1);
   }

   public Token current() {
      return LL(0);
   }
}
