package org.zeyda.clawcircus.Data.Simulink;

import org.zeyda.clawcircus.misc.AnnotationInterface;

import org.zeyda.clawcircus.utils.StringUtils;

import java.util.Iterator;

/* Note that in order to render the typical use case where a method call to
 * hasAttribute() is immediately followed by a call to getAttribute() more
 * efficiently, a simple one-element cache has been implemented. */

public class MdlBlock extends MdlElement implements
   AnnotationInterface<MdlBlock>, Iterable<MdlElement> {

   protected final MdlElementList elements;
   private MdlElement cache;

   /* Computed differentially for efficiency! */
   private int max_length_name = 0;

   public MdlBlock(String name) {
      super(name);
      elements = new MdlElementListImpl();
      cache = null;
   }

   public int getMaxLengthName() {
      return max_length_name;
   }

   /* Currently the get/set methods for MdlAttribute and MdlBlock assume there
    * is only one attribute or block in scope with the given name. In reality
    * however there could be more, to access those there is no real support at
    * the moment yet other than iterating through the block elements manually.
    */

   public  MdlAttribute getAttribute(String name) {
      name = name.trim();
      if (cache != null) {
         if (cache.isAttribute() && cache.getName().equals(name)) {
            return (MdlAttribute) cache;
         }
      }
      for(MdlElement element : this) {
         if (element.isAttribute() && element.getName().equals(name)) {
            cache = element;
            return (MdlAttribute) element;
         }
      }
      return null;
   }

   public String getAttributeValue(String name) {
      MdlAttribute attr = getAttribute(name);
      if (attr != null) {
         return attr.getValue();
      }
      return null;
   }

   public void setAttribute(String name, String value) {
      MdlAttribute attribute = getAttribute(name);
      if (attribute != null) {
         attribute.setValue(value);
      }
      else {
         add(new MdlAttribute(name, value));
      }
   }

   public boolean hasAttribute(String name) {
      return getAttribute(name) != null;
   }

   public boolean removeAttribute(String name) {
      MdlAttribute attr = getAttribute(name);
      if (attr != null) {
         return remove(attr);
      }
      return false;
   }

   public MdlBlock getBlock(String name) {
      name = name.trim();
      if (cache != null) {
         if (cache.isBlock() && cache.getName().equals(name)) {
            return (MdlBlock) cache;
         }
      }
      for(MdlElement element : this) {
         if (element.isBlock() && element.getName().equals(name)) {
            cache = element;
            return (MdlBlock) element;
         }
      }
      return null;
   }

   public boolean containsBlock(String name) {
      return getBlock(name) != null;
   }

   public boolean removeBlock(String name) {
      MdlBlock block = getBlock(name);
      if (block != null) {
         return remove(block);
      }
      return false;
   }

   public void add(MdlElement element) {
      assert element != null;
      assert !element.hasParent();
      /*if (element.isAttribute()) {
         assert !hasAttribute(element.getName());
      }*/
      /*if (element.isBlock()) {
         assert !containsBlock(element.getName());
      }*/
      elements.add(element);
      element.setParent(this);
      /* A bit more efficient than calling recalcMaxLengthName(). */
      if (element.isAttribute()) {
         if (element.getName().length() > max_length_name) {
            max_length_name = element.getName().length();
         }
      }
      /*cache = element;*/
   }

   public boolean remove(MdlElement element) {
      boolean success = elements.remove(element);
      if (success) {
         element.setParent(null);
         recalcMaxLengthName();
      }
      return success;
   }

   /* High-level utility methods, if Java would support partial classes I'd
    * rather have them in a separate file.. */

   /* The following method might still be subject to testing.. */

   public String getBlockPath() {
      assert getName().equals("Block") || getName().equals("System");
      if (getName().equals("System")) {
         if (hasParent() && !getParent().getName().equals("Model")) {
            return getParent().getBlockPath();
         }
         return getAttributeValue("Name");
      }
      String result = "";
      if (hasParent()) {
         result = StringUtils.removeQuotes(getParent().getBlockPath()) + "/";
      }
      result += StringUtils.removeQuotes(getAttributeValue("Name"));
      result = StringUtils.addQuotes(result);
      return result;
   }

   public boolean isBlockType(String block_type_str) {
      assert getName().equals("Block");
      if (hasAttribute("BlockType")) {
         return getAttributeValue("BlockType").equals(block_type_str);
      }
      return false;
   }

   /* End of utility methods. */

   protected void recalcMaxLengthName() {
      max_length_name = 0;
      for(MdlElement element : this) {
         if (element.isAttribute()) {
            if (element.getName().length() > max_length_name) {
               max_length_name = element.getName().length();
            }
         }
      }
   }

   public String serialise() {
      StringBuilder result = new StringBuilder();
      result.append(getName() + " {\n");
      String indent = Formatting.getTabSpace(getLevel() + 1);
      for(MdlElement element : this) {
         result.append(indent + element.serialise() + "\n");
      }
      result.append(Formatting.getTabSpace(getLevel()) + "}");
      return result.toString();
   }

   public Class<MdlBlock> getAnnotationKey() {
      return MdlBlock.class;
   }

   public Iterator<MdlElement> iterator() {
      return elements.listIterator();
   }

   public Object clone() {
      MdlBlock clone = (MdlBlock) super.clone();
      clone.elements.clear();
      for(MdlElement element : elements) {
         elements.add((MdlElement) element.clone());
      }
      return clone;
   }
}
