package org.zeyda.clawcircus.Data.Diagram;

import org.zeyda.clawcircus.Data.Simulink.MdlBlock;

import org.zeyda.clawcircus.Data.ClaSP.BlockWiring;
import org.zeyda.clawcircus.Data.ClaSP.BlockWiringType;
import org.zeyda.clawcircus.Data.ClaSP.Enabled;
import org.zeyda.clawcircus.Data.ClaSP.Flow;
import org.zeyda.clawcircus.Data.ClaSP.Signal;

import org.zeyda.clawcircus.Toolbox.TranslationStrategy;

import org.zeyda.clawcircus.exceptions.MdlSemanticException;
import org.zeyda.clawcircus.exceptions.BlockInstantiationException;

import org.zeyda.clawcircus.collections.*;
import org.zeyda.clawcircus.collections.impl.*;

import org.zeyda.clawcircus.utils.ClaSPUtils;

public class SubSystem extends Block implements Iterable<Block> {
   protected final BlockList blocks;

   /* Used to cache internal block wiring once computed. */
   private BlockWiring block_wiring_inner_cache = null;

   public SubSystem() {
      super(0, 0);
      blocks = new BlockListImpl();
      setProperty("translatable", true);
      setProperty("state", false);
   }

   public SubSystem(String name) {
      this();
      setName(name);
   }

   public @Override void initialise() throws MdlSemanticException {
      super.initialise();
      if (hasAnnotation(MdlBlock.class)) {
         MdlBlock mdl_block = getAnnotation(MdlBlock.class);
         /* TODO: Carry out diagram construction here. */
      }
   }

   public BlockList getBlocks() {
      return blocks;
   }

   public void add(Block block) {
      blocks.add(block);
      block.setParent(this);
      if (block.isPort()) {
         updateLinks(block);
      }
   }

   public void addAll(BlockList blocks) {
      for(Block block : blocks) {
         add(block);
      }
   }

   public Block find(String name) {
      for(Block block : blocks) {
         if (block.getName().equals(name)) {
            return block;
         }
      }
      return null;
   }

   protected void updateLinks(Block block) {
      if (block.isInport()) {
         int port_num = ((PortBlock) block).getPortNum();
         if (port_num > getInputPortsNum()) {
            updateIncomingLinks(port_num);
         }
      }
      if (block.isOutport()) {
         int port_num = ((PortBlock) block).getPortNum();
         if (port_num > getOutputPortsNum()) {
            updateOutgoingLinks(port_num);
         }
      }
   }

   public void updateLinks() {
      int max_inputs = 0;
      int max_outputs = 0;
      for(Block block : blocks) {
         if (block.isInport()) {
            int port = ((PortBlock) block).getPortNum();
            max_inputs = Math.max(port, max_inputs);
         }
         if (block.isOutport()) {
            int port = ((PortBlock) block).getPortNum();
            max_outputs = Math.max(port, max_outputs);
         }
      }
      updateLinks(max_inputs, max_outputs);
   }

   /* Resolution of (port) blocks into ports, works internal-to-external. */

   public Port resolveIntoPort(Block block) {
      /*assert block.isPort();*/
      assert blocks.contains(block);
      Port port = null;
      if (block.isInport()) {
         port =
            new Port(this, ((PortBlock) block).getPortNum(), PortDir.INPUT);
      }
      if (block.isOutport()) {
         port =
            new Port(this, ((PortBlock) block).getPortNum(), PortDir.OUTPUT);
      }
      if (block.isEnablePort()) {
         port = new Port(this, PortType.ENABLE, PortDir.INPUT);
      }
      if (block.isTriggerPort()) {
         port = new Port(this, PortType.TRIGGER, PortDir.INPUT);
      }
      if (block.isActionPort()) {
         port = new Port(this, PortType.ACTION, PortDir.INPUT);
      }
      if (port != null) {
         assert port.isValid();
      }
      return port;
   }

   /* Resolution of ports into (port) blocks, works external-to-internal. */

   public PortBlock resolveIntoBlock(Port port) {
      assert port.getBlock() == this;
      for (Block block : blocks) {
         if (block.isPort()) {
            PortBlock port_block = (PortBlock) block;
            if (port.isStandard()) {
               if (block.isInport() && port.isInput()) {
                  if (port_block.getPortNum() == port.getPortNum()) {
                     return port_block;
                  }
               }
               if (block.isOutport() && port.isOutput()) {
                  if (port_block.getPortNum() == port.getPortNum()) {
                     return port_block;
                  }
               }
            }
            else {
               assert port.isSpecial();
               if (port.isInput()) {
                  if (block.isEnablePort() && port.isEnable()) {
                     return port_block;
                  }
                  if (block.isTriggerPort() && port.isTrigger()) {
                     return port_block;
                  }
                  if (block.isActionPort() && port.isAction()) {
                     return port_block;
                  }
               }
            }
         }
      }
      return null; /* Resolution failed. */
   }

   /* Methods to obtain all internal and external links of the subsystem. */

   public LinkSet getExternalLinks() {
      return super.getAllLinks();
   }

   public LinkSet getInternalLinks() {
      LinkSet result = new LinkSetImpl();
      for(Block block : getChildren()) {
         if (block instanceof SubSystem) {
            result.addAll(((SubSystem) block).getExternalLinks());
         }
      }
      return result;
   }

   /* Computation of ClaSP block wiring information. */

   public BlockWiring getBlockWiring(BlockWiringType block_wiring_type) {
      if (block_wiring_type == BlockWiringType.OUTER) {
         return getBlockWiring();
      }
      else {
         assert block_wiring_type == BlockWiringType.INNER;
         if (block_wiring_inner_cache == null) {
            block_wiring_inner_cache = calcBlockWiring(BlockWiringType.INNER);
         }
         assert block_wiring_inner_cache != null;
         return block_wiring_inner_cache;
      }
   }

   public @Override BlockWiring calcBlockWiring() {
      return calcBlockWiring(BlockWiringType.OUTER);
   }

   /* For subsystems the block wiring information is constructed dynamically.
    * The current implementation doesn't yet take into account enabling signals
    * and their propagation. */

   protected BlockWiring calcBlockWiring(BlockWiringType block_wiring_type) {
      BlockWiring result;
      /*if (isRoot()) {
         assert block_wiring_type == BlockWiringType.INNER;
      }*/
      /*updateLinks();*/
      ResolveMethod method;
      if (block_wiring_type == BlockWiringType.INNER) {
         method = ResolveMethod.INTERNAL;
      }
      else {
         assert block_wiring_type == BlockWiringType.OUTER;
         method = ResolveMethod.EXTERNAL;
      }
      SignalList inps = new SignalListImpl();
      for(Port port : getInputPorts()) {
         inps.add(ClaSPUtils.getSignalFromPort(port, method));
      }
      SignalList outs = new SignalListImpl();
      for(Port port : getOutputPorts()) {
         outs.add(ClaSPUtils.getSignalFromPort(port, method));
      }
      FlowSet flows = new FlowSetImpl();
      for(Signal pout : outs) {
         if (pout != null) {
            SignalSet pouts = new SignalSetImpl();
            pouts.add(pout);
            if (block_wiring_type != BlockWiringType.INNER) {
               pout = pout.unlift();
            }
            if (pout != null) {
               SignalSet rinps = pout.traceBackToInputs();
               if (block_wiring_type != BlockWiringType.INNER) {
                  rinps = rinps.lift();
               }
               flows.add(
                  new Flow(Enabled.ALWAYS, false, rinps, pouts));
            }
         }
      }
      result = new BlockWiring(inps, outs, flows);
      result.normalise();
      return result;
   }

   /* Experimental method giving a sequential semantics to the subsystem. */

   public BlockWiring getFlatBlockWiring(BlockWiringType block_wiring_type) {
      /*if (isRoot()) {
         assert block_wiring_type == BlockWiringType.INNER;
      }*/
      /*updateLinks();*/
      ResolveMethod method;
      if (block_wiring_type == BlockWiringType.INNER) {
         method = ResolveMethod.INTERNAL;
      }
      else {
         assert block_wiring_type == BlockWiringType.OUTER;
         method = ResolveMethod.EXTERNAL;
      }
      SignalList inps = new SignalListImpl();
      for(Port port : getInputPorts()) {
         inps.add(ClaSPUtils.getSignalFromPort(port, method));
      }
      SignalList outs = new SignalListImpl();
      for(Port port : getOutputPorts()) {
         outs.add(ClaSPUtils.getSignalFromPort(port, method));
      }
      FlowSet flows = new FlowSetImpl();
      SignalSet rinps = new SignalSetImpl(inps);
      SignalSet pouts = new SignalSetImpl(outs);
      flows.add(new Flow(Enabled.ALWAYS, false, rinps, pouts));
      return new BlockWiring(inps, outs, flows);
   }

   public @Override void clearAllBlockWiring() {
      super.clearAllBlockWiring();
      block_wiring_inner_cache = null;
   }

   public void printDetails() {
      System.out.println("SubSystem: " + getPath());
      System.out.println("---");
      System.out.println("[Blocks]");
      for (Block block : blocks) {
         System.out.println("  " + block);
      }
      System.out.println("---");
      System.out.println("[Links]");
      for (Link link : getInternalLinks()) {
         System.out.println("  " + link);
      }
   }

   public void printAllDetails() {
      printDetails();
      for(Block block : blocks) {
         if (block.isSubSystem()) {
            System.out.println();
            ((SubSystem) block).printAllDetails();
         }
      }
   }

   public BlockList getChildren() {
      return getBlocks();
   }

   public static SubSystem create() {
      try {
         return (SubSystem) BlockFactory.create("SubSystem");
      }
      catch(BlockInstantiationException e) {
         throw new AssertionError(e);
      }
   }

   /* Methods to record the translations strategy for the subsystem. */

   public TranslationStrategy getTranslationStrategy() {
      return getAnnotation(TranslationStrategy.class);
   }

   public void setTranslationStrategy(TranslationStrategy trans_strat) {
      setAnnotation(trans_strat);
      notifyUI();
   }

   public final boolean isCentralised() {
      return getTranslationStrategy() == TranslationStrategy.CENTRALISED;
   }

   public final boolean isParallel() {
      return getTranslationStrategy() == TranslationStrategy.PARALLEL;
   }
}
