package org.zeyda.clawcircus.Data.Diagram;

import org.zeyda.clawcircus.utils.StringUtils;

/* To increase efficiency it might be worth to use the fly-weight pattern at
 * some point to reduce the number of port objects create when processing the
 * diagram, i.e. by caching them in some kind of sorted array. */

public final class Port { /* Immutable class. */
   private final Block block;
   private final int port_num;
   private final PortType type;
   private final PortDir dir;

   protected Port(Block block, int port_num, PortType type, PortDir dir) {
      assert block != null;
      assert type != null;
      assert (type.isStandard() && port_num >= 1)
          || (type.isSpecial()  && port_num == 0);
      assert dir != null;
      this.block = block;
      this.port_num = port_num;
      this.type = type;
      this.dir = dir;
      /*assert isValid();*/ /* Too strong upon creation. */
   }

   public Port(Block block, int port_num, PortDir dir) {
      this(block, port_num, PortType.STANDARD, dir);
   }

   public Port(Block block, PortType type, PortDir dir) {
      /*assert type != PortType.STANDARD;*/
      this(block, 0, type, dir);
   }

   public Block getBlock() {
      return block;
   }

   public int getPortNum() {
      assert isStandard();
      return port_num;
   }

   public PortType getType() {
      return type;
   }

   public PortDir getDir() {
      return dir;
   }

   public boolean isValid() {
      if (type.isStandard()) {
         switch (dir) {
            case INPUT:
               return port_num >= 1 && port_num <= block.getInputPortsNum();

            case OUTPUT:
               return port_num >= 1 && port_num <= block.getOutputPortsNum();

            default:
               throw new AssertionError();
         }
      }
      else {
         assert port_num == 0; /* Guaranteed by the constructor. */
         return true; /* Special ports are currently always valid. */
      }
   }

   /* May return null if no link connects the port. */

   public Link getLink() {
      assert isValid();
      if (type.isStandard()) {
         switch (dir) {
            case INPUT:
               return block.getIncomingLink(port_num);

            case OUTPUT:
               return block.getOutgoingLink(port_num);

            default:
               throw new AssertionError();
         }
      }
      else {
         switch (dir) {
            case INPUT:
               return block.getSpecialIncomingLink(type);

            case OUTPUT:
               return block.getSpecialOutgoingLink(type);

            default:
               throw new AssertionError();
         }
      }
   }

   public boolean isStandard() {
      return type.isStandard();
   }

   public boolean isSpecial() {
      return type.isSpecial();
   }

   public boolean isInput() {
      return dir == PortDir.INPUT;
   }

   public boolean isOutput() {
      return dir == PortDir.OUTPUT;
   }

   public boolean isEnable() {
      return type == PortType.ENABLE;
   }

   public boolean isTrigger() {
      return type == PortType.TRIGGER;
   }

   public boolean isAction() {
      return type == PortType.ACTION;
   }

   public boolean isConnected() {
      assert isValid();
      return getLink() != null;
   }

   public @Override boolean equals(Object obj) {
      if (obj instanceof Port) {
         Port port = (Port) obj;
         return block.equals(port.block)
            && port_num == port.port_num
            && type == port.type
            && dir == port.dir;
      }
      else {
         return super.equals(obj);
      }
   }

   public @Override int hashCode() {
      return block.hashCode() ^ (new Integer(port_num)).hashCode()
         ^ type.hashCode() ^ dir.hashCode();
   }

   public String toString() {
      String suffix;
      switch(type) {
         case STANDARD:
            suffix = "";
            break;

         case ENABLE:
            suffix = "Enable";
            break;

         case TRIGGER:
            suffix = "Trigger";
            break;

         case ACTION:
            suffix = "Action";
            break;

         default:
            throw new AssertionError();
      }
      switch(dir) {
         case INPUT:
            suffix += "In";
            break;

         case OUTPUT:
            suffix += "Out";
            break;

         default:
            throw new AssertionError();
      }
      /* Maybe revise the String representation generated here. */
      if (type.isStandard()) {
         return StringUtils.quote(block.getName()) + "." + suffix + port_num;
      }
      else {
         return StringUtils.quote(block.getName()) + "." + suffix;
      }
   }
}
