package org.zeyda.clawcircus.Data.ClaSP;

import org.zeyda.clawcircus.Application.GlobalErrorHandler;

import org.zeyda.clawcircus.misc.AnnotationInterface;

import org.zeyda.clawcircus.collections.*;
import org.zeyda.clawcircus.collections.impl.*;

import java.util.Iterator;

public class BlockWiring implements AnnotationInterface<BlockWiring> {
   protected final SignalList inps;
   protected final SignalList outs;
   protected /*final*/ FlowSet flows; /* Normalise modifies this member. */

   public BlockWiring(SignalList inps, SignalList outs, FlowSet flows) {
      assert inps != null;
      assert outs != null;
      assert flows != null;
      this.inps = inps;
      this.outs = outs;
      this.flows = flows;
   }

   public SignalList getInps() {
      return inps;
   }

   public SignalList getOuts() {
      return outs;
   }

   public FlowSet getFlows() {
      return flows;
   }

   public SignalSet getInterface() {
      SignalSet result = new SignalSetImpl();
      for(Signal inp : inps) {
         result.addIfNotNull(inp);
      }
      for(Signal out : outs) {
         result.addIfNotNull(out);
      }
      return result;
   }

   public Flow getFlow(Signal out) {
      assert outs.contains(out);
      for(Flow flow : flows) {
         if (flow.getPOuts().contains(out)) {
            return flow;
         }
      }
      /*assert false;*/
      GlobalErrorHandler.warning("Couldn't obtain rinps for output signal "
         + out + "; flow sets seems to be incomplete.");
      return null;
   }

   /* According to the discussion I had with Ana for the ICFEM 2008 paper,
    * the following should be maximise(); */

   public void normalise() {
      minimise();
      /*maximise();*/
   }

   /* Minimise merges flows with the same input and enabled signal sets. */

   public void minimise() {
      FlowSet min_flows = new FlowSetImpl();
      while(!flows.isEmpty()) {
         Flow next = flows.select();
         flows.remove(next);
         /* Find and remove all other flows with similar input sets. */
         for(Iterator<Flow> iter = flows.iterator(); iter.hasNext();) {
            Flow flow = iter.next();
            if (flow.getRInps().equals(next.getRInps()) &&
               flow.getEnabled().equals(next.getEnabled()) &&
               flow.isOrdered() == next.isOrdered()) {
               next.getPOuts().addAll(flow.getPOuts());
               iter.remove();
            }
         }
         min_flows.add(next);
      }
      flows = min_flows;
   }

   /* Maximise ensures existence of a separate flow each output. */

   public void maximise() {
      FlowSet max_flows = new FlowSetImpl();
      for(Flow flow : flows) {
         for(Signal out : flow.getPOuts()) {
            SignalSet pouts = new SignalSetImpl();
            pouts.add(out);
            max_flows.add(new Flow(
               flow.getEnabled(), flow.isOrdered(), flow.getRInps(), pouts));
         }
      }
      flows = max_flows;
   }

   public BlockWiring lift() {
      return new BlockWiring(inps.lift(), outs.lift(), flows.lift());
   }

   public Class<BlockWiring> getAnnotationKey() {
      return BlockWiring.class;
   }

   public @Override String toString() {
      StringBuilder result = new StringBuilder();
      result.append("[BlockWiring]\n");
      result.append(" inps = " + inps.toString() + "\n");
      result.append(" outs = " + outs.toString() + "\n");
      result.append(" flows = " + flows.toString() + "\n");
      return result.toString();
   }
}
