package org.zeyda.clawcircus.Toolbox;

import org.zeyda.clawcircus.Data.Diagram.*;

import org.zeyda.clawcircus.Data.Simulink.MdlElement;
import org.zeyda.clawcircus.Data.Simulink.MdlBlock;

import org.zeyda.clawcircus.Application.GlobalErrorHandler;

import org.zeyda.clawcircus.collections.*;
import org.zeyda.clawcircus.collections.impl.*;

import org.zeyda.clawcircus.exceptions.MdlSemanticException;
import org.zeyda.clawcircus.exceptions.BlockInstantiationException;

import org.zeyda.clawcircus.utils.MdlUtils;
import org.zeyda.clawcircus.utils.StringUtils;
import org.zeyda.clawcircus.utils.DiagramUtils;

public class DiagramConstructor {
   public static SubSystem invoke(MdlBlock model) throws MdlSemanticException {
      if (!(model.isBlock() && model.getName().equals("Model"))) {
         throw new IllegalArgumentException("Model block expected.");
      }
      MdlBlock system = model.getBlock("System");
      if (system == null) {
         throw new MdlSemanticException("System block not present.", model);
      }
      return createSubSystem(system);
   }

   public static SubSystem createSubSystem(MdlBlock system) throws
      MdlSemanticException {
      if (!(system.isBlock() && system.getName().equals("System"))) {
         throw new IllegalArgumentException("System block expected.");
      }
      String name = MdlUtils.getStringAttribute("Name", system);
      SubSystem subsystem = new SubSystem(name);
      /* Create all Blocks. */
      for (MdlElement element : system) {
         if (element.isBlock() && element.getName().equals("Block")) {
            MdlBlock block = (MdlBlock) element;
            if (block.isBlockType("SubSystem")) {
               MdlBlock system_block = block.getBlock("System");
               if (system_block == null) {
                  throw new MdlSemanticException("System block not present.",
                     block);
               }
               subsystem.add(createSubSystem(system_block));
            }
            else {
               try {
                  subsystem.add(BlockFactory.create(block));
               }
               catch(BlockInstantiationException e) {
                  String type =
                     MdlUtils.getStringAttribute("BlockType", block);
                  GlobalErrorHandler.warning(
                     "Block of type " + type + " couldn't be instantiated.");
               }
            }
         }
      }
   outer_loop: /* Create all Links. */
      for (MdlElement element : system) {
         if (element.isBlock() && element.getName().equals("Line")) {
            MdlBlock line = (MdlBlock) element;
            Port src_port = fetchSrcPort(line, subsystem);
            if (src_port == null) {
               continue outer_loop;
            }
            PortList dst_ports = fetchDstPorts(line, subsystem);
            for(Port dst_port : dst_ports) {
               DiagramUtils.connect(src_port, dst_port);
            }
         }
      }
      return subsystem;
   }

   private static Port fetchSrcPort(MdlBlock line, SubSystem subsystem) throws
      MdlSemanticException {
      assert line.getName().equals("Line");
      /* We assume the source pots cannot be any of the special ports. */
      String src_name = MdlUtils.getStringAttribute("SrcBlock", line);
      int src_port_num = MdlUtils.getIntegerAttribute("SrcPort", line);
      Block src_block = subsystem.find(src_name);
      Port src_port = null;
      if (src_block == null) {
         /*throw new MdlSemanticException(
            "Couldn't resolve source in Line or Branch from" +
            StringUtils.quote(src_name) + ".");*/
         GlobalErrorHandler.warning(
            "Couldn't resolve source in Line or Branch from " +
            StringUtils.quote(src_name) + ".");
         return null;
      }
      else {
         src_port = src_block.getOutputPort(src_port_num);
      }
      return src_port;
   }

   private static PortList fetchDstPorts(MdlBlock line_or_branch,
      SubSystem subsystem) throws MdlSemanticException {
      assert
         line_or_branch.getName().equals("Line") ||
         line_or_branch.getName().equals("Branch");
      PortList result = new PortListImpl();
      if (line_or_branch.hasAttribute("DstBlock")) {
         String dst_name =
            MdlUtils.getStringAttribute("DstBlock", line_or_branch);
         String dst_port_str =
            MdlUtils.getStringAttribute("DstPort", line_or_branch);
         Block dst_block = subsystem.find(dst_name);
         if (dst_block == null) {
            /*throw new MdlSemanticException(
               "Couldn't resolve target in Line or Branch to "
               + StringUtils.quote(dst_name) + ".");*/
            GlobalErrorHandler.warning(
               "Couldn't resolve target in Line or Branch to "
               + StringUtils.quote(dst_name) + ".");
         }
         else {
            Port dst_port = null;
            int dst_port_num = 0;
            try {
               dst_port_num = Integer.parseInt(dst_port_str);
            }
            catch (NumberFormatException e) {
            }
            if (dst_port_num == 0) {
               if (dst_port_str.equals("enable")) {
                  dst_port = dst_block.getInputPort(PortType.ENABLE);
               }
               if (dst_port_str.equals("trigger")) {
                  dst_port = dst_block.getInputPort(PortType.TRIGGER);
               }
               if (dst_port_str.equals("ifaction")) {
                  dst_port = dst_block.getInputPort(PortType.ACTION);
               }
               /* Possibly more here.. */
            }
            else {
               dst_port = dst_block.getInputPort(dst_port_num);
            }
            if (dst_port != null) {
               result.add(dst_port);
            }
            else {
               /*throw new MdlSemanticsException("Unsupported target "
                  "Unsupported destination " + StringUtils.quote(dst_port_str)
                  + " in Line or Branch");*/
               System.err.println(
                  "Unsupported destination " + StringUtils.quote(dst_port_str)
                  + " in Line or Branch.");
            }
         }
      }
      for (MdlElement element : line_or_branch) {
         if (element.isBlock() && element.getName().equals("Branch")) {
            result.addAll(fetchDstPorts((MdlBlock) element, subsystem));
         }
      }
      return result;
   }
}
