package org.zeyda.clawcircus.resources;

import org.zeyda.clawcircus.Application.GlobalErrorHandler;

import org.antlr.stringtemplate.StringTemplate;
import org.antlr.stringtemplate.StringTemplateGroup;
import org.antlr.stringtemplate.StringTemplateErrorListener;

import java.io.File;

/* The primary purpose of this class is that it provides a two stage
 * resolution process for string templates within groups first using the local
 * file store, and if this fails looking them up in the respective resource
 * folder in the jar.
 * This functionality useful for development purposes as modifications to
 * string templates take immediate effect without the need to recompile the
 * source code and assemble the jar. */

class ResourceStringTemplateGroup extends StringTemplateGroup {
   /* The following two should at some point be best isolated into some system
    * properties. Maybe they can be provided by Maven? */

   public final String ST_JAR_ROOT = "resources/stg";
   public final String ST_LOCAL_ROOT = "src/resources/stg";

   private StringTemplateGroup primary_source;

   /* ErrorListener for redirecting errors to GlobalErrorHandler. */

   class ErrorListener implements StringTemplateErrorListener {
      public void error(String msg, Throwable e) {
         GlobalErrorHandler.raiseError(e, msg);
      }

      public void warning(String msg) {
         GlobalErrorHandler.warning(msg);
      }
   }

   /* Note that the name argument refers to the subdirectory of ST_ROOT
    * where templates are located. */

   public ResourceStringTemplateGroup(String name) {
      super(name);
      setErrorListener(
         new ResourceStringTemplateGroup.ErrorListener());
      /* Initialise primary source. */
      primary_source =
         new StringTemplateGroup(name, ST_LOCAL_ROOT + File.separator + name);
      /* Never cache templates obtained from file store. */
      primary_source.setRefreshInterval(0);
      primary_source.setErrorListener(
         new ResourceStringTemplateGroup.ErrorListener());
   }

   public @Override String getFileNameFromTemplateName(String templname) {
      return ST_JAR_ROOT + File.separator + getName() + File.separator +
         templname + ".st";
   }

   public @Override String getTemplateNameFromFileName(String filename) {
      String templname = filename;
      if (templname.startsWith(
         ST_JAR_ROOT + File.separator + getName() + File.separator)) {
         templname = templname.replaceFirst(
            ST_JAR_ROOT + File.separator + getName() + File.separator, "");
      }
      return super.getTemplateNameFromFileName(templname);
   }

   public synchronized @Override StringTemplate lookupTemplate(
      StringTemplate enclosingInstance, String name) {
      try {
         return primary_source.lookupTemplate(enclosingInstance, name);
      }
      catch(IllegalArgumentException e) {
         return super.lookupTemplate(enclosingInstance, name);
      }
   }
}
