package org.zeyda.clawcircus.Toolbox;

import org.zeyda.clawcircus.Data.Simulink.*;
import org.zeyda.clawcircus.Data.ClawZ.LibraryMetaFile;

import org.zeyda.clawcircus.utils.PrintUtils;

import java.util.List;
import java.util.LinkedList;

public class SimulinkAnalyser {
   public static final int SPACES_PER_TAB = 2;

   private static boolean USE_COLOURED_OUTPUT = false;

   public final static String[] DEFAULT_TRANSLATABLE_BLOCKS = {
      "Inport", "Outport", "EnablePort", "Constant", "Terminator", "Mux",
      "Demux" /* More to be given here.. */
   };

   protected LibraryMetaFile[] lmfs;

   public SimulinkAnalyser(LibraryMetaFile[] lmfs) {
      this.lmfs = lmfs;
   }

   public static void invoke(MdlBlock model, LibraryMetaFile[] lmfs) {
      (new SimulinkAnalyser(lmfs)).invoke(model);
   }

   private void invoke(MdlBlock model) {
      MdlBlock system = model.getBlock("System");
      if (system == null) {
         System.out.println(
            "System block not present, no analysis performed.");
         return ;
      }
      analyseSystem(system);
   }

   private void analyseSystem(MdlBlock system) {
      analyseSystem(system, 0);
   }

   private void analyseSystem(MdlBlock system, int level) {
      assert system.getName().equals("System");
      if (level == 0) {
         printBlockInfo("System", system.getAttributeValue("Name"), level);
      }
      else {
         System.out.println(getIndentSpace(level) + "---");
         printBlockInfo("SubSystem", system.getAttributeValue("Name"), level);
      }
      List<MdlElement> done = new LinkedList<MdlElement>();
      for (MdlElement element : system) {
         if (element.isBlock() && element.getName().equals("Block")) {
            MdlBlock block = (MdlBlock) element;
            if (block.isBlockType("Inport")) {
               analyseBlock(block, level+1);
               done.add(block);
            }
         }
      }
      System.out.println(getIndentSpace(level+1) + "---");
      for (MdlElement element : system) {
         if (element.isBlock() && element.getName().equals("Block")) {
            MdlBlock block = (MdlBlock) element;
            if (block.isBlockType("Outport")) {
               analyseBlock(block, level+1);
               done.add(block);
            }
         }
      }
      System.out.println(getIndentSpace(level+1) + "---");
      for (MdlElement element : system) {
         if (element.isBlock() && element.getName().equals("Block")) {
            MdlBlock block = (MdlBlock) element;
            if (!done.contains(block) && !block.isBlockType("SubSystem")) {
               analyseBlock(block, level+1);
               done.add(block);
            }
         }
      }
      for (MdlElement element : system) {
         if (element.isBlock() && element.getName().equals("Block")) {
            MdlBlock block = (MdlBlock) element;
            if (!done.contains(block)) {
               analyseBlock(block, level+1);
            }
         }
      }
   }

   private void analyseBlock(MdlBlock block, int level) {
      assert block.getName().equals("Block");
      if (block.isBlockType("SubSystem")) {
         if (block.containsBlock("System")) {
            analyseSystem(block.getBlock("System"), level);
         }
         else {
            printBlockInfo(
               "SubSystem", block.getAttributeValue("Name"), level);
            System.out.println(getIndentSpace(level+1) + "?");
         }
      }
      else {
         if (block.hasAttribute("BlockType")) {
            String name = block.getAttributeValue("Name");
            BlockInfoTag tag = BlockInfoTag.VOID;
            if (!checkTranslatable(block)) {
               tag = BlockInfoTag.NONTRANSLATABLE;
            }
            printBlockInfo(
               block.getAttributeValue("BlockType"), name, level, tag);
         }
         else {
            /* Maybe display a warning message here. */
         }
      }
   }

   private void printBlockInfo(String block_type, String name, int level) {
      printBlockInfo(block_type, name, level, BlockInfoTag.VOID);
   }

   private void printBlockInfo(String block_type, String name, int level,
      BlockInfoTag tag) {
      if (name == null) {
         name = "?";
      }
      assert tag != null;
      String output;
      switch (tag) {
      case VOID:
         output = getIndentSpace(level) + block_type + " " + name;
         break;
      case NONTRANSLATABLE:
         output = getIndentSpace(level) + block_type + " " + name +
            " (non-translatable)";
         if (USE_COLOURED_OUTPUT) {
            output = PrintUtils.intoRed(output);
         }
         break;
      default:
         throw new AssertionError();
      }
      System.out.println(output);
   }

   private boolean checkTranslatable(MdlBlock block) {
      for(String block_type : DEFAULT_TRANSLATABLE_BLOCKS) {
         if (block.isBlockType(block_type)) {
            return true;
         }
      }
      for(LibraryMetaFile lmf : lmfs) {
         if (lmf.matches(block)) {
            return true;
         }
      }
      return false;
   }

   public static void useColouredOutput(boolean flag) {
      USE_COLOURED_OUTPUT = flag;
   }

   private String getIndentSpace(int level) {
      return PrintUtils.getIndentSpace(level, SPACES_PER_TAB);
   }
}
