package org.zeyda.clawcircus.Data.Diagram;

import org.zeyda.clawcircus.Data.Simulink.MdlBlock;

import org.zeyda.clawcircus.Data.Diagram.BlockTypes.gen.Inport;
import org.zeyda.clawcircus.Data.Diagram.BlockTypes.gen.Outport;
import org.zeyda.clawcircus.Data.Diagram.BlockTypes.gen.EnablePort;
import org.zeyda.clawcircus.Data.Diagram.BlockTypes.gen.TriggerPort;
import org.zeyda.clawcircus.Data.Diagram.BlockTypes.gen.ActionPort;

import org.zeyda.clawcircus.Data.ClaSP.BlockWiring;

import org.zeyda.clawcircus.misc.AnnotationInterface;

import org.zeyda.clawcircus.utils.ClaSPUtils;
import org.zeyda.clawcircus.utils.MdlUtils;
import org.zeyda.clawcircus.utils.MiscUtils;
import org.zeyda.clawcircus.utils.StringUtils;

import org.zeyda.clawcircus.collections.*;
import org.zeyda.clawcircus.collections.impl.*;

import org.zeyda.clawcircus.exceptions.MdlSemanticException;

import java.util.*;

import javax.swing.tree.TreePath;
import javax.swing.tree.TreeModel;

/* Note that getIncomingLinks() and getOutgoingLinks() both return one-based
 * arrays, the zero position is not used. This unifies the access to Link
 * elements to directly use port numbers which too start counting from one. */

public abstract class Block implements Iterable<Block>, Comparable<Block> {
   /* Name of the block in the diagram. */
   protected String name;
   protected SubSystem parent; /* Subsystem including the block. */

   /* Links for all ports of the block. */
   protected Link[] incoming_links;
   protected Link[] outgoing_links;
   protected Map<PortType, Link> special_incoming_links;
   protected Map<PortType, Link> special_outgoing_links;

   /* Members to record properties and annotations. */
   protected Map<String, Object> properties;
   protected Map<Class, Object> annotations;

   protected Block(int inputs, int outputs) {
      createLinks(inputs, outputs);
      properties = new HashMap<String, Object>();
      annotations = new HashMap<Class, Object>();
   }

   protected Block() {
      this(0, 0);
   }

   public String getName() {
      return name;
   }

   public void setName(String name) {
      /*this.name = name.trim();*/
      this.name = name;
   }

   public SubSystem getParent() {
      return parent;
   }

   public void setParent(SubSystem parent) {
      assert !hasParent();
      this.parent = parent;
   }

   public boolean hasParent() {
      return parent != null;
   }

   public boolean isRoot() {
      return !hasParent();
   }

   /* Utility methods for the creation of links. */

   private void createLinks(int incoming_nr, int outgoing_nr) {
      incoming_links = new Link[incoming_nr + 1];
      outgoing_links = new Link[outgoing_nr + 1];
      special_incoming_links = new EnumMap<PortType, Link>(PortType.class);
      special_outgoing_links = new EnumMap<PortType, Link>(PortType.class);
   }

   protected void updateIncomingLinks(int inputs) {
      Link[] new_incoming_links = new Link[inputs + 1];
      System.arraycopy(incoming_links, 0, new_incoming_links, 0,
         Math.min(incoming_links.length, new_incoming_links.length));
      incoming_links = new_incoming_links;
   }

   protected void updateOutgoingLinks(int outputs) {
      Link[] new_outgoing_links = new Link[outputs + 1];
      System.arraycopy(outgoing_links, 0, new_outgoing_links, 0,
         Math.min(outgoing_links.length, new_outgoing_links.length));
      outgoing_links = new_outgoing_links;
   }


   protected void updateLinks(int inputs, int outputs) {
      updateIncomingLinks(inputs);
      updateOutgoingLinks(outputs);
   }

   /* The following method performs initialisation of the Block instance from
    * the associated MDL block. Note that it does not make sense to call this
    * method in the constructor as the MDL block would not have been
    * initialised at this point. */

   public void initialise() throws MdlSemanticException {
      if (hasAnnotation(MdlBlock.class)) {
         MdlBlock mdl_block = getAnnotation(MdlBlock.class);
         /* Infer block name. */
         setName(MdlUtils.getStringAttribute("Name", mdl_block));
         /* Infer number of input ports. */
         if (hasProperty("infer_inputs_dynamically") &&
            getProperty("infer_inputs_dynamically", Boolean.class)) {
            String inputs_str =
               MdlUtils.getStringAttribute("Inputs", mdl_block);
            int inputs;
            if (MiscUtils.canBeParsedAsInt(inputs_str)) {
               inputs = Integer.parseInt(inputs_str);
            }
            else {
               inputs = MdlUtils.filterOpMask(inputs_str).length();
            }
            updateIncomingLinks(inputs);
         }
         /* Infer number of output ports. */
         if (hasProperty("infer_outputs_dynamically") &&
            getProperty("infer_outputs_dynamically", Boolean.class)) {
            String outputs_str =
               MdlUtils.getStringAttribute("Outputs", mdl_block);
            int outputs;
            if (MiscUtils.canBeParsedAsInt(outputs_str)) {
               outputs = Integer.parseInt(outputs_str);
            }
            else {
               outputs = MdlUtils.filterOpMask(outputs_str).length();
            }
            updateOutgoingLinks(outputs);
         }
      }
   }

   /* Methods facilitating access to the Link objects of this block. */

   public Link[] getIncomingLinks() {
      return incoming_links;
   }

   public Link[] getOutgoingLinks() {
      return outgoing_links;
   }

   /* Setter methods for standard links are not required since the respective
    * arrays can be directly modified. */

   public Link getIncomingLink(int index) {
      assert index >= 1 && index <= incoming_links.length;
      return incoming_links[index];
   }

   public Link getOutgoingLink(int index) {
      assert index >= 1 && index <= outgoing_links.length;
      return outgoing_links[index];
   }

   public Link getSpecialIncomingLink(PortType port_type) {
      return special_incoming_links.get(port_type);
   }

   public Link getSpecialOutgoingLink(PortType port_type) {
      return special_outgoing_links.get(port_type);
   }

   public Link setSpecialIncomingLink(PortType port_type, Link link) {
      assert link != null;
      return special_incoming_links.put(port_type, link);
   }

   public Link setSpecialOutgoingLink(PortType port_type, Link link) {
      assert link != null;
      return special_outgoing_links.put(port_type, link);
   }

   public LinkSet getAllIncomingLinks() {
      LinkSet result = new LinkSetImpl();
      for (Link link : incoming_links) {
         result.addIfNotNull(link);
      }
      result.addAll(special_incoming_links.values());
      return result;
   }

   public LinkSet getAllOutgoingLinks() {
      LinkSet result = new LinkSetImpl();
      for (Link link : outgoing_links) {
         result.addIfNotNull(link);
      }
      result.addAll(special_outgoing_links.values());
      return result;
   }

   public LinkSet getAllLinks() {
      LinkSet result = new LinkSetImpl();
      result.addAll(getAllIncomingLinks());
      result.addAll(getAllOutgoingLinks());
      return result;
   }

   public Link getUniqueLink() {
      assert getInputPortsNum() + getOutputPortsNum() == 1;
      if (getInputPortsNum() == 1) {
         return getIncomingLink(1);
      }
      else {
         assert getOutputPortsNum() == 1;
         return getOutgoingLink(1);
      }
   }

   /* Method to facilitate the (dis)connections of links to ports. */

   /* However, to limit the possibility for inconsistency of data, they should
    * not be called directly; instead use the respective methods from the Link
    * class i.e. attachToPort(Port port) and detachFromPort(Port port). */

   public void attachLink(Link link, Port port) {
      assert link != null;
      assert port.isValid();
      assert port.getBlock().equals(this);
      assert !port.isConnected();
      if (port.isStandard()) {
         switch (port.getDir()) {
            case INPUT:
               incoming_links[port.getPortNum()] = link;
               break;

            case OUTPUT:
               outgoing_links[port.getPortNum()] = link;
               break;
         }
      }
      if (port.isSpecial()) {
         switch (port.getDir()) {
            case INPUT:
               special_incoming_links.put(port.getType(), link);
               break;

            case OUTPUT:
               special_outgoing_links.put(port.getType(), link);
               break;
         }
      }
   }

   public void detachLink(Link link, Port port) {
      assert port.isValid();
      assert port.getBlock().equals(this);
      assert port.isConnected();
      if (port.isStandard()) {
         switch (port.getDir()) {
            case INPUT:
               assert incoming_links[port.getPortNum()] == link;
               incoming_links[port.getPortNum()] = null;
               break;

            case OUTPUT:
               assert outgoing_links[port.getPortNum()] == link;
               outgoing_links[port.getPortNum()] = null;
               break;
         }
      }
      if (port.isSpecial()) {
         switch (port.getDir()) {
            case INPUT:
               assert special_incoming_links.get(port.getType()) == link;
               special_incoming_links.put(port.getType(), null);
               break;

            case OUTPUT:
               assert special_outgoing_links.get(port.getType()) == link;
               special_outgoing_links.put(port.getType(), null);
               break;
         }
      }
   }

   /* The next methods seems slightly unsafe, maybe remove at some point. */

   public void detachLink(Port port) {
      assert port.isValid();
      assert port.getBlock().equals(this);
      assert port.isConnected();
      if (port.isStandard()) {
         switch (port.getDir()) {
            case INPUT:
               incoming_links[port.getPortNum()] = null;
               break;

            case OUTPUT:
               outgoing_links[port.getPortNum()] = null;
               break;
         }
      }
      else {
         assert port.isSpecial();
         switch (port.getDir()) {
            case INPUT:
               special_incoming_links.put(port.getType(), null);
               break;

            case OUTPUT:
               special_outgoing_links.put(port.getType(), null);
               break;
         }
      }
   }

   /* Methods facilitating access to the Port objects of this block. */

   public int getInputPortsNum() {
      return incoming_links.length - 1;
   }

   public int getOutputPortsNum() {
      return outgoing_links.length - 1;
   }

   /* Methods to obtain standard ports. */

   public Port getInputPort(int port_num) {
      assert port_num >= 1 && port_num <= getInputPortsNum();
      return new Port(this, port_num, PortDir.INPUT);
   }

   public Port getOutputPort(int port_num) {
      assert port_num >= 1 && port_num <= getOutputPortsNum();
      return new Port(this, port_num, PortDir.OUTPUT);
   }

   /* Methods to obtain special ports. */

   public Port getInputPort(PortType port_type) {
      assert port_type.isSpecial();
      return new Port(this, port_type, PortDir.INPUT);
   }

   public Port getOutputPort(PortType port_type) {
      assert port_type.isSpecial();
      return new Port(this, port_type, PortDir.OUTPUT);
   }

   /* Methods to obtain the cumulative list of all ports. */

   public PortSet getInputPorts() {
      PortSet result = new PortSetImpl();
      for(int port_num = 1; port_num <= getInputPortsNum(); port_num++) {
         result.add(getInputPort(port_num));
      }
      for(PortType port_type : special_incoming_links.keySet()) {
         result.add(getInputPort(port_type));
      }
      return result;
   }

   public PortSet getOutputPorts() {
      PortSet result = new PortSetImpl();
      for(int port_num = 1; port_num <= getOutputPortsNum(); port_num++) {
         result.add(getOutputPort(port_num));
      }
      for(PortType port_type : special_outgoing_links.keySet()) {
         result.add(getOutputPort(port_type));
      }
      return result;
   }

   public PortSet getAllPorts() {
      PortSet result = new PortSetImpl();
      result.addAll(getInputPorts());
      result.addAll(getOutputPorts());
      return result;
   }

   public PortSet getConnectedPorts(Link link) {
      PortSet result = new PortSetImpl();
      for(int index = 1; index <= getInputPortsNum(); index++) {
         if (link == incoming_links[index]) {
            result.add(new Port(this, index, PortDir.INPUT));
         }
      }
      for(int index = 1; index <= getOutputPortsNum(); index++) {
         if (link == outgoing_links[index]) {
            result.add(new Port(this, index, PortDir.OUTPUT));
         }
      }
      for(Map.Entry<PortType, Link> entry : special_incoming_links.entrySet()) {
         if (link == entry.getValue()) {
            result.add(new Port(this, entry.getKey(), PortDir.INPUT));
         }
      }
      for(Map.Entry<PortType, Link> entry : special_outgoing_links.entrySet()) {
         if (link == entry.getValue()) {
            result.add(new Port(this, entry.getKey(), PortDir.OUTPUT));
         }
      }
      return result;
   }

   /* Resolution of blocks into Port objecst of the including subsystem. This
    * method should only be applied to Block objects representing ports. */

   public Port resolveIntoPort() {
      assert isPort();
      if (hasParent()) {
         return getParent().resolveIntoPort(this);
      }
      return null;
   }

   public boolean connects(Link link) {
      return !getConnectedPorts(link).isEmpty();
   }

   /* Generation of ClaSP block wiring information. */

   /* We use an annotation to cache the BlockWiring object once it has been
    * computed, mostly for efficiency reasons. */

   public final BlockWiring getBlockWiring() {
      if (!hasAnnotation(BlockWiring.class)) {
         setAnnotation(calcBlockWiring());
      }
      return getAnnotation(BlockWiring.class);
   }

   /* The default behaviour for calculating block wiring information is to
    * negotiate this task to the ClaSP library utilities. The subclass might
    * like to override this behaviour. */

   public BlockWiring calcBlockWiring() {
      return ClaSPUtils.calcBlockWiringFromLib(this);
   }

   /* Recursively clear block wiring information to allow it to be recomputed
    * from scratch. */

   public void clearAllBlockWiring() {
      removeAnnotation(BlockWiring.class);
      for (Block block : getChildren()) {
         block.clearAllBlockWiring();
      }
   }

   public Path getPath() {
      return new Path(this);
   }

   public String getClawZName() {
      return getPath().getClawZName();
   }

   /* Methods the subclass might have to override. */

   public String getBlockType() {
      return getClass().getSimpleName();
   }

   public abstract BlockList getChildren();

   /* Methods to reduce instanceof operator invocations. */

   public boolean isPrimitive() {
      return !isSubSystem();
   }

   public boolean isPort() {
      return this instanceof PortBlock;
   }

   public boolean isSubSystem() {
      return this instanceof SubSystem;
   }

   public final boolean isInport() {
      return this instanceof Inport;
   }

   public final boolean isOutport() {
      return this instanceof Outport;
   }

   public final boolean isEnablePort() {
      return this instanceof EnablePort;
   }

   public final boolean isTriggerPort() {
      return this instanceof TriggerPort;
   }

   public final boolean isActionPort() {
      return this instanceof ActionPort;
   }

   /* Methods to manage properties. */

   public Object getProperty(String name) {
      assert name != null;
      return properties.get(name);
   }

   public @SuppressWarnings("unchecked") <T> T getProperty(String name,
      Class<T> type) {
      assert name != null;
      Object value = properties.get(name);
      if (value == null) {
         assert false : /* Produce some kind of catchable error here. */
            "Property " + name + " not assigned in block " + toString() + ".";
      }
      assert type.isAssignableFrom(value.getClass());
      return (T) value;
   }

   public <T> void setProperty(String name, T value) {
      assert name != null;
      properties.put(name, value);
   }

   public boolean hasProperty(String name) {
      return properties.containsKey(name);
   }

   /* Methods to manage annotations. */

   public @SuppressWarnings("unchecked") <T> T getAnnotation(Class<T> type) {
      Object value = annotations.get(type);
      if (value != null) {
         assert type.isAssignableFrom(value.getClass());
      }
      return (T) value;
   }

   /* Implementing AnnotationInterface<T> is the safest way to declare and
    * store annotations. */

   public <T> void setAnnotation(AnnotationInterface<T> value) {
      assert value != null;
      /* Constraint for annotation keys (cannot be statically enforced). */
      assert value.getAnnotationKey().isAssignableFrom(value.getClass());
      annotations.put(value.getAnnotationKey(), value);
   }

   /* For flexibility reasons we allow annotations which are do not implement
    * AnnotationInterface<T>. */

   public <T> void setAnnotation(Class<T> type, Object value) {
      assert value != null;
      assert type.isAssignableFrom(value.getClass());
      annotations.put(type, value);
   }

   /* The following method should be used with care as it deduces the type key
    * to store the object automatically. In case of storing objects of subclass
    * hierarchies don't use it, implemented AnnotationInterface<T> instead. */

   public void setAnnotation(Object value) {
      setAnnotation(value.getClass(), value);
   }

   public boolean hasAnnotation(Class type) {
      return annotations.containsKey(type);
   }

   public void removeAnnotation(Class type) {
      annotations.remove(type);
   }

   /* Methods to synchronise changes if this blocks occurs in the context of
    * a JBlockTree GUI control. */

   public TreePath getTreePath() {
      if (hasParent()) {
         return getParent().getTreePath().pathByAddingChild(this);
      }
      else {
         return new TreePath(this);
      }
   }

   public void updateUI() {
      for(Block block : getChildren()) {
         block.updateUI();
      }
      notifyUI();
   }

   protected void notifyUI() {
      notifyUI(getTreePath(), this);
   }

   private void notifyUI(TreePath path, Block newValue) {
      if (hasAnnotation(TreeModel.class)) {
         getAnnotation(TreeModel.class).valueForPathChanged(path, newValue);
      }
      if (hasParent()) {
         ((Block) getParent()).notifyUI(path, newValue);
      }
   }

   /* Implementation of Java API interfaces and Object methods. */

   public Iterator<Block> iterator() {
      return getChildren().iterator();
   }

   public int compareTo(Block obj2) {
      Block obj1 = this;
      /* First order blocks with respect to their block type. */
      if (!obj1.getBlockType().equals(obj2.getBlockType())) {
         Integer i1 =
            BlockComparator.BLOCK_TYPES_ORDER.get(obj1.getBlockType());
         Integer i2 =
            BlockComparator.BLOCK_TYPES_ORDER.get(obj2.getBlockType());
         if (i1 == null && i2 == null) {
            return obj1.getBlockType().compareTo(obj2.getBlockType());
         }
         else {
            if (i1 == null) {
               assert i2 != null;
               return 1;
            }
            if (i2 == null) {
               assert i1 != null;
               return -1;
            }
            return i1 - i2;
         }
      }
      /* Secondly order blocks with respect to their name. */
      if (!obj1.getName().equals(obj2.getName())) {
         return obj1.getName().compareTo(obj2.getName());
      }
      /* In the end we have to resolve to comparing the references in order to
       * be consistent with equals(). Interestingly, there is no requirement
       * for hashCode() to return distinct values for different objects,
       * although the JVM very likely implements it like this. In that it is
       * principally impossible to ensure consistency here. */
      return obj1.hashCode() - obj2.hashCode();
   }

   public @Override String toString() {
      return (name == null ? "<?>" : StringUtils.quote(getName())) + " "
         + "(" + getBlockType() + "/"
         + "[" + getInputPortsNum() + "," + getOutputPortsNum() + "])";
   }
}
