package org.zeyda.clawcircus.Application;

import org.zeyda.clawcircus.Data.Simulink.MdlBlock;

import org.zeyda.clawcircus.Data.Diagram.Block;
import org.zeyda.clawcircus.Data.Diagram.SubSystem;

import org.zeyda.clawcircus.IO.RedirectThread;

import org.zeyda.clawcircus.Toolbox.MdlClawZAdaptor;
import org.zeyda.clawcircus.Toolbox.DiagramConstructor;
import org.zeyda.clawcircus.Toolbox.CircusTranslator;
import org.zeyda.clawcircus.Toolbox.TranslationStrategy;

import org.zeyda.clawcircus.UI.JBlockTree;

import org.zeyda.clawcircus.exceptions.MdlSemanticException;

import org.zeyda.clawcircus.utils.AppUtils;
import org.zeyda.clawcircus.utils.FileUtils;

import org.zeyda.clawcircus.resources.Resources;

import net.sourceforge.czt.z.ast.Spec;

import net.sourceforge.czt.session.SectionManager;
import net.sourceforge.czt.session.Key;
import net.sourceforge.czt.session.Source;
import net.sourceforge.czt.session.StringSource;
import net.sourceforge.czt.session.CommandException;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;

import java.io.File;
import java.io.IOException;

import javax.swing.AbstractAction;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JCheckBox;
import javax.swing.JTree;
import javax.swing.JPopupMenu;
import javax.swing.JMenuItem;
import javax.swing.BoxLayout;

import javax.swing.border.TitledBorder;

import javax.swing.event.ChangeListener;
import javax.swing.event.ChangeEvent;

class CircusTranslatorApp {
   public static final File tmp_dir = new File("tmp");
   private static SubSystem system;
   private static JBlockTree block_tree;

   static {
      GlobalErrorHandler.init();
      CircusTranslator.setSimplify(true);
   }

   private static void welcome() {
      AppUtils.makeWelcome("Circus Translator");
   }

   private static void usage() {
      System.out.println("Usage: circtrans <file>.mdl");
      System.exit(-1);
   }

   private static void loadMdl(String filename) throws MdlSemanticException {
      MdlBlock model = FileUtils.readMdl(filename);
      MdlClawZAdaptor.invoke(model);
      system = DiagramConstructor.invoke(model);
   }

   private static void initUI() {
      JFrame frame = new JFrame("Circus Producer");
      frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
      JPanel top_panel = new JPanel();
      /*top_panel.setLayout(new FlowLayout(FlowLayout.LEFT));*/
      top_panel.setLayout(new BoxLayout(top_panel, BoxLayout.X_AXIS));
      top_panel.setBorder(new TitledBorder("Options"));
      JCheckBox hide_ports_checkbox = new JCheckBox("Enable Simplification");
      hide_ports_checkbox.setSelected(CircusTranslator.getSimplify());
      hide_ports_checkbox.addChangeListener(
         new ChangeListener() {
            public void stateChanged(ChangeEvent e) {
               CircusTranslator.setSimplify(
                  ((JCheckBox) e.getSource()).isSelected());
            }
         }
      );
      top_panel.add(hide_ports_checkbox);
      frame.getContentPane().add(top_panel, BorderLayout.NORTH);
      block_tree = new JBlockTree(system);
      /*block_tree.setToggleClickCount(1);*/
      JPopupMenu popup_menu = new JPopupMenu();
      JMenuItem popup_title = new JMenuItem("Translation Menu");
      popup_title.setEnabled(false);
      popup_menu.add(popup_title);
      popup_menu.addSeparator();
      popup_menu.add("Show LaTeX").addActionListener(
         new AbstractAction() {
            public void actionPerformed(ActionEvent e) {
               doShowLatex();
            }
         }
      );
      popup_menu.add("Show DVI").addActionListener(
         new AbstractAction() {
            public void actionPerformed(ActionEvent e) {
               doShowDVI();
            }
         }
      );
      popup_menu.addSeparator();
      popup_menu.add("Expand All").addActionListener(
         new AbstractAction() {
            public void actionPerformed(ActionEvent e) {
               doExpandAll();
            }
         }
      );
      popup_menu.add("Collapse All").addActionListener(
         new AbstractAction() {
            public void actionPerformed(ActionEvent e) {
               doCollapseAll();
            }
         }
      );
      block_tree.setComponentPopupMenu(popup_menu);
      block_tree.setSelectionPath(system.getTreePath());
      frame.getContentPane().add(
         new JScrollPane(block_tree), BorderLayout.CENTER);
      frame.setSize(520, 600);
      frame.setVisible(true);
      block_tree.requestFocus();
   }

   private static void initSemantics() {
      initSemantics(system);
   }

   private static void initSemantics(SubSystem subsystem) {
      if (subsystem.isRoot()) {
         subsystem.setTranslationStrategy(TranslationStrategy.PARALLEL);
      }
      else {
         subsystem.setTranslationStrategy(TranslationStrategy.CENTRALISED);
      }
      for(Block child : subsystem.getChildren()) {
         if (child.isSubSystem()) {
            initSemantics((SubSystem) child);
         }
      }
   }

   private static void setTranslationStrategy(SubSystem subsystem,
      TranslationStrategy semantics) {
      subsystem.setTranslationStrategy(semantics);
      for(Block block : subsystem.getChildren()) {
         if (block.isSubSystem()) {
            setTranslationStrategy((SubSystem) block, semantics);
         }
      }
   }

   private static int invokeLatex(String filename) {
      Process latex_proc = null;
      try {
         String[] latex_cmd = { "latex", "-halt-on-error", filename };
         String[] latex_env = {
            "TEXINPUTS=../latex/style:",
            /* Setting PATH as well fixes a problem with running circtrans
             * on the York departmental lab machines. A better solution here
             * is to pass all current environment variables with TEXINPUTS
             * overwritten as above. */
            "PATH=" + System.getenv("PATH") };
         latex_proc =
            Runtime.getRuntime().exec(latex_cmd, latex_env, tmp_dir);
         /* It is always crucial to consume any output produced by the
          * instantiated process, otherwise we might end up in a deadlock
          * when the output buffer reaches its limit. */
         RedirectThread redirect_input =
            RedirectThread.create(latex_proc.getInputStream());
         RedirectThread redirect_error =
            RedirectThread.create(latex_proc.getErrorStream());
         latex_proc.waitFor();
         if (latex_proc.exitValue() != 0) {
            redirect_error.flush();
         }
      }
      catch(Exception e) {
         System.out.println(e.getMessage());
         System.exit(-1);
      }
      return latex_proc.exitValue();
   }

   private static void doShowLatex() {
      try {
         File tmp_file = File.createTempFile("trans", ".tex", tmp_dir);
         String tex_file = tmp_file.getPath();
         if (translate(tex_file)) {
            String[] nedit_cmd = { "nedit", tex_file };
            Runtime.getRuntime().exec(nedit_cmd);
         }
      }
      catch(IOException e) {
         System.out.println(e.getMessage());
         System.exit(-1);
      }
   }

   private static void doShowDVI() {
      try {
         File tmp_file = File.createTempFile("trans", ".tex", tmp_dir);
         String tex_file = tmp_file.getPath();
         if (translate(tex_file)) {
            if(invokeLatex(tmp_file.getName()) == 0) {
               String dvi_file =
                  tex_file.substring(0, tex_file.length() - 4) + ".dvi";
               String[] xdvi_cmd = { "xdvi", "-s", "5", dvi_file };
               Runtime.getRuntime().exec(xdvi_cmd);
            }
         }
      }
      catch(IOException e) {
         System.out.println(e.getMessage());
         System.exit(-1);
      }
   }

   private static void doExpandAll() {
      if (!block_tree.isSelectionEmpty()) {
         expandAll(block_tree.getSelectedBlock());
      }
   }

   private static void doCollapseAll() {
      if (!block_tree.isSelectionEmpty()) {
         collapseAll(block_tree.getSelectedBlock());
      }
   }

   private static void expandAll(Block block) {
      block_tree.expandPath(block.getTreePath());
      for (Block child : block.getChildren()) {
         expandAll(child);
      }
   }

   private static void collapseAll(Block block) {
      for (Block child : block.getChildren()) {
         collapseAll(child);
      }
      block_tree.collapsePath(block.getTreePath());
   }

   private static boolean translate(String filename) {
      if (!block_tree.isSelectionEmpty()) {
         Block block = block_tree.getSelectedBlock();
         if (block.getProperty("translatable", Boolean.class)) {
            String latex_doc = CircusTranslator.translateAll(block);
            /*System.out.println(latex_doc);*/
            FileUtils.writeTextFile(filename, latex_doc);
            return true;
         }
      }
      return false;
   }

   private static void parse(String latex_doc) throws CommandException {
      SectionManager sect_man = Resources.createSectionManager();
      Source source = new StringSource(latex_doc);
      sect_man.put(new Key<Source>(source.getName(), Source.class), source);
      Spec spec =
         (Spec) sect_man.get(new Key<Spec>(source.getName(), Spec.class));
   }

   /* Hack, should really make this change in the diagram. */

   private static void removePrefixFromSystemName(String prefix) {
      if (system.getName().startsWith(prefix)) {
         system.setName(system.getName().substring(prefix.length()));
      }
   }

   public static void main(String[] args) throws Exception {
      if (args.length != 1) {
         usage();
      }
      /*welcome();*/
      loadMdl(args[0]);
      removePrefixFromSystemName("ProjetoPiloto");
      initUI();
      initSemantics();
   }
}
