/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.opsdischarge.syntacticfunctions;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import net.sourceforge.czt.base.ast.ListTerm;
import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.util.UnsupportedAstClassException;
import net.sourceforge.czt.base.visitor.ListTermVisitor;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.circus.ast.TransformerPred;
import net.sourceforge.czt.circus.visitor.TransformerPredVisitor;
import net.sourceforge.czt.rules.UnboundJokerException;
import net.sourceforge.czt.rules.rewriter.RewriteUtils;
import net.sourceforge.czt.session.CommandException;
import net.sourceforge.czt.z.ast.ApplExpr;
import net.sourceforge.czt.z.ast.BindExpr;
import net.sourceforge.czt.z.ast.BindSelExpr;
import net.sourceforge.czt.z.ast.ConstDecl;
import net.sourceforge.czt.z.ast.Decl;
import net.sourceforge.czt.z.ast.DecorExpr;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.MemPred;
import net.sourceforge.czt.z.ast.NewOldPair;
import net.sourceforge.czt.z.ast.Pred;
import net.sourceforge.czt.z.ast.Qnt1Expr;
import net.sourceforge.czt.z.ast.QntPred;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.RenameExpr;
import net.sourceforge.czt.z.ast.SchExpr;
import net.sourceforge.czt.z.ast.SetExpr;
import net.sourceforge.czt.z.ast.ZExprList;
import net.sourceforge.czt.z.ast.ZName;
import net.sourceforge.czt.z.ast.ZNumeral;
import net.sourceforge.czt.z.ast.ZSchText;
import net.sourceforge.czt.z.visitor.ApplExprVisitor;
import net.sourceforge.czt.z.visitor.BindSelExprVisitor;
import net.sourceforge.czt.z.visitor.DecorExprVisitor;
import net.sourceforge.czt.z.visitor.ExprVisitor;
import net.sourceforge.czt.z.visitor.MemPredVisitor;
import net.sourceforge.czt.z.visitor.PredVisitor;
import net.sourceforge.czt.z.visitor.Qnt1ExprVisitor;
import net.sourceforge.czt.z.visitor.QntPredVisitor;
import net.sourceforge.czt.z.visitor.RefExprVisitor;
import net.sourceforge.czt.z.visitor.RenameExprVisitor;
import net.sourceforge.czt.z.visitor.SchExprVisitor;
import net.sourceforge.czt.z.visitor.ZExprListVisitor;
import net.sourceforge.czt.z.visitor.ZNumeralVisitor;
import circusRefine.core.InternalManager;
import circusRefine.core.crules.utils.NormalizationApplicationException;
import circusRefine.core.crules.utils.NormalizeApplier;
import circusRefine.core.opsdischarge.SyntacticFunctionsUtils;
import circusRefine.core.util.ChildrenTermExtractor;

/**
 * Classe que efetua a aplicao da Funo FV_Expr: Expr --> \power Name
 * (free-variables of an expression)
 * 
 * @author Cristiano Castro
 */
public class FreeVariablesApplier 
extends SyntacticFunctionApplier<Expr, Expr> {

	/** Termos expandidos */
	private Set<String> termosExpandidos;

	/**
	 * Inicia o objeto informando alguns termos que foram alvos de 
	 * expanso
	 * 
	 * @param termosExpandidos os nomes que foram alvos de expanso 
	 *  quando se expandiu a OP.
	 */
	public FreeVariablesApplier(Set<String> termosExpandidos) {
		this.setTermosExpandidos(termosExpandidos);
	}

	/**
	 * @return the termosExpandidos
	 */
	protected Set<String> getTermosExpandidos() {
		return termosExpandidos;
	}

	/**
	 * @param termosExpandidos the termosExpandidos to set
	 */
	private void setTermosExpandidos(Set<String> termosExpandidos) {
		this.termosExpandidos = termosExpandidos;
	}

	/**
	 * Aplica a funo sinttica FV_Expr a expresso argumento
	 * 
	 * @param args o argumento da funo FV_Expr
	 * @return o conjunto de RefExpr contendo os nomes das variveis 
	 *  livres da funo
	 * @throws CannotEvaluateException caso a expresso no possa ser
	 *  avaliada
	 * 
	 */
	public Expr apply(Expr args) throws CannotEvaluateException {
		try {
			return this.transformarRepresentacaoConjuntoNomes(this.getFV(args));
		} catch (CannotEvaluateRunTimeException e) {
			throw new CannotEvaluateException(e);
		}
	}

	/**
	 * Mtodo utilitrio para buscar o conjunto de nomes das variveis 
	 * livres.
	 * 
	 * @param arg a expresso argumento de FV_Expr
	 * @return o conjunto de nomes avaliados
	 */
	public HashSet<ZName> getFV(Expr arg) throws CannotEvaluateException {
		return this.getFVAux(arg);
	}

	/**
	 * Mtodo utilitrio para buscar o conjunto de nomes das variveis 
	 * livres.
	 * 
	 * @param arg a expresso argumento de FV_Expr
	 * @return o conjunto de nomes avaliados
	 */
	public HashSet<ZName> getFV(Pred arg) throws CannotEvaluateException {
		HashSet<ZName> result = this.getFVAux(arg);
		return result;
	}
	
	/**
	 * Retorna as variveis livres de um termo genrico.
	 *  
	 * @param termo o termo a ser analisado
	 * @return o conjunto de variveis livres do termo
	 */
	private HashSet<ZName> getFVAux(Term termo) throws CannotEvaluateException {
		try {
			FreeVariablesVisitor visitor = new FreeVariablesVisitor();
			return termo.accept(visitor);
		} catch (UnsupportedAstClassException e) {

			/* Caso caia em alguma classe ainda no tratvel */
			throw new CannotEvaluateException(e);
		} catch (CannotEvaluateRunTimeException e) {
			throw new CannotEvaluateException(e);
		}
	}
	
	/**
	 * Classe que implementa os visitors necessrios  aplicao
	 * da funo FV_Expr
	 * 
	 * @author Cristiano Castro
	 */
	protected class FreeVariablesVisitor implements 
	RefExprVisitor<HashSet<ZName>>, TermVisitor<HashSet<ZName>>, 
	SchExprVisitor<HashSet<ZName>>, ApplExprVisitor<HashSet<ZName>>, 
	BindSelExprVisitor<HashSet<ZName>>, DecorExprVisitor<HashSet<ZName>>, 
	ExprVisitor<HashSet<ZName>>, RenameExprVisitor<HashSet<ZName>>, 
	PredVisitor<HashSet<ZName>>, TransformerPredVisitor<HashSet<ZName>>, 
	QntPredVisitor<HashSet<ZName>>, ListTermVisitor<HashSet<ZName>>, 
	ZExprListVisitor<HashSet<ZName>>, ZNumeralVisitor<HashSet<ZName>>, 
	Qnt1ExprVisitor<HashSet<ZName>>, MemPredVisitor<HashSet<ZName>> {

		/**
		 * No pode visitar um termo genrico
		 * 
		 * @param arg0 o termo a ser visitado
		 * @throws CannotEvaluateRunTimeException
		 */
		public HashSet<ZName> visitTerm(Term arg0) {

			/* No pode avaliar um termo genrico */
			String msg = "Visitando um termo genrico";
			throw new CannotEvaluateRunTimeException(msg);

		}
		
		/**
		 * Percorre uma lista de termos
		 * 
		 * @param arg0 a lista de termos a ser visitada
		 * @return o conjunto de nomes livres da lista de termos
		 */
		public HashSet<ZName> visitListTerm(ListTerm arg0) {
			HashSet<ZName> result = new HashSet<ZName>();
			
			/* Visita os elementos da lista */
			for (Object elemento : arg0) {
				if (elemento instanceof Term) {
					result.addAll(((Term) elemento).accept(this)); 
				}
			}
			
			return result;
		}
		
		/**
		 * Visita uma lista de expresses. Percorre a lista capturando
		 * variveis livres na expresso.
		 * 
		 * @param arg0 a lista a ser visitada
		 * @return o conjunto de nomes de variveis livres na lista
		 */
		public HashSet<ZName> visitZExprList(ZExprList arg0) {
			HashSet<ZName> result = new HashSet<ZName>();
			
			/* Captura as variveis livres das expresses na lista */
			for (Expr expr : arg0) {
				result.addAll(expr.accept(this));
			}
			
			return result;
		}
		
		/**
		 * Visita uma expresso genrica. Apenas vai percorrendo a 
		 * rvore procurando por referncias  variveis livres
		 * 
		 * @param arg0 a expresso genrica a ser visitada
		 * @return o conjunto dos nomes de variveis livres 
		 *  capturadas
		 * @see #coletarNomesFilhos(Term)
		 */
		public HashSet<ZName> visitExpr(Expr arg0) {
			return this.coletarNomesFilhos(arg0);
		}
		
		/**
		 * Mtodo utilizado para percorrer a ASt pesquisando os nomes
		 * livres nos filhos de um termo
		 * 
		 * @param termo a raiz da rvore a ser percorrida
		 * @return o conjunto de variveis livres
		 */
		private HashSet<ZName> coletarNomesFilhos(Term arg0) {
			List<Term> filhos = ChildrenTermExtractor.extrairFilhos(arg0);		

			/* Retorna o conjunto de resultado dos filhos */
			HashSet<ZName> result = new HashSet<ZName>();
			for (Term filho : filhos) {
				result.addAll(filho.accept(this));
			}
			return result;
		}

		/**
		 * Visita uma referncia a uma expresso com a finalidade de 
		 * pegar uma referncia a uma varivel livre
		 * 
		 * @param arg0 a {@link RefExpr} a ser visitada
		 */
		public HashSet<ZName> visitRefExpr(RefExpr arg0) {
			HashSet<ZName> result = new HashSet<ZName>();
			if (arg0.getName() instanceof ZName) {

				/* Pode ter o nome tratado */
				ZName ref = arg0.getZName();

				/* 
				 * Testa se a referncia foi expandida em algum 
				 * momento 
				 */
				if (!getTermosExpandidos().contains(ref.getWord())) {

					/* No foi expandido portanto pode ser uma varivel */
					result.add(ref);
				}
			} 

			return result;
		}

		/**
		 * Visita um {@link SchExpr}. As varivies livres so os nomes 
		 * declarados em sua lista de declaraes.
		 * 
		 * @param arg0 o esquema a ser tratado
		 * @return o conjunto das variveis livres no predicado do 
		 *  esquema
		 */
		public HashSet<ZName> visitSchExpr(SchExpr arg0) {
			HashSet<ZName> result;

			/* Normalisa o esquema */

				SchExpr resultRewrite;
				try {
					resultRewrite = 
						NormalizeApplier.getInstance().applyRewrite(arg0);
				} catch (NormalizationApplicationException e) {
					 throw new CannotEvaluateRunTimeException(e); 
				}

				if (resultRewrite.getSchText() instanceof ZSchText) {

					/*  uma expresso de esquema vlida */
					ZSchText schema = resultRewrite.getZSchText();

					/* 
					 * V se tem IncDecl para tirar do Conjunto de 
					 * Nomes
					 */
					result = 
						SyntacticFunctionsUtils.variaveisDeclaradas(
								schema.getZDeclList());


				} else {

					/* arg0 no  vlido */
					result = new HashSet<ZName>();
				}

			return result;
		}

		/**
		 * Visita uma aplicao de funo. O visitor visita 
		 * recursivamente os argumentos da funo para capturar 
		 * variveis livres. O nome da funo  visitado
		 * 
		 * @param arg0 a aplicao de funo a ser avaliada
		 * @throws a excesso de que no pode ser avaliada 
		 */
		public HashSet<ZName> visitApplExpr(ApplExpr arg0) {
			
			/* Os argumentos entram na lista de variveis livres */
			HashSet<ZName> result = arg0.getRightExpr().accept(this);
			return result;
			
		}

		/**
		 * Visita uma seleo de binding. Caso a expresso seja 
		 * avalivel, isto , um simples Binding com uma RefExpr,
		 * ento as FV_Expr  capturada somente na expresso selecionada.
		 * 
		 * @param arg0 a seleo a ser testada
		 * @return o conjunto de nomes (FV_Expr) da expresso selecionada
		 * @throws CannotEvaluateRunTimeException caso a expresso no
		 *  seja avalivel
		 */
		public HashSet<ZName> visitBindSelExpr(BindSelExpr arg0) {
			boolean naoPodeAvaliar = false;
			HashSet<ZName> result = new HashSet<ZName>();
			Exception cause = null;

			try {
				BindExpr binding = (BindExpr) arg0.getExpr();
				Expr selecioned = null;

				/* Se preocupa somente com a expresso selecionada */
				for (Decl decl : binding.getZDeclList()) {
					ConstDecl constDecl = (ConstDecl) decl;

					if (constDecl.getZName().equals(arg0.getZName())) {

						/* Encontrou a expresso selecionada */
						selecioned = constDecl.getExpr();
					}
				}

				/* V se achou a expresso selecionada ou no */
				/* Pode disparar uma NullPointerException */
				result = selecioned.accept(this);

			} catch (UnsupportedAstClassException e) {
				cause = e;
				naoPodeAvaliar = true;
			} catch (ClassCastException e) {
				cause = e;
				naoPodeAvaliar = true;
			} catch (NullPointerException e) {
				cause = e;
				naoPodeAvaliar = true;
			}

			/* Fim da funo */
			if (naoPodeAvaliar) {
				throw new CannotEvaluateRunTimeException(cause);
			} else {
				return result;
			}
		}

		/**
		 * Visita uma expresso decorada. Primeiro normaliza a 
		 * expresso, depois efetua a procura por variveis livres.
		 * 
		 * @param arg0 a expresso a ser normalizada e pesquisada
		 * @return as variveis livres da normalizao
		 * @throws CannotEvaluateRunTimeException caso a expresso no 
		 *  possa normalizada ou avaliada
		 */
		public HashSet<ZName> visitDecorExpr(DecorExpr arg0) {
			ApplExpr applNormalize = RewriteUtils.createNormalizeAppl(arg0);
			HashSet<ZName> result;
			try {
				Term resultRewrite = 
					RewriteUtils.rewrite(applNormalize, 
							InternalManager.getManager(), "expansion_rules");

				/* 
				 * Se conseguiu normalizar ento busca as variveis 
				 * livres na expresso normalizada 
				 */
				result = resultRewrite.accept(this);
			} catch (CommandException e) {
				throw new CannotEvaluateRunTimeException(e);
			} catch (UnboundJokerException e) {
				throw new CannotEvaluateRunTimeException(e);
			}

			return result;
		}

		/**
		 * Visita uma expresso tipo quantificao. As variveis 
		 * quantificada ou "declaradas" no so incluidas na lista
		 * de variveis livres 
		 * 
		 * @param arg0 a expresso do tipo quantificao
		 * @return o conjunto de variveis livres
		 * @throws UnsupportedAstClassException se o esquema declarado 
		 *  em um esquema  
		 */
		public HashSet<ZName> visitQnt1Expr(Qnt1Expr arg0) {
			HashSet<ZName> variaveisLivres = arg0.getExpr().accept(this);
			HashSet<ZName> declaradas = 
				SyntacticFunctionsUtils.variaveisDeclaradas(arg0.getZSchText());
			variaveisLivres.removeAll(declaradas);
			return variaveisLivres;
		}
		
		/**
		 * Visita uma expresso do tipo de renomeao de variveis,
		 * as variavis livres que foram renomeadas so trocadas 
		 * tambm no conjunto de variveis livres
		 *
		 * @param arg0 a expresso com variveis renomeadas
		 * @return a lista de variveis livres da expresso
		 * @throws UnsupportedAstClassException caso alguns mtodos de
		 *  convenincia utilizados da classe no possam ser 
		 *  utilizados.
		 */
		public HashSet<ZName> visitRenameExpr(RenameExpr arg0) {
			HashSet<ZName> variaveisLivres = arg0.getExpr().accept(this);
			
			/* Procura nas renomeacoes */
			for (NewOldPair renomeacao : arg0.getZRenameList()) {
				if (variaveisLivres.remove(renomeacao.getZRefName())) {
					
					/* Realmente removeu o nome antigo */
					variaveisLivres.add(renomeacao.getZDeclName());
				}
			}
			
			/* Retorna o conjunto resultante da renomeacao */
			return variaveisLivres;
		}
		
		/**
		 * Percorre a rvore de Predicados procurando variveis livres
		 * 
		 * @param arg0 o predicado a ser percorrido recursivamente
		 * @return o conjunto de nomes representando o predicado
		 * @see #coletarNomesFilhos(Term)
		 */
		public HashSet<ZName> visitPred(Pred arg0) {
			return this.coletarNomesFilhos(arg0);
		}

		/* 
		 * TODO modificar isso para capturar um nome de funo caso a 
		 * funo seja declarada na especificao 
		 */
		/**
		 * Visita uma funo de pertinncia. Retorna as variveis livres 
		 * dos argumentos da funo.
		 */
		public HashSet<ZName> visitMemPred(MemPred arg0) {
			HashSet<ZName> result;
			if (!arg0.getMixfix() || arg0.getMixfix() && 
					arg0.getRightExpr() instanceof SetExpr ) {
				
				/* Predicado de pertinncia ou igualdade */
				result = this.coletarNomesFilhos(arg0);
			} else {
				
				/* Outro operador qualquer, ignora o nome da funo */
				result = this.coletarNomesFilhos(arg0.getLeftExpr());
			}
			return result;
		}
		
		/**
		 * Visita uma predicado de transfomao. Esses predicados no
		 * possuem variveis livres.
		 * 
		 * @param arg0 o predicado de tranformao a ser visitado
		 * @return um conjunto vazio de nomes
		 */
		public HashSet<ZName> visitTransformerPred(TransformerPred arg0) {
			return new HashSet<ZName>();
		}
		
		/**
		 * Visita um predicado quantificado. Retira das variveis 
		 * livres do predicado, as variveis declaradas na 
		 * quantificao.
		 * 
		 * @param arg0 o predicado do tipo quantificao
		 * @return o conjunto de variveis livres do predicado
		 * @throws UnsupportedAstClassException caso as variveis 
		 *  quantificadas em um esquema no forem declaradas em um
		 *  {@link ZSchText}.
		 */
		public HashSet<ZName> visitQntPred(QntPred arg0) {
			HashSet<ZName> variaveisLivres = arg0.getPred().accept(this);
			HashSet<ZName> declaradas = 
				SyntacticFunctionsUtils.variaveisDeclaradas(arg0.getZSchText());
			variaveisLivres.removeAll(declaradas);
			return variaveisLivres;
		}
		
		/**
		 * Visita um numeral. Este no tem variveis livres.
		 * 
		 * @param arg0 o numeral a ser visitado
		 * @return um conjunto vazio de nomes, pois um numeral no tem
		 *  variveis livres de modo nenhum
		 */
		public HashSet<ZName> visitZNumeral(ZNumeral arg0) {
			return new HashSet<ZName>();
		}
		
	}

}
