/*
 * Projeto: Circus Refine
 * Autor: Cristiano Gurgel de Castro
 */
package circusRefine.core.crules.utils;

import java.util.List;

import circusRefine.core.util.ChildrenTermExtractor;

import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.circus.ast.BasicProcess;
import net.sourceforge.czt.circus.ast.CircusProcess;
import net.sourceforge.czt.circus.ast.ProcessPara;
import net.sourceforge.czt.circus.visitor.BasicProcessVisitor;
import net.sourceforge.czt.circus.visitor.CircusProcessVisitor;
import net.sourceforge.czt.circus.visitor.ProcessParaVisitor;
import net.sourceforge.czt.z.ast.Para;
import net.sourceforge.czt.z.ast.ParaList;
import net.sourceforge.czt.z.ast.Sect;
import net.sourceforge.czt.z.ast.Spec;
import net.sourceforge.czt.z.ast.ZParaList;
import net.sourceforge.czt.z.ast.ZSect;
import net.sourceforge.czt.z.visitor.ParaListVisitor;
import net.sourceforge.czt.z.visitor.ParaVisitor;
import net.sourceforge.czt.z.visitor.SectVisitor;
import net.sourceforge.czt.z.visitor.SpecVisitor;
import net.sourceforge.czt.z.visitor.ZParaListVisitor;
import net.sourceforge.czt.z.visitor.ZSectVisitor;

/**
 * Classe para retornar a lista de pargrafos de uma especificao a
 * a partir de um processo.
 * 
 * @author Cristiano Castro
 */
public class ParaListGetter implements TermVisitor<ZParaList>, 
		SpecVisitor<ZParaList>, ZSectVisitor<ZParaList>, 
		SectVisitor<ZParaList> {

	/**
	 * Busca pela lista de pargrafos onde est definido o 
	 * BasicProcess especificado pela lista de pargrafos dada como
	 * parmetro
	 * 
	 * @param ast a rvore no qual ser efetuada a busca
	 * @param aPesquisar a lista de pargrafos que define o 
	 *  {@link BasicProcess} a ser pesquisado
	 * @return a {@link ZParaList} onde est definido o pargrafo do
	 *  processo especificado
	 */
	public static ZParaList retornarZParaList(Term ast, ParaList aPesquisar) {
		ParaListGetter visitor = new ParaListGetter(aPesquisar);
		return ast.accept(visitor);
	}
	
	private ParaList aPesquisar;

	/**
	 * Cria a classe de procura informando a {@link ParaList} do
	 * {@link BasicProcess} a ser pesquisado 
	 * 
	 * @param aPesquisar a lista de pargrafos de um BasicProcess
	 */
	public ParaListGetter(ParaList aPesquisar) {
		this.setAPesquisar(aPesquisar);
	}
	
	private ParaList getAPesquisar() {
		return aPesquisar;
	}
	
	private void setAPesquisar(ParaList pesquisar) {
		aPesquisar = pesquisar;
	}
	
	public ZParaList visitTerm(Term arg0) {
		List<Term> filhos = ChildrenTermExtractor.extrairFilhos(arg0);
		
		for (Term filho : filhos) {
			ZParaList result = filho.accept(this);
			
			if (result != null) {
				return result;
			}
		}
		return null;
	}

	public ZParaList visitSpec(Spec arg0) {
		
		/* Procura em cada seo de um programa */ 
		for (Sect section : arg0.getSect()) {
			ZParaList result = section.accept(this);
			
			if (result != null) {
				return result;
			}
		}
		
		/* No achou em nenhuma seo */
		return null;
	}
	
	public ZParaList visitZSect(ZSect arg0) {
		PesquisadorBasicProcess visitor = 
			new PesquisadorBasicProcess(this.getAPesquisar());
		
		if (arg0.getParaList().accept(visitor)) {
			
			/* Achou a lista de pargrafos */
			return arg0.getZParaList(); 
		} else {
			
			/* No achou a lista */
			return null;
		}
	}

	public ZParaList visitSect(Sect arg0) {
		return null;
	}
	
	/**
	 * Classe utlizada para pesquisar se o BasicProcess est definido 
	 * na seo especificada
	 * 
	 * @author Cristiano Castro
	 */
	private class PesquisadorBasicProcess implements ZParaListVisitor<Boolean>, 
		ProcessParaVisitor<Boolean>, ParaVisitor<Boolean>, 
		BasicProcessVisitor<Boolean>, CircusProcessVisitor<Boolean>, 
		ParaListVisitor<Boolean> {
		
		/** A ParaList do {@link BasicProcess} a ser pesquisada */
		private ParaList aPesquisar;
		
		/**
		 * Construtor. Informa a ParaList a ser pesquisada
		 * 
		 * @param procurado
		 */
		public PesquisadorBasicProcess(ParaList procurado) {
			this.setAPesquisar(procurado);
		}
		
		private ParaList getAPesquisar() {
			return aPesquisar;
		}
		
		private void setAPesquisar(ParaList pesquisar) {
			aPesquisar = pesquisar;
		}
		
		public Boolean visitPara(Para arg0) {
			return false;
		}
		
		public Boolean visitProcessPara(ProcessPara arg0) {
			return arg0.getCircusProcess().accept(this);
		}
		
		public Boolean visitBasicProcess(BasicProcess arg0) {
			if (arg0.getParaList().equals(this.getAPesquisar())) {
				
				/* Achei a ParaList Procurada */
				return true;
			} else {
				
				/* As ParaLists so diferentes */
				return false;
			}
		}
		
		public Boolean visitCircusProcess(CircusProcess arg0) {
			return false;
		}
		
		public Boolean visitZParaList(ZParaList arg0) {
			
			/* Procura em cada pargrafo */
			for (Para paragrafo : arg0) {
				if (paragrafo.accept(this)) {
					
					/* Achou!! */
					return true;
				}
			}
			
			/* No achou o termo em nenhum pargrafo */
			return false;
		}
		
		public Boolean visitParaList(ParaList arg0) {
			return false;
		}
		
	}

}
