/*
 * Projeto: Circus Refine
 */
package circusRefine.core.crules.utils;

import java.util.Arrays;
import java.util.List;
import java.util.Set;

import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.base.visitor.VisitorUtils;
import net.sourceforge.czt.circus.ast.ActionD;
import net.sourceforge.czt.circus.ast.Communication;
import net.sourceforge.czt.circus.ast.Field;
import net.sourceforge.czt.circus.ast.InputField;
import net.sourceforge.czt.circus.ast.PrefixingAction;
import net.sourceforge.czt.circus.ast.ProcessD;
import net.sourceforge.czt.circus.ast.ProcessPara;
import net.sourceforge.czt.circus.ast.QualifiedDecl;
import net.sourceforge.czt.circus.ast.VarDeclCommand;
import net.sourceforge.czt.circus.visitor.ActionDVisitor;
import net.sourceforge.czt.circus.visitor.CommunicationVisitor;
import net.sourceforge.czt.circus.visitor.FieldVisitor;
import net.sourceforge.czt.circus.visitor.InputFieldVisitor;
import net.sourceforge.czt.circus.visitor.PrefixingActionVisitor;
import net.sourceforge.czt.circus.visitor.ProcessDVisitor;
import net.sourceforge.czt.circus.visitor.ProcessParaVisitor;
import net.sourceforge.czt.circus.visitor.QualifiedDeclVisitor;
import net.sourceforge.czt.circus.visitor.VarDeclCommandVisitor;
import net.sourceforge.czt.circuspatt.ast.CircusPatternFactory;
import net.sourceforge.czt.circuspatt.impl.CircusPatternFactoryImpl;
import net.sourceforge.czt.z.ast.AxPara;
import net.sourceforge.czt.z.ast.ConstDecl;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.Name;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.Stroke;
import net.sourceforge.czt.z.ast.VarDecl;
import net.sourceforge.czt.z.ast.ZName;
import net.sourceforge.czt.z.ast.ZStrokeList;
import net.sourceforge.czt.z.visitor.AxParaVisitor;
import net.sourceforge.czt.z.visitor.ConstDeclVisitor;
import net.sourceforge.czt.z.visitor.VarDeclVisitor;
import net.sourceforge.czt.z.visitor.ZNameVisitor;
import circusRefine.core.crules.anotations.CopyRuleForNameAnn;
import circusRefine.core.opsdischarge.OPsDischargeUtils;
import circusRefine.core.util.ClonerVisitor;
import circusRefine.util.Pair;

/**
 * Busca a definio de uma expresso a partir do nome pesquisado
 * 
 * @author Cristiano Castro
 */
public class ExpressionFromATermGetter extends DefinitionFromATermGetter 
implements AxParaVisitor<Void>, ConstDeclVisitor<Void>, 
ProcessParaVisitor<Void>, ProcessDVisitor<Void>, ActionDVisitor<Void>, 
PrefixingActionVisitor<Void>, VarDeclCommandVisitor<Void> {

	/** 
	 * Booleano para inidicar se o processo no qual o termo reside 
	 * j foi visitado 
	 */
	private Status status;
	
	/**
	 * Construtor com parmetros. Inicia com o nome a ser pesquisado e 
	 * com a classe do termo no qual o nome est encapsulado
	 * 
	 * @param nome o nome a ser pesquisado
	 * @param tipoNome a classe que encapsula o nome
	 */
	public ExpressionFromATermGetter(ZName nome, Term tipoNome) {
		super(nome, tipoNome);
		this.setStatus(Status.NAO_VISITOU_PROCESSO);
	}

	/**
	 * Faz a definio da expresso dentro de um processo prevalecer 
	 * sobre a definio de fora.
	 * 
	 * @param definition definio a ser setada
	 */
	@Override
	protected void setDefinition(Term definition) {
		if (this.getStatus() == null ||
			this.getStatus().equals(Status.NAO_VISITOU_PROCESSO) || 
			this.getStatus().equals(Status.VISITANDO_PROCESSO) ||
			this.getDefinition() == null) {
			super.setDefinition(definition);			
		} 
	}
	
	/**
	 * @return the status
	 */
	private Status getStatus() {
		return status;
	}

	/**
	 * @param status the status to set
	 */
	private void setStatus(Status status) {
		this.status = status;
	}

	/**
	 * Testa se um termo (Lista de declaraes) est declarando o nome
	 * pesquisado
	 * 
	 * @return <code>true</code> caso o nome esteja declarando o nome 
	 *  pesquisado, <code>false</code> caso contrrio
	 */
	private boolean hasNameInDecl(Term toTest) {
		HasNameTester visitor = new HasNameTester(this.getNomeAPesquisar());
		boolean result;
		try {
			toTest.accept(visitor);
			result = false;
		} catch (CancelSearchException e) {
			result = true;
		}		
		return result;
	}
	
	/**
	 * Visita uma definio axiomtica no qual a expresso equivalente
	 * ao nome pesquisado pode estar definida
	 * 
	 * @param arg0 a definio axiomtica pesquisada
	 */
	public Void visitAxPara(AxPara arg0) {
		arg0.getSchText().accept(this);
		return null;
	}

	/**
	 * Visita a declarao constante que pode conter a definio do 
	 * nome pesquisado.
	 * 
	 * @param arg0 a declarao constante visitada
	 */
	public Void visitConstDecl(ConstDecl arg0) {
		
		// TODO tem q fazer uma busca sem comparar os Strokes dos nomes
		String strDecl = arg0.getZName().getWord();
		ZName nomeASubstituir = this.getNomeAPesquisar();
		
		if (strDecl.equals(nomeASubstituir.getWord())) {
			
			/* O nome bate com a declarao */
			Expr definicao = arg0.getExpr();
			definicao = 
				ClonerVisitor.cloneTermRemovingRelationsStack(definicao);
			
			ZStrokeList aAdicionar = nomeASubstituir.getZStrokeList();
			CircusPatternFactory factory = new CircusPatternFactoryImpl();
			
			/* 
			 * Trata os strokes
			 */
			if (definicao instanceof RefExpr) {
				
				/* 
				 * Se for um ZName adiciona os strokes  lista de 
				 * strokes do ZName 
				 */
				ZName nomeDef = ((RefExpr) definicao).getZName(); 
				nomeDef.getZStrokeList().addAll(aAdicionar);
			} else {
				
				/*
				 * Os Strokes so adicionados como DecorExpr
				 */
				
				
				/* Os strokes so adicionados */
				for (Stroke stk : aAdicionar) {
					definicao = factory.createDecorExpr(definicao, stk);
				}
			}
			
			/* 
			 * Testa tambm se o tipo do nome  um RefExpr e caso 
			 * positivo v se h aplicao de funes 
			 */
			Term tipo = this.getTipoNome();
			if (tipo instanceof RefExpr) {
				
				/* Ocorrncia do nome na expresso de referncia */
				RefExpr ocorrencia = (RefExpr) tipo;
				Name nomeOcorrencia = ocorrencia.getName();
				if (nomeOcorrencia instanceof ZName) {
					String strOcorrencia = ((ZName) nomeOcorrencia).getWord();
					
					/* Aplica as funes  definio encontrada */
					definicao = this.criarFuncoes(strOcorrencia, definicao);
				}
				
			}
			
			
			/* A definio foi encontrada!!! */
			this.setDefinition(definicao);
		}
		return null;
	}
	
	/**
	 * 
	 * @param ocorrencia
	 * @param definicao
	 * @return
	 */
	private Expr criarFuncoes(String ocorrencia, Expr definicao) {
		Set<String> funcoes = CopyRuleForNameAnn.funcoesEsquemas();
		Pair<List<String>, String> analise = 
			TermFromANameGetter.capturarSubstringParaTestar(funcoes, 
					ocorrencia);
		if (definicao instanceof RefExpr) {

			/* Adiciona as aplicaes ao ZName do RefExpr */
			StringBuilder novaStr = new StringBuilder();
			for (String funcao : analise.getFirst()) {
				
				/* Adiciona a aplicao de funo */
				novaStr.append(funcao);
				novaStr.append(' ');
			}
			
			/* Mudando o ZName da definio */
			RefExpr ref = (RefExpr) definicao;
			ZName nome = ref.getZName();
			novaStr.append(nome.getWord());
			nome.setWord(novaStr.toString());
			
		} else {
			
			/* Adiciona as funes como ApplExpr */
			for (String funcao : analise.getFirst()) {
				
				CircusPatternFactory factory = new CircusPatternFactoryImpl();
				
				/* Adiciona a aplicao de funo */
				RefExpr refFuncao = 
					OPsDischargeUtils.refFuncao(funcao);
				definicao = 
					factory.createApplExpr(Arrays.asList(refFuncao, definicao), 
							true);
			}
		}
		return definicao;
	}

	/**
	 * Visita o pargrafo do processo no qual pode estar definido o 
	 * termo
	 * 
	 * @param arg0 o pargrafo a ser visitado
	 */
	public Void visitProcessPara(ProcessPara arg0) {
		if (ContainsTermTester.containsTerm(this.getTipoNome(), arg0)) {
			
			/* 
			 * Termo est no processo, procura tambm por definies 
			 * no processo 
			 */
			this.setStatus(Status.VISITANDO_PROCESSO);
			arg0.getCircusProcess().accept(this);
			this.setStatus(Status.VISITOU_PROCESSO);
		}
		
		return null;
	}

	/**
	 * Visita um processo com uma lista de declaraes para ver se o
	 * nome procurado se refere a um nome de varivel e portanto se a 
	 * busca deve parar
	 * 
	 * @param arg0 o processo com declaraes Z
	 */
	public Void visitProcessD(ProcessD arg0) {
		if (hasNameInDecl(arg0.getDeclList())) {
			
			/* 
			 * Lista de declarao est declarando o nome pesquisado.
			 * Logo o resultado da pesquisa no leva a nada
			 */
			throw new NotFoundException();
		} else {
			arg0.getCircusProcess().accept(this);
		}
		return null;
	}
	
	/**
	 * Visita uma ao com lista de declaraes para testar se o nome
	 * pesquisado  apenas um nome de varivel.
	 * 
	 * @param arg0 o nome a ser pesquisado
	 */
	public Void visitActionD(ActionD arg0) {
		if (ContainsTermTester.containsTerm(this.getTipoNome(), 
				arg0.getCircusAction())) {
			
			/* Termo est submetido ao contexto da ao */
			if (this.hasNameInDecl(arg0.getDeclList())) {
				
				/* Busca termina. Nome  apenas uma varivel */
				throw new NotFoundException();
			} else {
				
				/* Busca continua */
				arg0.getCircusAction().accept(this);
			}
		}
		
		return null;
	}
	
	/**
	 * Visita uma ao com comunicao testando se a comunicao tem 
	 * um campo de Input declarando uma varivel com mesmo nome da 
	 * procurada
	 *  
	 * @param arg0 a ao prefixada a ser pesquisada
	 * @return <code>null</code>
	 */
	public Void visitPrefixingAction(PrefixingAction arg0) {
		if (ContainsTermTester.containsTerm(this.getTipoNome(), 
				arg0.getCircusAction())) {
			
			/* Nome pesquisado est na ao */
			if (this.hasNameInDecl(arg0.getCommunication())) {
				
				/* 
				 * Comunicao declara o nome pesquisado. 
				 * Busca acaba 
				 */
				throw new NotFoundException();
			} else {
				
				/* Busca continua */
				arg0.getCircusAction().accept(this);
			}
		}
		return null;
	}
	
	public Void visitVarDeclCommand(VarDeclCommand arg0) {
		if (ContainsTermTester.containsTerm(this.getTipoNome(), 
				arg0.getCircusAction())) {
			
			/* O termo procurado est na ao */
			if (this.hasNameInDecl(arg0.getDeclList())) {
				
				/* Nome est na lista de declaraes. Busca acabou */
				throw new NotFoundException();
			} else {
				
				/* Busca continua */
				arg0.getCircusAction().accept(this);
			}
		}
		
		return null;
	}	
		
	/**
	 * Classe utilizada para representar o estado da busca da 
	 * definio e assim levar em considerao o maior escopo de
	 * definies dentro do processo
	 * 
	 * @author crisgc
	 *
	 */
	private enum Status {
		NAO_VISITOU_PROCESSO,
		VISITANDO_PROCESSO,
		VISITOU_PROCESSO
	}

	private class HasNameTester implements TermVisitor<Void>, 
		VarDeclVisitor<Void>, QualifiedDeclVisitor<Void>,
		ZNameVisitor<Void>, CommunicationVisitor<Void>, 
		InputFieldVisitor<Void>, FieldVisitor<Void> {

		/** Nome a testar se est na lista de declaraes */
		private ZName nome;
		
		/**
		 * Inicia a classe com o nome a ser pesquisado na lista de 
		 * declaraes
		 * 
		 * @param aTestar o nome a ser testado se est na lista de 
		 *  declaraes
		 */
		public HasNameTester(ZName aTestar) {
			this.setNome(aTestar);
		}
		
		/**
		 * @return the nome
		 */
		private ZName getNome() {
			return nome;
		}

		/**
		 * @param nome the nome to set
		 */
		private void setNome(ZName nome) {
			this.nome = nome;
		}

		/**
		 * Efetua a busca na rvore percorrendo os filhos
		 * 
		 * @param arg0 a rvore a ser pesquisada
		 */
		public Void visitTerm(Term arg0) {
			VisitorUtils.visitTerm(this, arg0);
			return null;
		}

		/**
		 * Procura na lista de nomes de uma declarao de varivel
		 * 
		 * @param arg0 a lista de declaraes de varivel
		 */
		public Void visitVarDecl(VarDecl arg0) {
			arg0.getNameList().accept(this);
			return null;
		}
		
		/**
		 * Procura na lista de nomes de uma declarao de varivel
		 * 
		 * @param arg0 a lista de declaraes de varivel a ser
		 *  pesquisada
		 */
		public Void visitQualifiedDecl(QualifiedDecl arg0) {
			arg0.getNameList().accept(this);
			return null;
		}
		
		/**
		 * Visita uma comunicao para testar seus campos
		 * 
		 * @param arg0 a comunicao a ser testada
		 * @return <code>null</code>
		 */
		public Void visitCommunication(Communication arg0) {
			arg0.getFieldList().accept(this);
			return null;
		}
		
		/**
		 * 
		 */
		public Void visitInputField(InputField arg0) {
			arg0.getVariableName().accept(this);
			return null;
		}
		
		/**
		 * Visita um {@link Field} que no  um InputField. Nesse caso
		 * a busca  podada aqui.
		 */
		public Void visitField(Field arg0) {
			return null;
		}
		
		/**
		 * Visita um nome presente na lista de declaraes para
		 * testar se ele  igual ao pesquisado
		 */
		public Void visitZName(ZName arg0) {
			if (arg0.getWord().equals(this.getNome().getWord())) {
				
				/* Achou o nome procurado na lista de declaraoes */
				/* Busca cancelada */
				throw new CancelSearchException();
			}
			return null;
		}
		
	}
}
