/*
 * Projeto: Circus Refine
 */
package circusRefine.core.crules.parseArgument;

import java.io.StringWriter;
import java.util.List;

import circusRefine.core.util.ChildrenTermExtractor;
import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.z.ast.DeclList;
import net.sourceforge.czt.z.ast.ZSchText;
import net.sourceforge.czt.z.visitor.ZSchTextVisitor;

/**
 * Um Visitor para retornar um DeclList a partir da AST. Usado no
 * parse de Argumentos
 *  
 * @author Cristiano Gurgel
 */
public class DeclListGetter implements TermVisitor<DeclList>, 
		ZSchTextVisitor<DeclList> {

	/**
	 * Retorna um DeclList a partir de uma AST, usando, para isso,
	 * uma instncia de {@link DeclList}.
	 * 
	 * @param ast a AST a ser visitada
	 * @return um DeclList contido na AST
	 */
	public static DeclList getADeclList(Term ast) {
		DeclListGetter getter = new DeclListGetter();
		DeclList result = ast.accept(getter);
		return result;
	}
	
	
	/** Log para a documentao */
	private StringWriter log;
	
	/**
	 * Construtor Padro. Somente cria um log para o objeto
	 */
	public DeclListGetter() { 
		this.setLog(new StringWriter());
		this.addMessage("Novo log do DeclListGetter: " + this);
	}
	
	
	/**
	 * Acessa o log da classe
	 * 
	 * @return a informacao da classe
	 */
	public String getInfo() {
		return this.getLog().toString();
	}
	
	private StringWriter getLog() {
		return log;
	}
	
	private void setLog(StringWriter log) {
		this.log = log;
	}
	
	
	/**
	 * Adiciona uma mensagem ao log de unificacao
	 * 
	 * @param msg a mensagem a ser adicionada
	 */
	private void addMessage(String msg) {
		this.getLog().append(msg + "\n");
	}	
	
	/**
	 * Percorre a rvore procurando uma instncia de 
	 * {@link DeckList}.
	 * 
	 * @param arg0 a AST a ser visitada.
	 * @return o {@link DeclList} encontrado na rvore ou 
	 * 	<code>null</code> caso esse no seja encontrado.
	 */
	public DeclList visitTerm(Term arg0) {
		this.addMessage("Visitando termo: " + arg0);
		List<Term> filhos = ChildrenTermExtractor.extrairFilhos(arg0);
		
		for (Term filho : filhos) {
			DeclList result = filho.accept(this);
			if (result != null) {
				return result;
			}
		}
		
		/* No encontrei o DeclList na Ast*/
		return null;
	}

	/**
	 * Caso encontre uma instncia de ZSchText, ento o DeclList
	 * 	da mesma e retornado 
	 * 
	 * @param arg0 o ZSchText a ser visitado.
	 * @return o DeclList encontrado no ZSchText.
	 */
	public DeclList visitZSchText(ZSchText arg0) {
		this.addMessage("Visitando um SchText: " + arg0);
		return arg0.getDeclList();
	}

}
