/*
 * Projeto: Circus Refine
 */
package circusRefine.core.crules.parseArgument;

import java.io.StringWriter;
import java.util.List;

import circusRefine.core.util.ChildrenTermExtractor;
import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.circus.ast.CircusAction;
import net.sourceforge.czt.circus.visitor.CircusActionVisitor;

/**
 * Um Visitor para retornar um CircusAction a partir da AST. Usado no
 * parse de Argumentos
 *  
 * @author Cristiano Gurgel
 */
public class CircusActionGetter implements TermVisitor<CircusAction>, 
		CircusActionVisitor<CircusAction> {

	/**
	 * Retorna um CircusAction a partir de uma AST, usando, para isso,
	 * uma instncia de {@link CircusActionGetter}.
	 * 
	 * @param ast a AST a ser visitada
	 * @return um CircusAction contido na AST
	 */
	public static CircusAction getACircusAction(Term ast) {
		CircusActionGetter getter = new CircusActionGetter();
		CircusAction result = ast.accept(getter);
		return result;
	}
	
	/** Log para a documentao */
	private StringWriter log;
	
	/**
	 * Construtor Padro. Somente cria um log para o objeto
	 */
	public CircusActionGetter() { 
		this.setLog(new StringWriter());
		this.addMessage("Novo log do CircusActionGetter: " + this);
	}
	
	
	/**
	 * Acessa o log da classe
	 * 
	 * @return a informacao da classe
	 */
	public String getInfo() {
		return this.getLog().toString();
	}
	
	private StringWriter getLog() {
		return log;
	}
	
	private void setLog(StringWriter log) {
		this.log = log;
	}
	
	
	/**
	 * Adiciona uma mensagem ao log de unificacao
	 * 
	 * @param msg a mensagem a ser adicionada
	 */
	private void addMessage(String msg) {
		this.getLog().append(msg + "\n");
	}	
	
	/**
	 * Percorre a rvore procurando uma instncia de 
	 * {@link CircusAction}.
	 * 
	 * @param arg0 a AST a ser visitada.
	 * @return o {@link CircusAction} encontrado na rvore ou 
	 * 	<code>null</code> caso esse no seja encontrado.
	 */
	public CircusAction visitTerm(Term arg0) {
		this.addMessage("Visitando termo: " + arg0);
		List<Term> filhos = ChildrenTermExtractor.extrairFilhos(arg0);
		
		for (Term filho : filhos) {
			CircusAction result = filho.accept(this);
			if (result != null) {
				return result;
			}
		}
		
		/* No encontrei o CircusAction na Ast*/
		return null;
	}

	/**
	 * Caso encontre uma instncia de CircusAction, ento o mesmo  
	 * 	retornado
	 * 
	 * @param arg0 o {@link CircusAction} a ser visitado.
	 * @return o {@link CircusAction} encontrado
	 */
	public CircusAction visitCircusAction(CircusAction arg0) {
		this.addMessage("Visitando um CircusAction: " + arg0);
		return arg0;
	}

}
