/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 * Orientador: Marcel Oliveira <marcel@dimap.ufrn.br>
 */
package circusRefine.core.crules.anotations.op;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.circuspatt.ast.CircusPatternFactory;
import net.sourceforge.czt.circuspatt.impl.CircusPatternFactoryImpl;
import net.sourceforge.czt.z.ast.And;
import net.sourceforge.czt.z.ast.Decl;
import net.sourceforge.czt.z.ast.DeclList;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.IffPred;
import net.sourceforge.czt.z.ast.MemPred;
import net.sourceforge.czt.z.ast.Name;
import net.sourceforge.czt.z.ast.Pred;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.VarDecl;
import net.sourceforge.czt.z.ast.ZDeclList;
import net.sourceforge.czt.z.ast.ZFactory;
import net.sourceforge.czt.z.ast.ZName;
import net.sourceforge.czt.z.visitor.DeclVisitor;
import net.sourceforge.czt.z.visitor.VarDeclVisitor;
import net.sourceforge.czt.zpatt.ast.Binding;
import net.sourceforge.czt.zpatt.ast.JokerDeclList;
import net.sourceforge.czt.zpatt.ast.JokerPred;
import net.sourceforge.czt.zpatt.ast.JokerPredBinding;
import circusRefine.core.crules.CRulesException;
import circusRefine.core.crules.CircusLawApplicationException;
import circusRefine.core.crules.anotations.LawApplAnn;
import circusRefine.core.crules.messages.MessagesManager;
import circusRefine.core.util.ChildrenTermExtractor;

/**
 * Classe utilizada para montar as OPs da lei de rearranjamento da
 * lista de declaraes
 * 
 * @author Cristiano Castro
 */
public class RearrangeDeclListPOMounter implements ProofObligationMounterAnn {
	
	/** Lista original de lista de declaraes */
	private JokerDeclList original;
	
	/** Lista rearranjada pelo usurio */
	private JokerDeclList rearranjed;
	
	/** Joker a unificar com a obrigao de prova montada */
	private JokerPred jokerPO;

	/**
	 * Construtor que informa as listas de declaraes do esquema
	 * antes e depois do rearranjamento
	 * 
	 * @param orig a lista de declaraes do esquema <b>antes</b> do 
	 *  rearranjamento
	 * @param rear a lista de declaraes do esquema <b>depois</b> do
	 *  rearranjamento
	 */
	public RearrangeDeclListPOMounter(JokerDeclList orig, JokerDeclList rear, 
			JokerPred po) {
		this.setOriginal(orig);
		this.setRearranjed(rear);
		this.setJokerPO(po);
	}
	
	private JokerDeclList getOriginal() {
		return original;
	}

	private void setOriginal(JokerDeclList original) {
		this.original = original;
	}

	private JokerDeclList getRearranjed() {
		return rearranjed;
	}

	private void setRearranjed(JokerDeclList rearranjed) {
		this.rearranjed = rearranjed;
	}

	private JokerPred getJokerPO() {
		return jokerPO;
	}

	private void setJokerPO(JokerPred jokerPO) {
		this.jokerPO = jokerPO;
	}
	
	/**
	 * Monta a Obrigao de prova que confere os tipos dos tipos das
	 * variveis na lista de declaraes
	 */ 
	public Set<Term> mount(Set<Binding> unification) 
			throws CRulesException {
		
		try {
			
			/* Lista de declaraes */
			ZDeclList orig = (ZDeclList) 
				LawApplAnn.findOriginal(this.getOriginal().getName(), 
						unification);
			ZDeclList rear = (ZDeclList) 
				LawApplAnn.findOriginal(this.getRearranjed().getName(), 
						unification);

			Map<Name, Expr> pairsOrig = this.getPairs(orig);
			Map<Name, Expr> pairsRear = this.getPairs(rear);
			
			/* Monta o predicado */
			CircusPatternFactory factory = new CircusPatternFactoryImpl();
			Pred po = this.predMounter(pairsOrig, pairsRear, factory);
			JokerPredBinding binding = 
				factory.createJokerPredBinding(this.getJokerPO(), po);
			unification.add(binding);
			
		} catch (ClassCastException e) {
			
			/* Cast nas listas de declaraes resultou em excesso */
			String msg = MessagesManager.getInstance().getMessage("Invalid" +
					"DeclListCRulesMsg");
			throw new CircusLawApplicationException(msg, e);
		} catch (RearranjeDeclListException e) {
			
			/* erro qualquer em tempo de execuo */
			String code = e.getMessage();
			String msg = MessagesManager.getInstance().getMessage(code);
			throw new CircusLawApplicationException(msg, e);
		}
		
		return Collections.singleton((Term) this.getJokerPO());
	}
	
	/**
	 * Monta uma conjuno de tipos do predicado de acordo com os
	 * pares de (nome, expr) especificando o tipo de cada varivel
	 * 
	 * @param set o conjunto de pares (nome de varivel, tipo)
	 * @return uma conjuno de predicados de pertinncia para cada
	 *  varivel e seu tipo
	 */
	private Pred predMounter(Map<Name, Expr> orig, 
			Map<Name, Expr> newDList, ZFactory factory) {
		
		/* Monta o conjunto de predicados de pertinncia */
		Set<Pred> conj = new HashSet<Pred>();
		for (Map.Entry<Name, Expr> elem : orig.entrySet()) {
			
			/* Captura o nome da entrada atual */
			Name tempOrig = elem.getKey();
			if (tempOrig instanceof ZName) {
				String strName = ((ZName) tempOrig).getWord();
				
				/* Busca a entrada na segunda lista */
				Name nomeRear = this.getName(newDList.entrySet(), strName);
				if (nomeRear != null) {
					
					/* 
					 * Monta a expresso de pertinncia da lista de 
					 * declaraes original 
					 */
					RefExpr name1 = factory.createRefExpr(tempOrig, 
							factory.createZExprList(), false, false); 
					MemPred memPred1 = 
						factory.createMemPred(Arrays.asList(name1, 
							elem.getValue()), false);
					
					/* 
					 * Monta a expresso de pertinncia da lista de 
					 * declaraes rearranjada
					 */
					RefExpr name2 = factory.createRefExpr(nomeRear, 
							factory.createZExprList(), false, false);
					Expr type = newDList.get(nomeRear);
					MemPred memPred2 = 
						factory.createMemPred(Arrays.asList(name2, type), 
								false);
					
					IffPred toAdd = 
						factory.createIffPred(Arrays.asList(memPred1, 
								memPred2));
					conj.add(toAdd);
				}
			}
			
			
		}
		
		/* Monta a conjuno dos predicados de pertinncia */
		Iterator<Pred> itr = conj.iterator();
		Pred result;
		if (itr.hasNext()) {
			result = itr.next();
			result = this.createAndPred(itr, result, factory);
		} else {
			result = factory.createTruePred();
		}
 		
		return result;
	}
	
	/**
	 * Retorna um nome no conjunto de entradas a partir de uma string
	 * com o nome.
	 * 
	 * @param set conjunto de entradas
	 * @param name o nome a ser pesquisado
	 * @return o {@link Name} encontrado ou <code>null</code> caso
	 *  este no seja encontrado
	 */
	private Name getName(Set<Entry<Name, Expr>> set, String name) {
		Name result = null;
		
		/* Efetua a busca as entradas do conjunto */
		for (Map.Entry<Name, ?> entry : set) {
			Name temp = entry.getKey();
			if (temp instanceof ZName) {
				
				/* Testa se achou o ZName correspondente a string */
				result = ((ZName)temp).getWord().equals(name) ? temp : result;
			}
		}
		return result;
	}
	
	/**
	 * Cria uma conjuno dos predicados de tipos
	 * 
	 * @param itr o iterador sobre o conjunto das clusulas
	 * @param pred o predicado que eu quero unificar a conjuno
	 * @param factory uma fbrica para os predicados
	 * @return a conjuno formada a partir do iterador apresentado
	 */
	private Pred createAndPred(Iterator<Pred> itr, Pred pred, 
			ZFactory factory) {
		Pred result = pred;
		if (itr.hasNext()) {
			Pred next = itr.next();
			Pred nextPred = this.createAndPred(itr, next, factory);
			result = factory.createAndPred(Arrays.asList(pred, nextPred), 
					And.NL);
		}
		return result;
	}
	
	/**
	 * Captura os pares (variavel, tipo) de uma lista de declara
	 * 
	 * @param list a lista de declaraes a ser pesquisada
	 * @return o conjunto com as com os pares das variveis com os
	 *  respectivos tipos
	 * @throws RearranjeDeclListException caso a lista possua
	 *  outros tipos de declaraes que no a declarao de
	 *  varivel.
	 */
	public Map<Name, Expr> getPairs(DeclList list) {
		PairNameTypeGetter getter = new PairNameTypeGetter();
		return list.accept(getter);
	}
	
	/**
	 * Recupera a lista de pares (nome, predicado) para montar as
	 * obrigaes de prova.
	 * 
	 * @author Cristiano Castro
	 */
	private class PairNameTypeGetter implements 
		TermVisitor<Map<Name, Expr>>, 
		VarDeclVisitor<Map<Name, Expr>>,
		DeclVisitor<Map<Name, Expr>> {
		
		
		
		/**
		 *   
		 */
		public Map<Name, Expr> visitTerm(Term arg0) {
			List<Term> filhos = ChildrenTermExtractor.extrairFilhos(arg0);
			Map<Name, Expr> result = new HashMap<Name, Expr>();
			for (Term filho : filhos) {
				result.putAll(filho.accept(this));
			}
			return result;
		}
		
		/**
		 * Adiciona pares do tipo (varivel, expresso)
		 */
		public Map<Name, Expr> visitVarDecl(VarDecl arg0) {
			Map<Name, Expr> result = new HashMap<Name, Expr>();
			for (Name nome : arg0.getZNameList()) {
				result.put(nome, arg0.getExpr());
			}
			return result;
		}
		
		/**
		 * Declarao invalida
		 * 
		 * @throws RearranjeDeclListException
		 */
		public Map<Name, Expr> visitDecl(Decl arg0) {
			String code = "InvalidDeclarationCRulesMsg";
			throw new RearranjeDeclListException(code);
		}
		
	}
	
	/**
	 * Excesso de tempo de execuo para o rearranjamento das 
	 * obrigaes de prova
	 * 
	 * @author Cristiano Castro
	 */
	private class RearranjeDeclListException extends RuntimeException {

		/** Nmero para a serializao */
		private static final long serialVersionUID = 2569025696918886693L;

		public RearranjeDeclListException() {
			super();
			// TODO Auto-generated constructor stub
		}

		public RearranjeDeclListException(String message, Throwable cause) {
			super(message, cause);
			// TODO Auto-generated constructor stub
		}

		public RearranjeDeclListException(String message) {
			super(message);
			// TODO Auto-generated constructor stub
		}

		public RearranjeDeclListException(Throwable cause) {
			super(cause);
			// TODO Auto-generated constructor stub
		}
		
	}

}
