/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.crules;

import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Locale;
import java.util.Set;

import org.w3c.dom.ls.LSException;

import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.util.UnmarshalException;
import net.sourceforge.czt.circus.ast.ActionTransformerPred;
import net.sourceforge.czt.circus.ast.CircusAction;
import net.sourceforge.czt.circus.ast.ProcessTransformerPred;
import net.sourceforge.czt.circus.ast.SchExprAction;
import net.sourceforge.czt.circus.ast.SeqAction;
import net.sourceforge.czt.circus.ast.Transformation;
import net.sourceforge.czt.circus.ast.TransformerPred;
import net.sourceforge.czt.circus.ast.VarDeclCommand;
import net.sourceforge.czt.circus.impl.BasicProcessImpl;
import net.sourceforge.czt.circus.impl.ParallelActionImpl;
import net.sourceforge.czt.circus.impl.ProcessParaImpl;
import net.sourceforge.czt.circus.util.Factory;
import net.sourceforge.czt.circuspatt.impl.CircusPatternFactoryImpl;
import net.sourceforge.czt.circuspatt.util.CircusLaw;
import net.sourceforge.czt.parser.util.ParseException;
import net.sourceforge.czt.rules.unification.UnificationException;
import net.sourceforge.czt.session.CommandException;
import net.sourceforge.czt.z.ast.Fact;
import net.sourceforge.czt.z.ast.Pred;
import net.sourceforge.czt.z.ast.SchExpr;
import net.sourceforge.czt.z.ast.ZName;
import net.sourceforge.czt.z.impl.ZNameImpl;
import net.sourceforge.czt.zpatt.ast.Binding;
import net.sourceforge.czt.zpatt.ast.Sequent;
import net.sourceforge.czt.zpatt.ast.SequentList;
import circusRefine.Tactic.Util.GerenciadorTaticas;
import circusRefine.core.ExternalManager;
import circusRefine.core.InternalManager;
import circusRefine.core.LawAnswer;
import circusRefine.core.annotations.StatementSchExprAnn;
import circusRefine.core.annotations.TemporaryTermAnn;
import circusRefine.core.crules.anotations.LawApplAnn;
import circusRefine.core.crules.anotations.op.ProofObligationMounterAnn;
import circusRefine.core.crules.messages.MessagesManager;
import circusRefine.core.crules.parseArgument.ParseArgumentException;
import circusRefine.core.crules.utils.TermFromANameGetter;
import circusRefine.core.opsdischarge.OPsDischargeUtils;
import circusRefine.core.print.Printer;
import circusRefine.core.util.BindingGetterVisitor;
import circusRefine.core.util.ClonerVisitor;
import circusRefine.core.util.JokerNameGetterVisitor;
import circusRefine.core.util.JokersGetterVisitor;
import circusRefine.core.util.POLog;
import circusRefine.util.OPTipos;
import circusRefine.util.Pair;

/**
 * A classe responsável pela unificacao de dois termos
 * 
 * @author Cristiano Gurgel
 */
public class CRulesUtils {

	/** 
	 * Modo de aplicação da lei: Diferencia se � uma aplicação de 
	 * uma nova lei ou se � uma aplicação proveniente de um 
	 * salvamento
	 */
	private ApplicationMode modoAplicacao;
	
	/**
	 * Gerenciador interno. Utilizado para pegar os par�metros e ter 
	 * acesso a AST original
	 */
	private InternalManager interno;
	
	/**
	 * Utilizado para pegar par�metros no caso de refinamentos Salvos
	 */
	private ListIterator<String> parametros;

	/**
	 * Inicia o Utils com o gerenciador interno.
	 * 
	 * @param manager o gerenciador interno utilizado na busca por 
	 * parametros e no acesso a arvore principal do programa 
	 */
	public CRulesUtils(InternalManager manager, Locale local) 
	{
		this.setInterno(manager);
		this.setModoAplicacao(ApplicationMode.NORMAL);
		MessagesManager.initInstance(local);
	}

	/**
	 * @return the interno
	 */
	public InternalManager getInterno() {
		return interno;
	}

	/**
	 * @param interno the interno to set
	 */
	private void setInterno(InternalManager interno) {
		this.interno = interno;
	}

	/**
	 * @return the modoAplicacao
	 */
	private ApplicationMode getModoAplicacao() {
		return modoAplicacao;
	}

	/**
	 * @param modoAplicacao the modoAplicacao to set
	 */
	private void setModoAplicacao(ApplicationMode modoAplicacao) {
		this.modoAplicacao = modoAplicacao;
	}

	/**
	 * @return the parametros
	 */
	private ListIterator<String> getParametros() {
		return parametros;
	}

	/**
	 * @param parametros the parametros to set
	 */
	private void setParametros(ListIterator<String> parametros) {
		this.parametros = parametros;
	}

	/**
	 * Muda o estilo da aplicação de lei para o modo de abertura 
	 * de programa
	 * 
	 * @param parametros a lista de parametros para a aplicação da lei
	 */
	public void mudarParaModoOpenSave(List<String> params) {
		this.setModoAplicacao(ApplicationMode.OPEN_SAVE);
		if (params != null) {
			this.setParametros(params.listIterator());
		}
	}
	
	/**
	 * Muda o estilo de aplicação de lei para o modo normal. 
	 */
	public void mudarParaModoNormal() {
		this.setModoAplicacao(ApplicationMode.NORMAL);
	}
	
	/**
	 * Efetua a unificação dos parametros. Caso o modo de aplicação 
	 * seja o normal, os parametros sao retirados de telas de 
	 * parametros; caso contrario, os parametros sao retirados de um 
	 * iterador sobre Strings.
	 * 
	 * @return os Bindings com as ligações sobre os parametros
	 * @throws CommandException 
	 * @throws UnmarshalException 
	 * @throws IOException 
	 * @throws ParseException 
	 * @throws CancelledApplException 
	 */
	public Set<Binding> getParameters(List<Term> jokers, LawAnswer resposta) 
		throws CRulesException, CancelledApplException {
		if (this.getModoAplicacao().equals(ApplicationMode.NORMAL)) {
			
			/* Unifica utilizando as telas de parametros */
			return this.getInterno().unificarParametros(jokers, resposta);
		} else {
			
			/* Unifica Utilizando as strings salvas */
			return this.getInterno().unificarParametros(jokers, 
					this.getParametros(), resposta);
		}
	}
	
	public Set<Binding> getParametersTactics(List<Term> jokers, List<Object> args, 
			LawAnswer resposta, List<Pair<String,Term>> mapeamento) 
	throws CRulesException, CancelledApplException {
	/*if (this.getModoAplicacao().equals(ApplicationMode.NORMAL)) {
		
		 Unifica utilizando as telas de parametros 
		return this.getInterno().unificarParametros(jokers, resposta);
	} else {*/
		
		/* Unifica Utilizando as strings salvas */
		return this.getInterno().unificarParametrosTatica(jokers, 
				args.listIterator(), resposta, mapeamento);
	//}
}

	
	/**
	 * Caso o termo no qual foi aplicado a lei tenha a anotação de 
	 * termo temporario, ele encapsula o termo gerado pela aplicação
	 * da lei
	 * 
	 * @param oldTerm o termo no qual foi aplicado a lei
	 * @param newTerm o termo gerado pela aplicação da lei
	 */
	public void unwrapFromTransformer(Term oldTerm, Term newTerm) {
		TemporaryTermAnn ann = oldTerm.getAnn(TemporaryTermAnn.class);
				
		if (ann != null) {

			/* Testa qual filho eu devo estrair do novo termo */
			int index = 0;
			for (Object child : oldTerm.getChildren()) {

				if (child.equals(ann.getRealTerm())) {

					Set<TemporaryTermAnn> anns = 
						new HashSet<TemporaryTermAnn>();
					
					/* Remove a indicicação de capsulas anteriores */
					for (Object anotacao : newTerm.getAnns()) {
						if (anotacao instanceof TemporaryTermAnn) {
							anns.add((TemporaryTermAnn)anotacao);
						}
					}
					for (TemporaryTermAnn anotacao : anns) {
						newTerm.getAnns().remove(anotacao);
					}
					
					/* Coloca a anotação de capsula ao newTerm */
					Term filho = (Term)newTerm.getChildren()[index];
					TemporaryTermAnn newAnn = new TemporaryTermAnn(filho);
					newTerm.getAnns().add(newAnn);
					
					break;
				}
				index++;
			}
			
		}
	}
	
	/**
	 * Aplica uma lei a um termo. de 
	 * 
	 * @param noPrograma a indentificacao do termo onde sera aplicada 
	 * 		a lei
	 * @param selectionedLaw a lei que sera aplicada
	 * @param ger o gerenciador Interno do CRefine, permite o acesso a
	 * 	AST atual do programa e permite tambam requisitar por 
	 *  parametros do usuario.
	 *  @param ast A arvore sintatica do desenvolvimento utilizado
	 * @return a resposta a lei aplicada
	 */
	public LawAnswer applyLaw(Term applicable, CircusLaw selectionedLaw, Term ast) 
		throws CRulesException, CancelledApplException {

		Transformation type;
		Term leiAntes;
		Term leiDepois;
		LawAnswer result = new LawAnswer();
		
		if (selectionedLaw.isActionLaw()) {
			
			/* Aplica lei de acao */
			ActionTransformerPred trans = selectionedLaw.getActionRel();
			type = trans.getTransformation();
			leiAntes = trans.getSpec();
			leiDepois = trans.getImpl();

		} else if (selectionedLaw.isProcessLaw()){
			
			/* Aplica lei de processos */
			ProcessTransformerPred trans = selectionedLaw.getProcessRel();
			type = trans.getTransformation();
			leiAntes = trans.getSpec();
			leiDepois = trans.getImpl();
			
		} else {
			
			/* Informa ao usuario que a lei nao a valida */
			throw new CircusLawApplicationException(selectionedLaw, 
					"LawNotValidCRulesMsg");
		}
		
		/* Unificação */
		Set<Binding> unificacao;
		
		try {
			
			/* tenta a unificacao */
			
			unificacao = UnifierVisitor.unify(applicable, leiAntes);
			
		} catch (UnificationException e) {
			
			/* 
			 * Se o tipo de tranformacao for uma equivalencia,
			 * ele tenta uma nova unificacao
			 */
			if (type == Transformation.Equivalence) {
				
				/* Troca */
				Term aux = leiAntes;
				leiAntes = leiDepois;
				leiDepois = aux;
				/* Tenta nova unificacao */
				try {
					unificacao = UnifierVisitor.unify(applicable, leiAntes);
					
				} catch (UnificationException exception) {
					System.out.println(selectionedLaw.getName());
					
					throw new CircusLawApplicationException(selectionedLaw, 
							exception);
				}
				
			} else {
				
				/* 
				 * Nao eh uma equivalencia! Nao ha esperanca de 
				 * nova unificacao 
				 */
				throw new CircusLawApplicationException(selectionedLaw, e);
			}
		}
		
		SequentList op = selectionedLaw.getProvisoos();
		
		/* Construcao #1 */
		Term novo = BuilderVisitor.build(leiDepois, unificacao);
		
		/* Checa se a lei não tem parâmetros */
		List<Term> parametros = JokersGetterVisitor.getJokers(novo);
		Set<Term> parametrosARemover = new HashSet<Term>();
		/* Se o conjunto de parametros nao for vazia */
		if (!parametros.isEmpty()) {
			/*
			 * busca por anotações especiais de aplicação de lei 
			 */
			for (Term parametro : parametros) {
				List<Object> anns = parametro.getAnns();
				
				for (Object ann : anns) {
					if (ann instanceof LawApplAnn) {

						/* Aplica, a lei, a anotação */
						try {
							Set<Term> aRemover = 
								((LawApplAnn)ann).apply(this, unificacao, 
										parametro, result);
							parametrosARemover.addAll(aRemover);
							
						} catch (Exception e) {
							e.printStackTrace();
							/* 
							 * Dispara a excessao gerada na anotação 
							 * de aplicação da lei 
							 */
							if (e instanceof CircusLawApplicationException) {
								throw (CircusLawApplicationException)e;
							} else {
								throw new 
								CircusLawApplicationException(selectionedLaw,e);
							}
						}
					}
				}
			}
			/*
			 * Remove os parametros com anotações especiais
			 */
			for (Term parametro : parametrosARemover) {
				parametros.remove(parametro);
			}
			/* Retira os elementos repetidos dos parametros */
			this.retirarRepetidos(parametros);
			/* Pega os parametros */
			Set<Binding> binds;
			try {
				
				binds = this.getParameters(parametros, result);
			} catch (ParseArgumentException e) {
				
				// TODO parei aqui
				throw new CircusLawApplicationException( selectionedLaw , e );
			}

			/* Checa se algum parametro foi retornado como null */
			for (Binding bind : binds) {
				Pair<String, Term> mapeamento = 
					BindingGetterVisitor.getMap(bind);
				
				/* Argumento a null, lei nao a aplicada */
				if (mapeamento.getSecond() == null) {
					throw new CircusLawApplicationException(selectionedLaw, 
							"InvallidArgCRulesMsg");
				}
			}
			unificacao.addAll(binds);
			
			/* Monta o term, levando em conta os parametros */
			novo = BuilderVisitor.build(novo, unificacao);
		}
		
		/* Monta as Obrigações de Prova */
		SequentList buildOP = 
			(SequentList)BuilderVisitor.build(op, unificacao);
		
		/* Va se ainda tem alguma obrigação de prova nao montada */
		List<Term> unmountedOps = JokersGetterVisitor.getJokers(buildOP);
		
		/* Lista nao vazia de OPs nao montadas */
		for (Term joker : unmountedOps) {
			ProofObligationMounterAnn mounter = 
				joker.getAnn(ProofObligationMounterAnn.class);
			if (mounter != null) {
				mounter.mount(unificacao);
			}
		}
		
		/* Tenta nova unificação das OPs */
		buildOP = (SequentList)BuilderVisitor.build(buildOP, unificacao);
		
		/* Atualizacao */
		this.unwrapFromTransformer(applicable, novo);
		Term novaAST;
		if ( ExternalManager.isTemporaryTerm( applicable ) ) {
			Term applicableTemp = 
				applicable.getAnn(TemporaryTermAnn.class).getRealTerm();
			Term novoTemp = 
				novo.getAnn(TemporaryTermAnn.class).getRealTerm();
			
			novaAST = UpdateVisitor.update(applicableTemp, novoTemp, ast,true);
		}
		else if (applicable.getAnn(StatementSchExprAnn.class)!= null) {
			CircusPatternFactoryImpl factory = new CircusPatternFactoryImpl();
			
			Term applicableTemp = 
				applicable.getAnn(StatementSchExprAnn.class).getTermo();
			
			SchExpr capsula = factory.createSchExpr();
			capsula.getAnns().add(new StatementSchExprAnn(novo));
			novaAST = UpdateVisitor.update(applicableTemp, capsula, ast,true);
		}
		else {
			
			novaAST = UpdateVisitor.update(applicable, novo, ast,true);
		}
		
		
		/* Sempre apas o tarmino de aplicação o modo de aplicação a mudado  */
		this.mudarParaModoNormal();
		
		/* Monta a resposta da lei */
		result.setProgResultante(novo);
		result.setNovaAST(novaAST);
		result.iniciarOPs(buildOP);
		
		/* Gera as obrigações extendidas */
		this.gerarOPsExtendidas(ast, 
				applicable, novo, result.getOps());
		return result;
		
	}
	
	
	/**
	 * Aplica uma lei a um termo. de 
	 * 
	 * @param noPrograma a indentificacao do termo onde sera aplicada 
	 * 		a lei
	 * @param selectionedLaw a lei que sera aplicada
	 * @param ger o gerenciador Interno do CRefine, permite o acesso a
	 * 	AST atual do programa e permite tambam requisitar por 
	 *  parametros do usuario.
	 *  @param ast A arvore sintatica do desenvolvimento utilizado
	 * @return a resposta a lei aplicada
	 */
	public LawAnswer applyLawTactic(Term applicable, CircusLaw selectionedLaw, Term ast,
			List<Object> args, List<Pair<String,Term>> map) 
		throws CRulesException, CancelledApplException {


		Transformation type;
		Term leiAntes;
		Term leiDepois;
		LawAnswer result = new LawAnswer();
		
		if (selectionedLaw.isActionLaw()) {
			
			/* Aplica lei de acao */
			ActionTransformerPred trans = selectionedLaw.getActionRel();
			type = trans.getTransformation();
			leiAntes = trans.getSpec();
			leiDepois = trans.getImpl();
			
			if (applicable instanceof ProcessParaImpl ||
					applicable instanceof BasicProcessImpl){
				/**
				 * Verifica se o argumento passado para aplicar a lei 
				 * pode ser uma ação. Transforma o argumento em um Term
				 * e applicable recebe esse Term
				 */
				ZName name = getInterno().getZName(args.get(0)); 
				Term acao = TermFromANameGetter.getTerm(name, ast);
				applicable = acao;
			}

		} else if (selectionedLaw.isProcessLaw()){
			
			/* Aplica lei de processos */
			ProcessTransformerPred trans = selectionedLaw.getProcessRel();
			type = trans.getTransformation();
			leiAntes = trans.getSpec();
			leiDepois = trans.getImpl();
			
		} else {
			
			/* Informa ao usuario que a lei nao a valida */
			throw new CircusLawApplicationException(selectionedLaw, 
					"LawNotValidCRulesMsg");
		}
		
		/* Unificação */
		Set<Binding> unificacao;
		
		try {
			
			/* tenta a unificacao */
			
			unificacao = UnifierVisitor.unify(applicable, leiAntes);
			
		} catch (UnificationException e) {
			
			/* 
			 * Se o tipo de tranformacao for uma equivalencia,
			 * ele tenta uma nova unificacao
			 */
			if (type == Transformation.Equivalence) {
				
				/* Troca */
				Term aux = leiAntes;
				leiAntes = leiDepois;
				leiDepois = aux;
				/* Tenta nova unificacao */
				try {
					unificacao = UnifierVisitor.unify(applicable, leiAntes);
					
				} catch (UnificationException exception) {
					System.out.println(selectionedLaw.getName());
					return result;
				}
				
			} else {
				
				/* 
				 * Nao eh uma equivalencia! Nao ha esperanca de 
				 * nova unificacao 
				 */
				throw new CircusLawApplicationException(selectionedLaw, e);
			}
		}
		
		SequentList op = selectionedLaw.getProvisoos();
		
		/* Construcao #1 */
		Term novo = BuilderVisitor.build(leiDepois, unificacao);
		
		/* Checa se a lei não tem parâmetros */
		List<Term> parametros = JokersGetterVisitor.getJokers(novo);
		Set<Term> parametrosARemover = new HashSet<Term>();
		/* Se o conjunto de parametros nao for vazia */
		if (!parametros.isEmpty()) {
			/*
			 * busca por anotações especiais de aplicação de lei 
			 */
			for (Term parametro : parametros) {
				List<Object> anns = parametro.getAnns();
				
				for (Object ann : anns) {
					if (ann instanceof LawApplAnn) {

						/* Aplica, a lei, a anotação */
						try {
							Set<Term> aRemover = 
								((LawApplAnn)ann).apply(this, unificacao, 
										parametro, result);
							parametrosARemover.addAll(aRemover);
							
						} catch (Exception e) {
							e.printStackTrace();
							/* 
							 * Dispara a excessao gerada na anotação 
							 * de aplicação da lei 
							 */
							if (e instanceof CircusLawApplicationException) {
								throw (CircusLawApplicationException)e;
							} else {
								throw new 
								CircusLawApplicationException(selectionedLaw,e);
							}
						}
					}
				}
			}
			/*
			 * Remove os parametros com anotações especiais
			 */
			for (Term parametro : parametrosARemover) {
				parametros.remove(parametro);
			}
			/* Retira os elementos repetidos dos parametros */
			this.retirarRepetidos(parametros);
			/* Pega os parametros */
			Set<Binding> binds;
			try {
				
				binds = this.getParametersTactics(parametros, args, result, map);
			} catch (ParseArgumentException e) {
				
				// TODO parei aqui
				throw new CircusLawApplicationException( selectionedLaw , e );
			}

			/* Checa se algum parametro foi retornado como null */
			for (Binding bind : binds) {
				Pair<String, Term> mapeamento = 
					BindingGetterVisitor.getMap(bind);
				
				/* Argumento a null, lei nao a aplicada */
				if (mapeamento.getSecond() == null) {
					throw new CircusLawApplicationException(selectionedLaw, 
							"InvallidArgCRulesMsg");
				}
			}
			unificacao.addAll(binds);
			
			/* Monta o term, levando em conta os parametros */
			novo = BuilderVisitor.build(novo, unificacao);
		}
		
		/* Monta as Obrigações de Prova */
		SequentList buildOP = 
			(SequentList)BuilderVisitor.build(op, unificacao);
		
		/* Va se ainda tem alguma obrigação de prova nao montada */
		List<Term> unmountedOps = JokersGetterVisitor.getJokers(buildOP);
		
		/* Lista nao vazia de OPs nao montadas */
		for (Term joker : unmountedOps) {
			ProofObligationMounterAnn mounter = 
				joker.getAnn(ProofObligationMounterAnn.class);
			if (mounter != null) {
				mounter.mount(unificacao);
			}
		}
		
		/* Tenta nova unificação das OPs */
		buildOP = (SequentList)BuilderVisitor.build(buildOP, unificacao);
		
		/* Atualizacao */
		this.unwrapFromTransformer(applicable, novo);
		Term novaAST;
		if ( ExternalManager.isTemporaryTerm( applicable ) ) {
			Term applicableTemp = 
				applicable.getAnn(TemporaryTermAnn.class).getRealTerm();
			Term novoTemp = 
				novo.getAnn(TemporaryTermAnn.class).getRealTerm();
			
			novaAST = UpdateVisitor.update(applicableTemp, novoTemp, ast,true);
		}
		else if (applicable.getAnn(StatementSchExprAnn.class)!= null) {
			CircusPatternFactoryImpl factory = new CircusPatternFactoryImpl();
			
			Term applicableTemp = 
				applicable.getAnn(StatementSchExprAnn.class).getTermo();
			
			SchExpr capsula = factory.createSchExpr();
			capsula.getAnns().add(new StatementSchExprAnn(novo));
			novaAST = UpdateVisitor.update(applicableTemp, capsula, ast,true);
		}
		else {
			
			Term oldAst = ast;
			novaAST = UpdateVisitor.update(applicable, novo, ast,true);
			if (oldAst.equals(novaAST)){
				String law = selectionedLaw.getName();
				if (law.equalsIgnoreCase("Var Exp Seq")){
					novaAST = getInterno().varrerSpec(applicable, novo, ast);
				}
				else if(law.equalsIgnoreCase("Parallel Composition Sequence step")){
					novaAST = getInterno().ParSeqStep(applicable, novo, ast);
				}
				else if(law.equalsIgnoreCase("Parallel composition commutativity")){
					novaAST = getInterno().ParComm(applicable, novo, ast);
				}
				else if(law.equalsIgnoreCase("Sequence Associativity")){
					novaAST = getInterno().SeqAssoc(applicable, novo, ast);
				}
				else if(law.equalsIgnoreCase("Sequence Unit part 2")){
					novaAST = getInterno().SeqUnit2(applicable, novo, ast);
				}
				else if(law.equalsIgnoreCase("Var Exp Rec")){
					novaAST = getInterno().VarExpRec(applicable, novo, ast);
				}
			}
		}
		
		
		/* Sempre apas o tarmino de aplicação o modo de aplicação a mudado  */
		this.mudarParaModoNormal();
		
		/* Monta a resposta da lei */
		if (!GerenciadorTaticas.controlTFails){
		
		result.setNovaAST(novaAST);
		result.iniciarOPs(buildOP);
		}
		result.setProgResultante(novo);
		
		/* Gera as obrigações extendidas */
		this.gerarOPsExtendidas(ast, 
				applicable, novo, result.getOps());
		return result;
		
	}
		
	/**
	 * Testa a aplicabilidade de uma lei a um Termo
	 * 
	 * @param prog o programa no qual a lei seria aplicada
	 * @param lei a lei que seria aplicada
	 * @return <code>true</code> caso a lei puder ser aplicada,
	 *  <code>false</code>, caso contrario
	 */
	public boolean testApplicability(Term prog, CircusLaw lei) {

		/* o LHS e o RHS da lei */
		Term left;
		Term right;
		TransformerPred pred;
		
		if (lei.isActionLaw()) {

			/* O LHS e o RHS recebem seus valores */
			pred = lei.getActionRel();
			left = lei.getActionRel().getCircusAction().get(0);
			right = lei.getActionRel().getCircusAction().get(1);
		} else if (lei.isProcessLaw()) {
			
			/* O LHS e o RHS recebem seus valores */
			pred = lei.getProcessRel();
			left = lei.getProcessRel().getCircusProcess().get(0);
			right = lei.getProcessRel().getCircusProcess().get(1);
		} else {
			
			/* Lei Invalida */
			return false;
		}

		try {
			UnifierVisitor.unify(prog, left);
			return true; /* Unifica!! */
		} catch (UnificationException e){
			
			Transformation leiTrans = pred.getTransformation();
			
			/* Testa uma nova unificacao  */
			if (leiTrans.equals(Transformation.Equivalence)) {
				try {
					UnifierVisitor.unify(prog, right);
					return true; /* Unifica!! */
				} catch (UnificationException e1) {
					return false; /* NaoUnifica */
				} catch (Exception e2) {
					return false;
				}
			} else {
				
				/* Tranformação nao a uma equivalancia */
				return false;
			}
		}
		
	}
	
	/**
	 * Retira os elementos repetidos da lista de parametros
	 * 
	 * @param parametros a lista inicial dos parametros
	 * @return a mesma instancia da lista dos parametros sa que agora
	 *  com os elementos repetidos removidos
	 */
	private List<Term> retirarRepetidos(List<Term> parametros) {
		Set<String> nomes = new HashSet<String>();
		Set<Term> aRemover = new HashSet<Term>();
		
		/* Percorre a lista de Jokers */
		for (Term parametro : parametros) {
			String nome = JokerNameGetterVisitor.getName(parametro);
			
			/* Va se o nome do joker ja esta presente */
			if (nomes.contains(nome)) {
				aRemover.add(parametro);
			} else {
				nomes.add(nome);
			}
		}
		
		/* Remove os elementos repetidos */
		for (Term repetido : aRemover) {
			parametros.remove(repetido);
		}
		
		/* Retorna a mesma lista passada como parametro */
		return parametros;
	}

	/**
	 * Metodo que a a partir das obrigações de prova da lei. Monta 
	 * as obrigações de prova estendidas
	 * 
	 * @param antigaAST a ast antes da aplicação da lei
	 * @param aplicacao o termo alvo da aplicação da lei
	 */
	public void gerarOPsExtendidas(Term antigaAST, Term alvoAplicacao, 
			Term resultanteAplicacao, 
			List<Pair<Pair<Sequent, Sequent>, Pair<OPTipos, POLog>>> ops) 
	{
		
		/* Monta a lista de ops extendidas */
		for (Pair<Pair<Sequent, Sequent>, Pair<OPTipos, POLog>> op : ops) {
			Sequent opNormal = op.getFirst().getFirst();
			Sequent opExtendida = 
				ClonerVisitor.cloneTermRemovingRelationsStack(opNormal);

			/* Tenta expandir a OP */
			Pred novaOP = OPsDischargeUtils.expandirOP(opNormal, antigaAST, 
					alvoAplicacao, resultanteAplicacao,interno );
			
			/* Modifica o predicado extendido */
			opExtendida.setPred(novaOP);

			/* Adiciona a OP extendida ao par */
			op.getFirst().setSecond(opExtendida);
		}
		
		
	}	
}
 