/*
 * Projeto: Circus Refine
 */
package circusRefine.core.astmodifiers;

import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.circus.ast.CircusAction;
import circusRefine.core.util.ClonerVisitor;

/**
 * <p>Classe utilizada para inserir uma ao como um argumento de 
 * funo. Ela deve ser uma anotao em uma expresso (RefExpr) que 
 * representem um argumento para uma funo</p>
 * 
 * <p>Uma aplicao de uma funo f: (EXPR x ACTION x EXPR) -> BOOLEAN
 * ter uma lista de expresses de tamanho 3 e o segundo argumento
 * ser um RefExpr com uma {@link ActionArgumentAnn}.
 * </p>
 * 
 * @author Cristiano Castro
 */
public class ActionArgumentAnn implements IASTModifierAnn {
	
	/** 
	 * Mapa com as aes que so argumentos da funo. Cada ao que 
	 * pode ser o argumento da funo tem como chave o inteiro que
	 * indica sua posio
	 */
	private CircusAction acao;
	
	/**
	 * Inicia o argumento com uma CircusAction e sua posio na lista 
	 * de argumentos
	 * 
	 * @param argumento a ao
	 * @param posicao a posio da lista de argumentos
	 */
	public ActionArgumentAnn(CircusAction argumento) {
		
		/* Cria a anotao com a ao */
		this.setAcao(argumento);
	}

	/**
	 * @return the acao
	 */
	private CircusAction getAcao() {
		return acao;
	}

	/**
	 * @param acao the acao to set
	 */
	private void setAcao(CircusAction acao) {
		this.acao = acao;
	}

	/**
	 * Retorna a ao.
	 * 
	 * @return a ao encapsulada pela anotao
	 */
	public CircusAction getTerm() {
		return this.getAcao();
	}
	
	/**
	 * Seta a ao encapsulada pela anotao
	 * 
	 * @param newTerm a nova ao
	 */
	public void setTerm(Term newTerm) {
		this.setAcao((CircusAction) newTerm);
	}
	
	/**
	 * Retorna um clone da anotao
	 * 
	 * @return o clone da anotao sem as pilhas de relacionamento
	 */
	public IASTModifierAnn clone() {
		CircusAction c = 
			ClonerVisitor.cloneTermRemovingRelationsStack(this.getAcao());
		return new ActionArgumentAnn(c);
	}

	/**
	 * Cria uma cpia da anotao com <code>null</code> como argumento
	 * 
	 * @return uma {@link ActionArgumentAnn} encapsulando 
	 *  <code>null</code>
	 */
	public ActionArgumentAnn create() {
		return new ActionArgumentAnn(null);
	}
	
}
