package jcircus.gui;
import java.awt.ComponentOrientation;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;
import jcircus.JCircusControllerFrame;
import jcircus.exceptions.FailParsingException;
import jcircus.exceptions.FailTypeCheckingException;
import jcircus.util.ProcInfo;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.VarDecl;

/**
 * ParametersDialog.java
 *
 * Dialog where the user enters the parameters for a parametrised process.
 *
 * @author Angela
 */
public class ParametersDialog extends JDialog implements ActionListener {

    private JCircusControllerFrame _controller;
    private ProcInfo _procInfo;
    
    // Interface components
    private JTextField[] _inputFields;
    private JButton _btnOk;
    private JButton _btnCancel;

    /**
     * Constructor.
     */
    public ParametersDialog(JCircusFrame jCircusFrame, 
            JCircusControllerFrame controller, ProcInfo procInfo) {
        super(jCircusFrame, "Parameters", true);
        
        this._controller = controller;
        this._procInfo = procInfo;
        
        // User can not close by clicking on "X"
        this.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
        this.getContentPane().setLayout(
                new BoxLayout(this.getContentPane(), BoxLayout.Y_AXIS));
        
        String[] text = new String[] {
            "Process " + procInfo.getProcessName() + " requires parameters.",
            "Please enter their values."
        };
        
        for (int i = 0; i < text.length; i++) {
            this.getContentPane().add(new JLabel(text[i]));
        }
        
        List<VarDecl> params = procInfo.getParameters();
        
        JPanel pnlParams = new JPanel();
        pnlParams.setLayout(new GridLayout((params.size() + 1), 3));
        pnlParams.setComponentOrientation(ComponentOrientation.LEFT_TO_RIGHT);
        
        pnlParams.add(new JLabel("Parameter"));
        pnlParams.add(new JLabel("Type"));
        pnlParams.add(new JLabel("Value"));
        
        _inputFields = new JTextField[params.size()];
        for (int i = 0; i < params.size(); i++) {
            // VarDecl declares at most one variable. Gets this variable.

        	//No tenho ctz se  assim
        	String param = params.get(i).getName()/*getDeclName()*/.get(0).toString();
            
        	// Gets the expression in the VarDecl.
            Expr expr = params.get(i).getExpr();
            
            // The type expression should be a name, that is, a reference to 
            // a Free Type or to \nat or \num or \arithmos. In the future, 
            // other types will be considered as well.
            assert(expr instanceof RefExpr): "expr is " + expr.getClass();
            
            //No tenho ctz se  assim
            String type = (String) ((RefExpr) expr).getName()/*getRefName()*/.toString();

            pnlParams.add(new JLabel(param));
            pnlParams.add(new JLabel(type));
            _inputFields[i] = new JTextField();
            pnlParams.add(_inputFields[i]);
        }
        
        this.getContentPane().add(pnlParams);
        
        JPanel pnlButtons = new JPanel();
        _btnOk = new JButton("OK");
        _btnOk.setSize(20, 10);
        _btnOk.addActionListener(this);
        pnlButtons.add(_btnOk);

        _btnCancel = new JButton("Cancel");
        _btnCancel.setSize(20, 10);
        _btnCancel.addActionListener(this);
        pnlButtons.add(_btnCancel);
        
        this.getContentPane().add(pnlButtons);
    }
    
    /**
     * This is invoked when a button is pressed.
     */
    public void actionPerformed(ActionEvent e) {
    
        if (e.getSource() == _btnOk) {
            btnOkPressed();
        } else if (e.getSource() == _btnCancel) {
            btnCancelPressed();
        }
    }
    
    /**
     * This is invoked when the button OK is pressed.
     *
     */
    private void btnOkPressed() {
    
        // Tests if all the parameters have been entered by the user
        for (int i = 0; i < _inputFields.length; i++) {
            if (_inputFields[i].getText().equals("")) {
                JOptionPane.showMessageDialog(this, 
                        "You must enter all the parameters.");
                return;
            }
        }
        
        for (int i = 0; i < _inputFields.length; i++) {
            
            // Parameter entered by the user
            String paramEntered = _inputFields[i].getText().trim();
            // Expression representing the type of the respective parameter
            Expr type = _procInfo.getTypeOfParameter(i);

            try {
                // "Parse" and "typecheck" the value entered by the user
                String codeForParam = _controller.codeForParameter(
                            paramEntered, type);
                
                // Add the code to the ProcInfo
                _procInfo.addCodeForActuals(i, codeForParam);

            } catch(FailParsingException fpe) {
                JOptionPane.showMessageDialog(this, fpe.getFirstError(),
                        "Parsing Error", JOptionPane.ERROR_MESSAGE);
                _inputFields[i].selectAll();

            } catch(FailTypeCheckingException ftce) {
                JOptionPane.showMessageDialog(this, ftce.getFirstError(),
                        "Type Checking Error", JOptionPane.ERROR_MESSAGE);
                _inputFields[i].selectAll();
            }
        }

        // Finishes and closes the window
        ((JCircusFrame) this.getOwner()).setCanceled(false);
        this.setVisible(false);
        this.dispose();
    }
    
    /**
     * This is invoked when the button Cancel is pressed.
     *
     */    
    private void btnCancelPressed() {

        // Finishes and closes the window
        ((JCircusFrame) this.getOwner()).setCanceled(true);
        this.setVisible(false);
        this.dispose();
    }    
}
