/*
 * Projeto: Circus Refine
 */
package circusRefine.core.storage;

import java.io.Serializable;
import java.util.LinkedList;
import java.util.ListIterator;
import circusRefine.core.CircusRefine;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamAsAttribute;

/**
 * Classe que representa o historico do programa, quais acoes foram feitas
 * sobre diferentes termos, quais termos foram coletados, o que foi feito,
 * desfeito e refeito.
 * 
 * @author Alessandro
 */
@XStreamAlias("Refinement")
public class ToSave implements Serializable {

	/** Nmero para a serializao */
	private static final long serialVersionUID = 9137708960354558182L;
	
	/** Nmero da verso do CRefine */
	@XStreamAsAttribute
	private String version;
	
	/** Histrico da execuo do refinamento */
	private LinkedList<StepOfExecution> historico;
	
	/** Histrico com os passos para o redo */
	private LinkedList<StepOfExecution> redoList;

	/**
	 * Inicia o ToSave com a verso corrente do CRefine e com os 
	 * passos de execuo iniciados em vazio.
	 */
	public ToSave() {
		this.setVersion(CircusRefine.version);
		this.setHistorico(new LinkedList<StepOfExecution>());
		this.setRedoList(new LinkedList<StepOfExecution>());
	}
	
	public ToSave(LinkedList<StepOfExecution> historico) {
		this.setVersion(CircusRefine.version);
		this.setHistorico(historico);
		this.setRedoList(new LinkedList<StepOfExecution>());
	}

	public LinkedList<StepOfExecution> getHistorico() {
		return historico;
	}

	private void setHistorico(LinkedList<StepOfExecution> historico) {
		this.historico = historico;
	}

	/**
	 * @return the redoList
	 */
	public LinkedList<StepOfExecution> getRedoList() {
		return redoList;
	}

	/**
	 * @param redoList the redoList to set
	 */
	private void setRedoList(LinkedList<StepOfExecution> redoList) {
		this.redoList = redoList;
	}

	/**
	 * @return the version
	 */
	public String getVersion() {
		return version;
	}

	/**
	 * @param version the version to set
	 */
	private void setVersion(String version) {
		this.version = version;
	}
	
	public <T extends StepOfExecution> void addStep(T step) {
		this.getHistorico().add(step);
	}
	
	/**
	 * Retira a ltima ao da pilha de histrico e empilha na 
	 * pilha de redo
	 * 
	 * @return o passo a desfazer
	 */
	public StepOfExecution desfazer() {
		
		StepOfExecution passo = this.getHistorico().removeLast();
		this.getRedoList().add(passo);
		return passo;
	}
	
	/**
	 * Retira a ltima ao da pilha de redo e empilha na pilha 
	 * de histrico
	 * 
	 * @return o passo a refazer
	 */
	public StepOfExecution refazer() {
		
		/* 
		 * Adiciona os passos desfeitos ao histrico de 
		 * desenvolvimento 
		 */
		StepOfExecution passo = this.getRedoList().removeLast();
		this.getHistorico().add(passo);
		return passo;
	}
	
	/**
	 * Limpa as listas que armazenam o histrico e permitem que se 
	 * inicie um novo desenvolvimento
	 */
	public void clear() {
		this.getHistorico().clear();
		this.getRedoList().clear();
	}
	
	/**
	 * Retorna o iterador para a lista de histrico
	 * 
	 * @return o iterador
	 */
	public ListIterator<StepOfExecution> getIterator() {
		return this.getHistorico().listIterator();
	}

}
