/*
 * Projeto: Circus Refine
 */
package circusRefine.core.storage;

import java.util.List;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamAsAttribute;
import net.sourceforge.czt.circuspatt.util.CircusLaw;
import circusRefine.core.ExternalManager;
import circusRefine.core.crules.CRulesException;
import circusRefine.core.crules.factories.LawNumberAnn;

/**
 * Classe que representa a ao da aplicao de uma lei
 *  
 * @author Cristiano Castro
 */
@XStreamAlias("LawApp")
public class LawApplication implements StepOfExecution {

	/**
	 * Nmero para a serializao
	 */
	private static final long serialVersionUID = 5745440461570172716L;

	/** linha Inicial da aplicao de lei */
	@XStreamAsAttribute
	private int linhaInicial;
	
	/** Linha Final da aplicao da lei */
	@XStreamAsAttribute
	private int linhaFinal;

	/** Parmetros da lei aplicada */
	private List<String> params;
	
	/** Nome da lei a ser aplicada */
	private String cod;
	
	/**
	 *  
	 */
	public LawApplication(String codigo, List<String> params, int lInicial, 
			int lFinal) {
		this.setParams(params);
		this.setCodDaLei(codigo);
		this.setarLinhas(lInicial, lFinal);
	}
	
	public List<String> getParams() {
		return params;
	}

	/**
	 * Parmetros da lei aplicada
	 * 
	 * @param params novos parmetros da lei
	 */
	private void setParams(List<String> params) {
		this.params = params;
	}
	
	private void setCodDaLei(String nomeDaLei) {
		this.cod = nomeDaLei;
	}
	
	public String getCodDaLei() {
		return this.cod;
	}
	
	/**
	 * Seta as linhas inicial e final de uma aplicao de lei
	 * 
	 * @param lInicial nova linha inicial
	 * @param lFinal nova linha final
	 */
	public void setarLinhas(int lInicial, int lFinal) {
		this.linhaInicial = lInicial;
		this.linhaFinal = lFinal;
	}
	
	/**
	 * Acessa as linhas nas quais foram aplicadas as leis de 
	 * refinamento.
	 * 
	 * @return o vetor com as linhas inicial e final, respectivamente.
	 */
	public int [] linhas() {
		int [] result = { this.linhaInicial , this.linhaFinal };
		return result;
	}

	public void doStep(ExternalManager externo) throws CRulesException {
		
		// No Gerenciador Interno preferencialmente
		List<CircusLaw> leis = externo.getLaws();

		/* procura pela lei a aplicar */
		for (CircusLaw lei : leis) {
			LawNumberAnn ann = lei.getAnn(LawNumberAnn.class);
			if (ann.getId().equals(this.getCodDaLei())) {
				externo.applyLaw(this.linhas()[0], this.linhas()[1], 
						lei, this.getParams());
				return;
			}
		}
	}

	public void undoStep(ExternalManager externo) {
		externo.desfazerInterno();
		externo.checkFinishedSubDevelopment(externo.retornarProgAtual());
		externo.desfazerIdentificadorLei();
		externo.retornarTelaDesenvolvimento().desfazerAplicLei();
		externo.desfazerTelaObri_RespostaLei();
		externo.desfazerPilhas();
	
	}
	
	public void undoPrintLatexStep(ExternalManager externo) {
		externo.desfazerInterno();
		externo.desfazerRespostasLeis();
		externo.desfazerPilhaDeLeis();
	
	}
	

	public void redoStep(ExternalManager externo) {
		externo.refazerInterno();
		externo.checkFinishedSubDevelopment(externo.retornarProgAtual());
		externo.reaplicarLei(); // dentro do reapplyscreen ele reinsere o idt de aplicacao
		externo.refazerPilhasLeis();
	}

	public void redoPrintLatexStep(ExternalManager externo) {
		externo.refazerInterno();
		externo.refazerRespostasLeis();
		externo.refazerPilhaDeLeis();
	}
	
}
