/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.specificationeditor;

import net.sourceforge.czt.base.ast.ListTerm;
import net.sourceforge.czt.session.SectionManager;
import net.sourceforge.czt.session.StringSource;
import net.sourceforge.czt.z.ast.Para;
import net.sourceforge.czt.z.ast.Sect;
import net.sourceforge.czt.z.ast.Spec;
import net.sourceforge.czt.z.ast.ZParaList;
import net.sourceforge.czt.z.ast.ZSect;
import circusRefine.core.InsertAnswer;
import circusRefine.core.crules.CRulesException;
import circusRefine.core.crules.parseArgument.ParseArgumentException;
import circusRefine.core.crules.parseArgument.ParseArgumentUtils;

/**
 * Insere um novo pargrafo global na ferramenta
 * 
 * @author Cristiano Castro
 */
public class SpecEditor {
	
	/** Instncia nica da classe */
	private static SpecEditor instance;
	
	/**
	 * Acessa a instncia da classe
	 * 
	 * @return a nica instncia de {@link SpecEditor}
	 */
	public static SpecEditor getInstance() {
		if ( SpecEditor.instance == null ) {
			SpecEditor.instance = new SpecEditor();
		}
		return SpecEditor.instance;
	}
	
	private SpecEditor() { }
	
	/**
	 * Insere um novo pargrafo  especificao.
	 * 
	 * @param userInput a {@link String} com a entrada do usurio.
	 * @param specification a AST onde o pargrafo ser inserido
	 * @param manager
	 * @throws CRulesException 
	 * @throws SpecificationEditorException 
	 * @throws ParseArgumentException caso o argumento digitado pelo usurio 
	 * 	no possa ser parseado
	 */
	public InsertAnswer insertNewPararagraph( String userInput , 
			Spec specification , SectionManager manager )
			throws CRulesException, SpecificationEditorException {
		
		StringSource src = new StringSource( userInput );
		Para paragraph = ParseArgumentUtils.parseGlobalParagraph( src , 
				manager );
		
		/* Insert the paragraph in the specification */
		this.insertIntoSpec( specification , paragraph );

		/* Retorna a resposta para o passo de execuo */
		return new InsertAnswer( paragraph );
	}
	
	/**
	 * Mtodo utilizado para inserir o pargrafo em uma especificao
	 * 
	 * @param specification a especificao
	 * @param paragraph o pargrafo a ser inserido
	 */
	private void insertIntoSpec( Spec specification , Para paragraph ) 
	throws SpecificationEditorException{
		
		ListTerm<Sect> lista = specification.getSect();
		ZSect zSect = null;
		
		/* Procura por uma seo Z */
		for ( Sect section : lista ) {
			if ( section instanceof ZSect ) {
				zSect = (ZSect) section;
			}
		}

		/* V se enconstrou a seo */
		if ( zSect != null ) {
			try {
				
				/* Insere o pargrafo */
				ZParaList listaPara = zSect.getZParaList();
				listaPara.add( paragraph );
			} catch ( UnsupportedClassVersionError e ) {
				
				throw new SpecificationEditorException( "Unable to parse the " +
					"global paragraph: the ZSect does not have a Z Paragraph " +
					"list" , e );
			}
		} else {
			
			/* No encontrou uma ZSect */
			throw new SpecificationEditorException( "Unable to parse the " +
					"global paragraph: there is no ZSect" );
		}
		
	}

	/**
	 * Mtodo utilizado para remover o pargrafo anteriormente inserido
	 *  em uma especificao
	 * @param specification a especificao
	 * @param paragraph o pargrafo a ser inserido
	 */
	public void removePararagraph(String userInput, Spec specification,
			SectionManager manager) throws CRulesException, SpecificationEditorException {
		StringSource src = new StringSource( userInput );
		Para paragraph = ParseArgumentUtils.parseGlobalParagraph( src , 
				manager );
		
		/* Insert the paragraph in the specification */
		this.removeIntoSpec( specification , paragraph );
	}

	/**
	 * Remove o pargrafo da especificao.
	 * 
	 * @param userInput a {@link String} com a entrada do usurio.
	 * @param specification a AST onde o pargrafo ser inserido
	 * @param manager
	 * @throws SpecificationEditorException 
	 * @throws CRulesException 
	 * @throws SpecificationEditorException 
	 * @throws ParseArgumentException caso o argumento digitado pelo usurio 
	 * 	no possa ser parseado
	 */
	private void removeIntoSpec(Spec specification, Para paragraph) throws SpecificationEditorException {
		ListTerm<Sect> lista = specification.getSect();
		ZSect zSect = null;
		
		/* Procura por uma seo Z */
		for ( Sect section : lista ) {
			if ( section instanceof ZSect ) {
				zSect = (ZSect) section;
			}
		}

		/* V se enconstrou a seo */
		if ( zSect != null ) {
			try {
				
				/* Insere o pargrafo */
				ZParaList listaPara = zSect.getZParaList();
				listaPara.remove( paragraph );
				
			} catch ( UnsupportedClassVersionError e ) {
				
				throw new SpecificationEditorException( "Unable to parse the " +
					"global paragraph: the ZSect does not have a Z Paragraph " +
					"list" , e );
			}
		} else {
			
			/* No encontrou uma ZSect */
			throw new SpecificationEditorException( "Unable to parse the " +
					"global paragraph: there is no ZSect" );
		}
	}
	
}
