/*
 * Projeto: Circus Refine
 */
package circusRefine.core.opsdischarge;

import java.util.Set;
import java.util.TreeSet;

import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.base.visitor.VisitorUtils;
import net.sourceforge.czt.circus.ast.CallAction;
import net.sourceforge.czt.circus.ast.CallProcess;
import net.sourceforge.czt.circus.ast.CircusAction;
import net.sourceforge.czt.circus.ast.CircusChannelSet;
import net.sourceforge.czt.circus.ast.CircusNameSet;
import net.sourceforge.czt.circus.ast.CircusProcess;
import net.sourceforge.czt.circus.visitor.CallActionVisitor;
import net.sourceforge.czt.circus.visitor.CallProcessVisitor;
import net.sourceforge.czt.circuspatt.ast.CircusPatternFactory;
import net.sourceforge.czt.circuspatt.impl.CircusPatternFactoryImpl;
import net.sourceforge.czt.z.ast.Decl;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.Spec;
import net.sourceforge.czt.z.ast.VarDecl;
import net.sourceforge.czt.z.ast.ZDeclList;
import net.sourceforge.czt.z.ast.ZSchText;
import net.sourceforge.czt.z.impl.SchExprImpl;
import net.sourceforge.czt.z.visitor.RefExprVisitor;
import circusRefine.core.InternalManager;
import circusRefine.core.astmodifiers.ASTModifiersUtils;
import circusRefine.core.astmodifiers.ActionArgumentAnn;
import circusRefine.core.astmodifiers.IASTModifierAnn;
import circusRefine.core.crules.utils.DefinitionFromATermGetter;
import circusRefine.core.crules.utils.TermFromANameGetter;
import circusRefine.core.util.ClonerVisitor;
import circusRefine.util.Pair;

/**
 * Classe utilizada para expandir uma obriga��o de prova. 
 * 
 * @author Cristiano Castro
 */
public class ExpandedProofObligationGenerator implements TermVisitor<Term>, 
RefExprVisitor<Term>, CallActionVisitor<Term>, CallProcessVisitor<Term> {

	/**
	 * M�todo utilit�rio para fazer uma expans�o completa de um OP
	 * 
	 * @param termo a OP a ser expandida
	 * @param astInicial a ast onde se busca as defini��es
	 * @param termoAplicacao o termo antes da aplica��o da lei
	 * @param termoResultante o termo resultante da aplica��o da lei
	 * @return o termo resultante da expans�o das leis
	 */
	public static Pair<Term, Set<String>> expandirOPs(Term termo, 
			Term astInicial, Term termoAplicacao, Term termoResultante, InternalManager gerInt) {
		Term anterior = termo;
		TreeSet<String> termosExpandidos = new TreeSet<String>();
		Pair<Term, TreeSet<String>> resultado;
		
		/* Executa v�rias expans�es */
		do {
			resultado = ExpandedProofObligationGenerator.expandirOPs(anterior, 
				astInicial, termoAplicacao, termoResultante, termosExpandidos, gerInt);

			/* Prepara para nova itera��o */
			anterior = resultado.getFirst();
			termoAplicacao = anterior;

			/* Adiciona os nomes expandidos � lista */
			termosExpandidos.addAll(resultado.getSecond());
		} while (!resultado.getSecond().isEmpty());

		/* Retorna a �ltima itera��o da expans�o */
		return new Pair<Term, Set<String>>(anterior, termosExpandidos);
	}

	/**
	 * M�todo utilit�rio respons�vel por efetuar uma itera��o na 
	 * expans�o das obriga��es de prova
	 * 
	 * @param termo a OP que eu quero expandir
	 * @param astInicial a �rvore utilizada para buscar as defini��es
	 * @param termoAplicacao o termo onde foi aplicada a lei
	 * @param termoResultante o termo resultante da aplica��o da lei
	 * @param termosExpandidos os nomes que j� foram alvos de expans�o
	 *  em itera��es anteriores
	 * @return a obriga��o de prova resultante da expans�o
	 */
	private static Pair<Term, TreeSet<String>> expandirOPs(Term termo, 
			Term astInicial, Term termoAplicado, Term termoResultante, 
			TreeSet<String> termosExpandidos, InternalManager gerInt) {
		ExpandedProofObligationGenerator visitor = 
			new ExpandedProofObligationGenerator(astInicial, termoAplicado, 
					termoResultante, termosExpandidos, gerInt);
		Term result = termo.accept(visitor);
		return new Pair<Term, TreeSet<String>>(result, 
				visitor.getNomesEspandidosCorrentes());
	}

	/** 
	 * Especifica��o de onde retira as defini��es de alguns nomes 
	 * do predicado
	 */
	private Term ast;

	/**
	 * O termo no qual a lei foi aplicada
	 */
	private Term termoAplicado;

	/** Termo resultante da aplica��o da lei */
	private Term termoResultante;

	/** Guarda os nomes expandidos at� agora */
	private TreeSet<String> nomesEspandidos;

	/** Guarda os nomes que foram espandidos na itera��o corrente */
	private TreeSet<String> nomesEspandidosCorrentes;
	
	/** Referencia para o Gerenciador Interno*/
	private InternalManager gerInterno;

	/**
	 * Inicia o objeto informando a AST onde as pesquisas dos termos
	 * dever�o ser feitas, bem como o termo onde foi aplicado a lei
	 * 
	 * @param ast a especifica��o Circus
	 * @param termoAplicado o termo no qual foi aplicado a lei
	 */
	public ExpandedProofObligationGenerator(Term ast, Term termoAplicado, 
			Term termoResultante, TreeSet<String> nomesEspandidos, InternalManager gerInt) {
		this.setAst(ast);
		this.setTermoAplicado(termoAplicado);
		this.setTermoResultante(termoResultante);
		this.setNomesEspandidos(nomesEspandidos);
		this.setNomesEspandidosCorrentes(new TreeSet<String>());
		gerInterno = gerInt;
	}

	/**
	 * @return the ast
	 */
	private Term getAst() {
		return ast;
	}

	/**
	 * @param ast the ast to set
	 */
	private void setAst(Term ast) {
		this.ast = ast;
	}

	/**
	 * @return the termoAplicado
	 */
	private Term getTermoAplicado() {
		return termoAplicado;
	}

	/**
	 * @param termoAplicado the termoAplicado to set
	 */
	private void setTermoAplicado(Term termoAplicado) {
		this.termoAplicado = termoAplicado;
	}

	/**
	 * @return the termoResultante
	 */
	private Term getTermoResultante() {
		return termoResultante;
	}

	/**
	 * @param termoResultante the termoResultante to set
	 */
	private void setTermoResultante(Term termoResultante) {
		this.termoResultante = termoResultante;
	}

	/**
	 * @return the nomesEspandidos
	 */
	private TreeSet<String> getNomesEspandidos() {
		return nomesEspandidos;
	}

	/**
	 * @param nomesEspandidos the nomesEspandidos to set
	 */
	private void setNomesEspandidos(TreeSet<String> nomesEspandidos) {
		this.nomesEspandidos = nomesEspandidos;
	}

	/**
	 * @return the nomesEspandidosCorrentes
	 */
	private TreeSet<String> getNomesEspandidosCorrentes() {
		return nomesEspandidosCorrentes;
	}

	/**
	 * @param nomesEspandidosCorrentes the nomesEspandidosCorrentes to set
	 */
	private void setNomesEspandidosCorrentes(
			TreeSet<String> nomesEspandidosCorrentes) {
		this.nomesEspandidosCorrentes = nomesEspandidosCorrentes;
	}

	/**
	 * Visita um termo gen�rico. N�o executa nenhuma tarefa de 
	 * expans�o.
	 * 
	 * @param arg0 o predicado a ser visitado
	 * @return o mesmo predicado
	 */
	public Term visitTerm(Term arg0) {

		return VisitorUtils.visitTerm(this, arg0, false);
	}

	/**
	 * Visita uma refer�ncia a uma express�o que pode ser substitu�da por 
	 * uma defini��o
	 * 
	 * @param arg0 
	 * @returnS
	 */
	public Term visitRefExpr(RefExpr arg0) {
		Term result;
		IASTModifierAnn modifierAnn = arg0.getAnn(IASTModifierAnn.class);

		/* Testa se o RefExpr n�o contem uma refer�ncia a uma a��o */
		if (modifierAnn == null) {

			/* Testa se o nome j� foi expandido ou n�o */
			if (!this.getNomesEspandidos().contains(arg0.getZName().getWord())) 
			{

				/* Ocorr�ncia do termo no lugar onde foi aplicada a lei */
				Term ocorrencia = TermFromANameGetter.getTerm(arg0.getZName(), 
						this.getTermoAplicado());

				/* 
				 * Caso n�o tenha achado a ocorr�ncia antes da aplica��o da 
				 * lei, testa depois da aplica��o
				 */
				if (ocorrencia == null) {
					ocorrencia = TermFromANameGetter.getTerm(arg0.getZName(), 
							this.getTermoResultante());
				}

				/* Procura a defini��o do termo na especifica��o */
				Term definicaoTerm = null;
				if (ocorrencia != null) {
					definicaoTerm = 
						DefinitionFromATermGetter.find(arg0.getZName(), 
								ocorrencia, this.getAst(), gerInterno);
				}

				/* Testa se achou uma defini��o decente */
				if (definicaoTerm != null) {

					/* Retira a express�o correspondente ao termo */
					result = this.getExpression(definicaoTerm);

					/* Insere a express�o na lista de termos expandidos */
					String expandida = arg0.getZName().getWord();
					this.getNomesEspandidosCorrentes().add(expandida);
				} else {

					/* Defini��o n�o foi encontrada */
					result = ClonerVisitor.cloneTerm(arg0);
				}
			} else {

				/* Termo j� foi expandido */
				result = ClonerVisitor.cloneTerm(arg0);
			}
		} else {

			/* 
			 * Visita o termo encapsulado e cria uma nova anota��o 
			 * contendo o termo visitado 
			 */
			Term encapsulado = modifierAnn.getTerm();
			encapsulado = encapsulado.accept(this);
			IASTModifierAnn novaAnn = modifierAnn.create();
			novaAnn.setTerm(encapsulado);

			/* C�pia do argumento para visita */
			result = ClonerVisitor.cloneTerm(arg0);

			/* Modifica a anota��o */
			ASTModifiersUtils.removeAnn(result, IASTModifierAnn.class);
			result.getAnns().add(novaAnn);
		}

		return result;
	}

	/**
	 * Visita uma refer�ncia a uma a��o que pode ser expandida
	 * 
	 * @param arg0 a {@link CallAction} a ser visitada
	 * @return o pr�prio argumento se o termo n�o puder ser expandido,
	 *  ou a expans�o do termo caso contr�rio
	 */
	public Term visitCallAction(CallAction arg0) {
		CircusAction result;

		/* Testa se o termo j� foi expandido */
		if (!this.getNomesEspandidos().contains(arg0.getZName().getWord())) {

			/* Ocorr�ncia do termo no lugar onde foi aplicada a lei */
			Term ocorrencia = TermFromANameGetter.getTerm(arg0.getZName(), 
					this.getTermoAplicado());

			/* 
			 * Caso n�o tenha achado a ocorr�ncia antes da aplica��o da 
			 * lei, testa depois da aplica��o
			 */
			if (ocorrencia == null) {
				ocorrencia = TermFromANameGetter.getTerm(arg0.getZName(), 
						this.getTermoResultante());
			}

			/* Procura a defini��o do termo na especifica��o */
			Term definicaoTerm = null;
			if (ocorrencia != null) {
				definicaoTerm = 
					DefinitionFromATermGetter.find(arg0.getZName(), 
							ocorrencia, this.getAst(), gerInterno);
			}

			/* Testa se achou uma defini��o decente */
			if (definicaoTerm != null && definicaoTerm instanceof CircusAction) {

				/* Retira a express�o correspondente ao termo */
				result = (CircusAction) definicaoTerm;

				/* Insere a express�o na lista de termos expandidos */
				String expandida = arg0.getZName().getWord();
				this.getNomesEspandidosCorrentes().add(expandida);
			} else {

				/* Defini��o n�o foi encontrada */
				result = ClonerVisitor.cloneTerm(arg0);
			}
		} else {

			/* Nome j� foi expandido */
			result = ClonerVisitor.cloneTermRemovingRelationsStack(arg0);
		}

		return result;
	}
	
	/**
	 * Visita uma refer�ncia a um processo que pode ser expandido
	 * 
	 * @param arg0 a refer�ncia ao processo que pode ser 
	 *  posteriormente expandido
	 * @return o processo expandido caso este seja pass�vel de 
	 *  expans�o
	 */
	public Term visitCallProcess(CallProcess arg0) {
		CircusProcess result;
		
		// TODO tratar processos parametrizados ou indexados
		
		RefExpr nomeProcesso = arg0.getCallExpr();
		String strProcesso = nomeProcesso.getZName().getWord();

		/* Testa se o termo j� foi expandido */
		if (!this.getNomesEspandidos().contains(strProcesso)) {

			/* Procura a defini��o do termo na especifica��o */
			Term definicaoTerm = 
				DefinitionFromATermGetter.find(nomeProcesso.getZName(), arg0, 
						this.getAst(), gerInterno);

			/* Testa se achou uma defini��o decente */
			if (definicaoTerm != null && definicaoTerm instanceof CircusProcess)
			{

				/* Retira a express�o correspondente ao termo */
				result = (CircusProcess) definicaoTerm;

				/* Insere a express�o na lista de termos expandidos */
				String expandida = nomeProcesso.getZName().getWord();
				this.getNomesEspandidosCorrentes().add(expandida);
			} else {

				/* Defini��o n�o foi encontrada */
				result = ClonerVisitor.cloneTerm(arg0);
			}
		} else {

			/* Nome j� foi expandido */
			result = ClonerVisitor.cloneTermRemovingRelationsStack(arg0);
		}

		return result;
	}

	/**
	 * Filtra a express�o da defini��o de algo, como um conjunto de
	 * canais ou um conjunto de nomes.
	 * 
	 * @param definicao a defini��o a ser "filtrada"
	 * @return a express�o que especifica o termo
	 */
	private Term getExpression(Term definicao) {

		//TODO tratar o caso da a��o
		Term result;
		if (definicao instanceof CircusChannelSet) {
			result = ((CircusChannelSet) definicao).getExpr();
		} else if (definicao instanceof CircusNameSet) {
			result = ((CircusNameSet) definicao).getExpr();
		} else if (definicao instanceof CircusAction) {

			/* 
			 * Retorna um RefExpr com uma anota��o de a��o como
			 * argumento 
			 */
			CircusPatternFactory factory = new CircusPatternFactoryImpl();
			result = factory.createRefExpr();
			ActionArgumentAnn ann = 
				new ActionArgumentAnn((CircusAction) definicao);
			result.getAnns().add(ann);
		} else {
			result = definicao;
		}
		return result;
	}

}
