/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.crules.utils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.util.UnsupportedAstClassException;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.base.visitor.VisitorUtils;
import net.sourceforge.czt.circus.ast.CircusFactory;
import net.sourceforge.czt.circus.impl.CircusFactoryImpl;
import net.sourceforge.czt.circus.util.Factory;
import net.sourceforge.czt.circuspatt.ast.CircusPatternFactory;
import net.sourceforge.czt.circuspatt.impl.CircusPatternFactoryImpl;
import net.sourceforge.czt.rules.UnboundJokerException;
import net.sourceforge.czt.rules.rewriter.RewriteUtils;
import net.sourceforge.czt.session.CommandException;
import net.sourceforge.czt.session.Markup;
import net.sourceforge.czt.z.ast.And;
import net.sourceforge.czt.z.ast.ApplExpr;
import net.sourceforge.czt.z.ast.Decl;
import net.sourceforge.czt.z.ast.DeclList;
import net.sourceforge.czt.z.ast.DecorExpr;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.Fact;
import net.sourceforge.czt.z.ast.InclDecl;
import net.sourceforge.czt.z.ast.Pred;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.SchExpr;
import net.sourceforge.czt.z.ast.SchText;
import net.sourceforge.czt.z.ast.ZDeclList;
import net.sourceforge.czt.z.ast.ZExprList;
import net.sourceforge.czt.z.ast.ZName;
import net.sourceforge.czt.z.ast.ZSchText;
import net.sourceforge.czt.z.util.ZString;
import net.sourceforge.czt.z.visitor.ApplExprVisitor;
import net.sourceforge.czt.z.visitor.SchExprVisitor;
import net.sourceforge.czt.z.visitor.ZSchTextVisitor;
import circusRefine.core.InternalManager;
import circusRefine.core.crules.UpdateVisitor;
import circusRefine.core.crules.anotations.CopyRuleForNameAnn;
import circusRefine.core.finder.SchemaFinder;
import circusRefine.core.opsdischarge.OPsDischargeUtils;
import circusRefine.core.opsdischarge.syntacticfunctions.CannotEvaluateException;
import circusRefine.core.print.Printer;
import circusRefine.core.util.ChildrenTermExtractor;
import circusRefine.core.util.ClonerVisitor;
import circusRefine.util.Pair;

/**
 * Classe com m�todos utilit�rios para a aplica��o da normaliza��o.
 * 
 * @author Cristiano Castro
 */
public class NormalizeApplier {

	/** �nica Inst�ncia da classe normalize */
	private static NormalizeApplier instance;
	
	/**
	 * M�todo est�tico para acessar a inst�ncia da classe
	 * 
	 * @return a inst�ncia da classe de normaliza��o
	 */
	public static NormalizeApplier getInstance() {
		if (NormalizeApplier.instance == null) {
			NormalizeApplier.instance = new NormalizeApplier();
		}
		return NormalizeApplier.instance;
	}
	
	/** Construtor privado */
	private NormalizeApplier() { }
	
	/**
	 * Monta a fun��o "normalize" e depois realiza a reescrita com o
	 * "expansion_rules" para aplicar a normaliza��o
	 * 
	 * @param esquema o esquema a ser normalizado
	 * @return o esquema normalizadp
	 * @throws UnboundJokerException 
	 * @throws CommandException
	 * @throws CannotEvaluateException caso ocorra algum erro na 
	 *  aplica��o da normaliza��o
	 */
	public SchExpr applyRewrite(Expr esquema) throws NormalizationApplicationException {
		
		try {
			
			/* Modifica o termo */
			esquema = this.wrapForNormalize(esquema);
			ApplExpr aSubstituir = RewriteUtils.createNormalizeAppl(esquema);
			Term result = RewriteUtils.rewrite(aSubstituir, 
					InternalManager.getManager(), "expansion_rules");
			SchExpr normalized = this.retirarAplicacaoFuncao(result);
			return this.unwrapAfterNormalize( normalized );
		} catch (CommandException e) {
			throw new NormalizationApplicationException(e);
			//System.out.println("command");
		} catch (UnboundJokerException e) {
			throw new NormalizationApplicationException(e);
		}
			//System.out.println("Unbound");
		//} catch (RuntimeException e) {
			//throw new NormalizationApplicationException(e);
		//	System.out.println("runtime");
		//}
	}
	
	/**
	 * M�todo para filtrar o resultado da normaliza��o para o 
	 * desejado. Utilizado para prevenir contra o bug da 
	 * normaliza��o de continuar a apresentar a aplica��o de fun��o 
	 * mesmo ap�s a normaliza��o. 
	 * 
	 * @param expr o resultado vindo da fun��o de aplica��o da 
	 *  normaliza��o.
	 * @return o esquema resultante da opera��o de normaliza��o 
	 *  retirando-se a aplica��o da fun��o normalize caso necess�rio.
	 * @throws ClassCastException caso o resultado da normaliza��o n�o
	 *  seja uma inst�ncia de {@link SchExpr}.
	 */
	private SchExpr retirarAplicacaoFuncao(Term expr) {
		SchExpr result;
		if (expr instanceof ApplExpr) {
			ApplExpr ap = (ApplExpr) expr;
			if (ap.getRightExpr() instanceof DecorExpr){
				DecorExpr dec = (DecorExpr) ap.getRightExpr();
				Expr ex = (Expr) dec.getExpr();
				//result = (SchExpr) ex.getAnns().get(0); 
				CircusPatternFactory f = new CircusPatternFactoryImpl();
				SchExpr sch= f.createSchExpr();
				ZDeclList decl = f.createZDeclList();
				Pred pred = f.createTruePred();
				SchText schT = f.createZSchText(decl, pred);
				sch.setSchText(schT);
				result = sch;
			}
			else
			result = (SchExpr) ((ApplExpr) expr).getRightExpr();
		} else {
			result = (SchExpr) expr;
		}
		return result;
	}
	
	/**
	 * Executa modifica��es no programa que tornam a normaliza��o 
	 * poss�vel
	 * 
	 * @param esquema o programa alvo da normaliza��o
	 * @return o programa preparado para a normaliza��o
	 */
	private Expr wrapForNormalize(Expr esquema) {
		Wrapper visitor = new Wrapper();
		return (Expr) esquema.accept(visitor);
	}
	
	/**
	 * 
	 * @param expr
	 * @return uma express�o 
	 */
	private SchExpr unwrapAfterNormalize( SchExpr expr ) {
		Unwrapper visitor = new Unwrapper();
		expr.accept( visitor );
		return expr;
	}
	
	// TODO tratar tamb�m o \Xi antes da normaliza��o
	
	/**
	 * Avalia a fun��o Delta de um esquema
	 * 
	 * @param esquema o esquema no qual ocorrer� a aplica��o de Delta
	 * @return a lista com os esquemas resultantes da aplica��o de delta
	 */
	private ZExprList applyDelta(Expr schema) {
		CircusFactory factory = new CircusFactoryImpl();
		
		/* Cria a express�o marcada */
		Expr schemaCopy = 
			ClonerVisitor.cloneTermRemovingRelationsStack(schema);
		DecorExpr dashed = factory.createDecorExpr(schemaCopy, 
				factory.createNextStroke());
		
		/* Monta a lista a ser retornada */
		ZExprList result = factory.createZExprList();
		result.add(schema);
		result.add(dashed);
		
		return result;
	}
	
	/**
	 * Classe para preparar o termo para a normaliza��o.
	 * 
	 * @author Cristiano Castro
	 */
	private class Wrapper implements SchExprVisitor<Term>, TermVisitor<Term>,
		ApplExprVisitor<Term> {
		
		/**
		 * Visita um esquema que podem conter elementos na sua lista
		 * de declara��es a serem substitu�dos
		 * 
		 * @param arg0 a express�o a ser visitada
		 * @return o termo resultante da substitui��o de poss�veis 
		 * aplica��es de Delta na lista de declara��es ou outras 
		 * substitui��es quaisquer
		 */
		public Term visitSchExpr(SchExpr arg0) {
			CircusFactory factory = new CircusFactoryImpl();
			arg0 = ClonerVisitor.cloneTermRemovingRelationsStack(arg0);
			
			try {
				ZSchText text = arg0.getZSchText();
				
				/* 
				 * Termos a serem inclu�dos e exclu�dos da lista de 
				 * declara��es 
				 */
				ZDeclList toInclude = factory.createZDeclList();
				ZDeclList toExclude = factory.createZDeclList();
				
				/* Percorre as declara��es da express�o */
				for (Decl decl : text.getZDeclList()) {
					if (decl instanceof InclDecl) {
						InclDecl incl = (InclDecl) decl;
						Expr inclExpr = incl.getExpr();
						
						if (inclExpr instanceof ApplExpr) {
						
							ApplExpr appl = (ApplExpr) inclExpr;
							
							/* Visita os argumentos */
							appl.setRightExpr((Expr) 
									appl.getRightExpr().accept(this));
							
							RefExpr deltaFun = 
								OPsDischargeUtils.refFuncao(ZString.DELTA);
							if (appl.getLeftExpr().equals(deltaFun)) {
								
								/* Aplica��o de delta */
								
								/* A declara��o anterior � removida */
								toExclude.add(decl);
								
								ZExprList deltaAppl = 
									applyDelta(appl.getRightExpr());
								
								/* As novas declara��es s�o adicionadas */
								for (Expr expr : deltaAppl) {
									InclDecl newDecl = 
										factory.createInclDecl(expr);
									toInclude.add(newDecl);
								}
							}
						}
					} 
					
				}
				
				/* Adiciona as listas de inclus�o e exclus�o a ZDeclList */
				text.getZDeclList().removeAll(toExclude);
				text.getZDeclList().addAll(toInclude);
				
			} catch (UnsupportedAstClassException e) {
				// N�o faz nada
			}
			
			return arg0;
		}

		/**
		 * Visita uma aplicação de função que pode ser uma aplicação 
		 * de delta
		 * 
		 * @param arg0 a aplicação de função a ser aplicada
		 * 
		 */
		public Term visitApplExpr(ApplExpr arg0) {
			Expr funName = arg0.getLeftExpr();
			Term result = arg0;
			CircusFactory factory = new CircusFactoryImpl();
			if (funName instanceof RefExpr) {
				RefExpr funRef = (RefExpr) funName;
				if (funRef.getZName().getWord().equals(ZString.DELTA)) {
					
					/* É uma aplicação de Delta */
					InclDecl decl = factory.createInclDecl(arg0);
					List<Decl> list = Collections.singletonList((Decl) decl);
					DeclList declList = factory.createZDeclList(list);
					SchText text = factory.createZSchText(declList, 
							factory.createTruePred());
					SchExpr expr = factory.createSchExpr(text);
					result = expr.accept(this);
				}
			}
			
			return result;
		}
		
		public Term visitTerm(Term arg0) {
			return VisitorUtils.visitTerm(this, arg0, false);
		}
		
	}
	
	/**
	 * 
	 * @author crisgc
	 *
	 */
	private class Unwrapper implements SchExprVisitor<Void> {
		
		public Void visitSchExpr(SchExpr arg0) {
			CircusPatternFactory factory = new CircusPatternFactoryImpl();
			ZSchText text = arg0.getZSchText();
			
			if (text.getZDeclList()!=null){
			ZDeclList declListMajorSchema = text.getZDeclList();
			
			ZDeclList declToExclude = factory.createZDeclList();
			ZDeclList declToInclude = factory.createZDeclList();
			List<Pred> predsToInclude = new ArrayList<Pred>();
			
			for ( Decl decl : declListMajorSchema ) {
				
				// TODO fazer um merge nas listas de declara��o, por enquanto est� fazendo apenas inclus�o
				/* 
				 * Se for uma inst�ncia de declara��o de inclus�o ent�o 
				 * tem que fazer um merge nas lista de declara��o 
				 */
				if ( decl instanceof InclDecl ) {
					
					InclDecl inclDecl = ( InclDecl ) decl;
					Expr inclExpr = inclDecl.getExpr();
					
					/* Se for um SchExpr faz o merge */
					if ( inclExpr instanceof SchExpr ) {
						
						SchExpr inclSchExpr = ( SchExpr ) inclExpr;
						ZSchText inclSchText = inclSchExpr.getZSchText();
						
						/* Inclui as declara��es na lista principal */
						declToInclude.addAll( inclSchText.getZDeclList() );
						
						/* 
						 * O predicado vai ser inclu�do ao predicado do 
						 * esquema principal 
						 */
						predsToInclude.add( inclSchText.getPred() );
						
						/*
						 * A declara��o inclu�da vai ser retirarda do 
						 * predicado original
						 */
						declToExclude.add( inclDecl );
					}
					
				}
				
			}
		
			
			/* Modifica o esquema original */
			declListMajorSchema.removeAll( declToExclude );
			declListMajorSchema.addAll( declToInclude );
			for ( Pred pred : predsToInclude ) {
				
				/* Faz o And dos predicados */
				Pred toInsert = factory.createAndPred( Arrays.asList( 
						text.getPred() , pred ) , And.Chain );
				text.setPred( toInsert );
			}
			}
			
			return null;
		} 
		
	}
	
	/**
	 * Substitui os nomes pela suas defini��es em uma �rvore que pode 
	 * conter esquemas.
	 * 
	 * @param expr a express�o com os esquemas cujos nomes devem se
	 *  substitu�dos.
	 * @return a nova express�o
	 */
	public Expr substituteNames(Expr expr, Term ast, InternalManager gerInterno) {
		List<ZSchText> schemas = this.getSchemas(expr);
		Expr result = expr;
		if (schemas != null)
		for (ZSchText schema : schemas) {
			result = (Expr)UpdateVisitor.update(schema, 
					this.substituteNames(schema, ast, gerInterno), result,false);
		}
		return result;
	}
	
	/**
	 * Faz uma busca exaustiva para substituir os nomes de um esquema.
	 * 
	 * @param schema o esquema a ter os nomes substituidos
	 * @return o novo esquema
	 */
	private SchText substituteNames(SchText schema, Term ast, InternalManager gerInterno) {
		SchText oldSchema = null;
		
		/* Itera��o teste para atender as defini��es recursivas */
		while (!schema.equals(oldSchema)) {
			oldSchema = schema;
			
			if (schema instanceof ZSchText) {
				ZSchText zSchema = (ZSchText) schema;

				List<Pair<ZName, Term>> referencias = 
					NamesFinder.findAllName(schema);
				for (Pair<ZName, Term> par : referencias) {
					zSchema = (ZSchText)
					CopyRuleForNameAnn.changeNameToDefinition(par.getFirst(), 
							par.getSecond(), ast, zSchema, gerInterno);

				}

				/* Para retornar */
				schema = zSchema;

			} 
		}
		
		return schema;
	}
	
	private List<ZSchText> getSchemas(Expr expr) {
		SchemasGetter getter = new SchemasGetter();
		return expr.accept(getter);
	}
	
	/**
	 * Classe para retornar os schemas de um {@link Term}.
	 * 
	 * @author Cristiano Castro
	 */
	private class SchemasGetter implements TermVisitor<List<ZSchText>>, 
		ZSchTextVisitor<List<ZSchText>> {

		public List<ZSchText> visitTerm(Term arg0) {
			List<Term> children = ChildrenTermExtractor.extrairFilhos(arg0);
			List<ZSchText> result = new LinkedList<ZSchText>();
			
			/* Busca exaustiva */
			for (Term child : children) {
				result.addAll(child.accept(this));
			}
			
			return result;
		}

		public List<ZSchText> visitZSchText(ZSchText arg0) {
			return Arrays.asList(arg0);
		}
		
	}

	
}
