/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.astmodifiers;

import circusRefine.core.util.ClonerVisitor;
import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.circus.ast.CircusProcess;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.ZFactory;
import net.sourceforge.czt.z.impl.ZFactoryImpl;

/**
 * <p>Classe utilizada para inserir um processo como um argumento de 
 * funo. Ela deve ser uma anotao em uma expresso (RefExpr) que 
 * representem um argumento para uma funo</p>
 * 
 * <p>Uma aplicao de uma funo
 * <br><br>f: (EXPR x PROCESS x EXPR) -> BOOLEAN<br><br>, por exemplo,
 * ter uma lista de expresses de tamanho 3 e o segundo argumento
 * ser um RefExpr com uma {@link ActionArgumentAnn}.
 * </p>
 * 
 * @author Cristiano Castro
 */
public class ProcessArgumentAnn implements IASTModifierAnn {
	
	/**
	 * Monta uma {@link RefExpr} que possui um processo em suas 
	 * anotaes. Utilizado para facilitar a utilizao de um processo
	 * como argumento de funo
	 * 
	 * @param process o processo a ser includo nas anotaes de um 
	 *  {@link RefExpr}
	 * @return o novo {@link RefExpr} que possui um processo em suas 
	 *  anotaes
	 */
	public static RefExpr montarReferenciaProcesso(CircusProcess process) {
		ZFactory factory = new ZFactoryImpl();
		RefExpr result = factory.createRefExpr();
		result.getAnns().add(new ProcessArgumentAnn(process));
		return result;
	}

	/** Processo encapsulado por essa anotao */
	private CircusProcess processo;
	
	/**
	 * Inicia a anotao informando qual o processo encapsulado pela
	 * mesma.
	 * 
	 * @param processo o processo encapsulado pela 
	 */
	public ProcessArgumentAnn(CircusProcess processo) {
		this.setTerm(processo);
	}
	
	/**
	 * Retorna o processo encapsulado na anotao de processos como 
	 * argumento de funo
	 * 
	 * @return o processo utilizadado como argumento de funo
	 */
	public CircusProcess getTerm() {
		return this.processo;
	}

	/**
	 * Modifica o processo encapsulado
	 * 
	 * @param newTerm o novo processo encapsulado por este termo
	 * @throws ClassCastException caso o argumento no seja uma 
	 *  instncia de {@link CircusProcess}
	 */
	public void setTerm(Term newTerm) {
		this.processo = (CircusProcess) newTerm;
	}
	
	/**
	 * Clona a anotao de processo como argumento de funo
	 * 
	 * @return uma nova anotao que  uma cpia da anterior
	 */
	public ProcessArgumentAnn clone() throws CloneNotSupportedException {
		CircusProcess clone = 
			ClonerVisitor.cloneTermRemovingRelationsStack(this.getTerm());
		return new ProcessArgumentAnn(clone);
	}
	
	/**
	 * Cria uma anotao encapsulando <code>null</code>
	 * 
	 * @return uma {@link ProcessArgumentAnn} com <code>null</code>
	 *  encapsulado
	 */
	public ProcessArgumentAnn create() {
		return new ProcessArgumentAnn(null);
	}

}
