/*
 * Cristiano Gurgel
 * 
 * Projeto: Circus Refine
 */
package circusRefine.storage;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.core.BaseException;

/**
 * Classe que armazena o Status do Circus Refine. Utilizada no 
 * salvamento/recuperao de um programa
 * 
 * @author Cristiano Gurgel
 * @version 1.1 06/01/07
 */
public class StatusCircusRefine {
	private static final int BUFFER = 2048;

	/** Status da tela de desenvolvimento */
	private StatusDesenvolvimento desenvolvimento;

	/** Status da tela principal */
	private StatusPrincipal principal;
	
	/** Status do gerenciador externo */
	private StatusExterno externo;
	
	/** Status do Gerenciador Interno*/
	private StatusInterno interno;
	
	/**
	 * Inicializa o status do circus Refine com todos os valores
	 * em <code>null</code> 
	 */
	public StatusCircusRefine() {
		this(null, null, null, null);
	}

	/**
	 * Inicializa o status do circusRefine com todos os valores
	 * definidos
	 * 
	 * @param novoDesenvolvimento Status da tela de desenvolvimento
	 * @param novoExterno Status do Gerenciador Externo
	 * @param novoInterno Status do Gerenciador Interno
	 * @param novoPrincipal Status da Tela Principal
	 */
	public StatusCircusRefine(StatusDesenvolvimento novoDesenvolvimento,
			StatusExterno novoExterno, StatusInterno novoInterno,
			StatusPrincipal novoPrincipal) {
		this.setDesenvolvimento(novoDesenvolvimento);
		this.setExterno(novoExterno);
		this.setInterno(novoInterno);
		this.setPrincipal(novoPrincipal);
	}
	
	/**
	 * Acessa o status do desenvolvimento
	 * 
	 * @return o status da tela de desenvolvimento
	 */
	public StatusDesenvolvimento getDesenvolvimento() {
		return desenvolvimento;
	}
	
	/**
	 * Seta o atributo do status da tela de desenvolvimento
	 * 
	 * @param desenvolvimento o novo status da tela de desenvolvimento
	 */
	public void setDesenvolvimento(StatusDesenvolvimento desenvolvimento) {
		this.desenvolvimento = desenvolvimento;
	}
	
	/**
	 * Acessa o status da tela principal
	 * 
	 * @return o status da tela principal
	 */
	public StatusPrincipal getPrincipal() {
		return principal;
	}
	
	/**
	 * Seta o atributo do status da tela principal
	 * 
	 * @param principal o novo status da tela principal
	 */
	public void setPrincipal(StatusPrincipal principal) {
		this.principal = principal;
	}
	
	/**
	 * Acessa o status do gerenciador externo
	 * 
	 * @return o status do gerenciador externo
	 */
	public StatusExterno getExterno() {
		return externo;
	}
	
	/**
	 * Seta o atributo do status do gerenciador externo
	 * 
	 * @param externo o novo status do gerenciador externo
	 */
	public void setExterno(StatusExterno externo) {
		this.externo = externo;
	}

	/**
	 * Acessa o status do gerenciador interno
	 * 
	 * @return o status do gerenciador interno
	 */
	public StatusInterno getInterno() {
		return interno;
	}
	
	/**
	 * Seta o status do gerenciador interno
	 * 
	 * @param interno novo status do gerenciador interno
	 */
	public void setInterno(StatusInterno interno) {
		this.interno = interno;
	}
	
	/**
	 * Salva o conteudo deste objeto em um arquivo passado como
	 * parametro
	 * 
	 * @param arquivo arquivo onde o status ser salvo
	 * @throws IOException caso um erro de IO ocorra durante o
	 * 			processo de salvamento
	 */
	public void salvar(File arquivo) throws IOException {
		
		/* OutpuStreams */
		FileOutputStream out = new FileOutputStream(arquivo);
		ZipOutputStream zos = new ZipOutputStream(out);
		
		/* Metodo de compresso */
		zos.setMethod(ZipOutputStream.DEFLATED);
		
		/* Gera o xml */
		XStream xml = new XStream();
		byte[] dados = xml.toXML(this).getBytes();

		/* Cria o arquivo .zip */
		ZipEntry entrada = new ZipEntry("prog.xml");
		zos.putNextEntry(entrada);
		zos.write(dados);
		zos.close();
	}
	
	/**
	 * Abre o contedo de um arquivo no formato aceito pela
	 * ferramenta.
	 * 
	 * @param arquivo o arquivo a ser aberto
	 * @return o status do circusRefine representado pelo arquivo
	 * @throws IOException caso um erro de I/O ocorra na abertura do
	 * 			arquivo
	 * @throws BaseException caso o arquivo no possa ser
	 * 			deserializado
	 */
	public static StatusCircusRefine abrir(File arquivo) 
			throws IOException, BaseException {
		
		/* Cria os input Streams */
		FileInputStream fin = new FileInputStream(arquivo);
		ZipInputStream zin = new ZipInputStream(new BufferedInputStream(fin));
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		
		/* Extrai os dados */
		ZipEntry entrada = zin.getNextEntry();
		if (entrada != null) {
			byte[] dados = new byte[StatusCircusRefine.BUFFER];
			int cont;
			while ((cont = zin.read(dados, 0 , StatusCircusRefine.BUFFER)) 
					!= -1) {
				out.write(dados, 0, cont);
			}
		}
		zin.close();
		
		/* Gera o objeto */
		XStream xml = new XStream();
		return (StatusCircusRefine)xml.fromXML(out.toString());
	}
	
}
