/*
 * Projeto: Circus Refine
 */
package circusRefine.core.storage;

import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamAsAttribute;

import circusRefine.core.CollectAnswer;
import circusRefine.core.ExternalManager;
import circusRefine.core.TipoColeta;

/**
 * Classe que representa a ao de coleta. Seja de ao ou processo.
 * 
 * @author Cristiano Gurgel
 */
@XStreamAlias("Collect")
public class Coleta implements StepOfExecution {

	/**
	 * N�mero da serializao
	 */
	private static final long serialVersionUID = -3138041485733735141L;

	/** Tipo da coleta que se efetua */
	@XStreamAsAttribute
	private TipoColeta tipo;
	
	/** Nome do elemento que se quer coletar */
	private String nome;
	
	/**
	 * Inicia a Coleta com o tipo da coleta e nome do par�grafo a ser
	 * coletado
	 * 
	 * @param tipo tipo da coleta
	 * @param nome nome do pargrafo a ser coletado
	 */
	public Coleta(TipoColeta tipo, String nome) {
		this.setTipo(tipo);
		this.setNome(nome);
	}

	public TipoColeta getTipo() {
		return tipo;
	}
	
	private void setTipo(TipoColeta tipo) {
		this.tipo = tipo;
	}
	
	public String getNome() {
		return nome;
	}
	
	private void setNome(String nome) {
		this.nome = nome;
	}

	public void doStep(ExternalManager externo) {
		
		/* Escolha pelo tipo de coleta */
		if (this.getTipo().equals(TipoColeta.ACAO)) {
			CollectAnswer resposta = externo.coletarCodigoAcao(this.getNome());
			if (resposta != null){
				externo.adicionarColetaAoHistorico(resposta);
			}
			
		} else if (this.getTipo().equals(TipoColeta.PROCESSO)) {
			CollectAnswer resposta = externo.coletarCodigoProcesso(this.getNome());
			if (resposta != null){
				externo.adicionarColetaAoHistorico(resposta);
			}
		}
		else {
			externo.coletarTermoInicial();
			externo.adicionarColetaInitialTermHist();
		}
	}

	public void undoStep(ExternalManager externo) {
		
		if (!this.getTipo().equals(TipoColeta.TERMO_INICIAL)){
			externo.desfazerUltimasRelacoes_Idts();
			externo.DesfazendoPilhaTermosColetados();
		}
		else {
			externo.desfazerRalacoesIdtsTermoInicial();
		}
		
		externo.retornarTelaDesenvolvimento().retirarUltimasLinhas();
		
	}

	public void redoStep(ExternalManager externo) {
		if (!this.getTipo().equals(TipoColeta.TERMO_INICIAL)){
			externo.refazerColeta();
		}
		else {
			externo.coletarTermoInicial();
		}
		
	}

	public void redoPrintLatexStep(ExternalManager externo) {
		if (this.getTipo() != TipoColeta.TERMO_INICIAL){
			externo.refazerAcoesColetadas();
		}
		
	}

	public void undoPrintLatexStep(ExternalManager externo) {
		if (this.getTipo() != TipoColeta.TERMO_INICIAL) {
			externo.desfazerAcoesColetadas();
		}
		
	}
	
}
