/*
 * Projeto: Circus Refine
 */
package circusRefine.core.relations;

import java.util.EmptyStackException;

import circusRefine.core.print.Printer;

import net.sourceforge.czt.base.ast.Term;

/**
 * Classe utilitaria para manipulao de relacionamentos
 * 
 * @author Cristiano Gurgel
 * @author Alessandro Gurgel
 */
public class RelationsUtils {
	
	/**
	 * Retorna a pilha de relacionamentos de um termo. Caso esta no 
	 * exista,  ento criada uma nova.
	 * 
	 * @param termo o termo a ser visitado.
	 */
	public static RelationsAnnStack retornaPilha(Term termo) {
		RelationsAnnStackGetter getter = new RelationsAnnStackGetter();
		return termo.accept(getter);
	}
	
	public static RelationsAnn retornarTopo(Term termo) {
		
		RelationsAnnStackGetter getter = new RelationsAnnStackGetter();
		return termo.accept(getter).peek();
	}
	
	/**
	 * Testa se um termo tem uma pilha de relacionamento no vazia.
	 * 
	 * @param termo o termo a ser testado.
	 * @return <code>true</code> se o termo possui uma anotao com a
	 * 	pilha de relacionamento com um relacionamento, 
	 * 	<code>false</code> caso contrrio.
	 */
	public static boolean temRelacionamento(Term termo) {
		try {
			RelationsUtils.retornarTopo(termo);
			return true;
		} catch (EmptyStackException e) {
			return false;
		}
	}
	
	/**
	 * retorna a pilha de redo do termo, caso esta exista. Caso
	 * contrrio, este visitor cria a pilha e a retorna.
	 * 
	 * @param termo o termo a ser visitado
	 * @return a pilha de redo do termo, criando-a caso essa no 
	 * 		exista.
	 */
	public static RelationsAnnStackForRedo retornaPilhaRedo(Term termo) {
		RelationsAnnStackForRedoGetter getter = 
				new RelationsAnnStackForRedoGetter();
		return termo.accept(getter);
	}
	
	/**
	 * Remove a pilha de relacionamentos de uma AST.
	 * 
	 * @param termo o termo a ser visitado
	 * @throws NullPointerException caso termo seja 
	 * 		<code>null</code>.
	 * @see #visitTerm(Term)
	 */
	public static void removeStack(Term termo) {
		StackRemoveVisitor visitor = new StackRemoveVisitor();
		termo.accept(visitor);
	}
	
	/**
	 * Insere um novo relacionamento a pilha de relacionamento de 
	 * um termo.
	 * 
	 * @param novo o novo relaciomento.
	 * @param termo o termo a ser visitado.
	 */
	public static void insertRelation(RelationsAnn novo, Term termo, boolean toInsert) {
		if (toInsert) {
			RelationsAnnStackGetter getter = new RelationsAnnStackGetter();
			RelationsAnnStack pilha = termo.accept(getter);
			pilha.add(novo);
		}
	}
	
	/**
	 * Mtodo utilitrio para remover o relacionamento mais atual de 
	 * um termo
	 * 
	 * @param termo o termo a ter o relacionamento removido
	 */
	public static void removeRelationsAnn(Term termo) {
		RelationsAnnStackGetter getter = new RelationsAnnStackGetter();
		RelationsAnnStack pilha = termo.accept(getter);
		
		try {
			pilha.pop();
		} catch (EmptyStackException e){
			// nao faz nada
		}
		
	}
	
	/**
	 * Insere um relacionamento na pilha de redo de um termo.
	 * 
	 * @param rel o relacionamento a ser inserido na pilha de redo
	 * @param termo o termo a ter o relacionamento inserido
	 */
	public static void insertRedoRelation(RelationsAnn rel, Term termo) {
		RelationsAnnStackForRedo pilha = RelationsUtils.retornaPilhaRedo(termo);
		pilha.add(rel);
	}
	
	/**
	 * Remove a pilha de redo de uma AST.
	 * 
	 * @param termo a AST a ser visitada.
	 */
	public static void removeRedoStack(Term termo) {
		StackForRedoRemoveVisitor visitor = new StackForRedoRemoveVisitor();
		termo.accept(visitor);
	}
	
}
