/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.opsdischarge.syntacticfunctions;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.util.UnsupportedAstClassException;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.circus.ast.Action1;
import net.sourceforge.czt.circus.ast.Action2;
import net.sourceforge.czt.circus.ast.ActionD;
import net.sourceforge.czt.circus.ast.AssignmentCommand;
import net.sourceforge.czt.circus.ast.AssignmentPairs;
import net.sourceforge.czt.circus.ast.BasicAction;
import net.sourceforge.czt.circus.ast.CallAction;
import net.sourceforge.czt.circus.ast.CircusAction;
import net.sourceforge.czt.circus.ast.GuardedAction;
import net.sourceforge.czt.circus.ast.IfGuardedCommand;
import net.sourceforge.czt.circus.ast.LetVarAction;
import net.sourceforge.czt.circus.ast.MuAction;
import net.sourceforge.czt.circus.ast.PrefixingAction;
import net.sourceforge.czt.circus.ast.SchExprAction;
import net.sourceforge.czt.circus.ast.SpecStmtCommand;
import net.sourceforge.czt.circus.ast.SubstitutionAction;
import net.sourceforge.czt.circus.ast.VarDeclCommand;
import net.sourceforge.czt.circus.visitor.Action1Visitor;
import net.sourceforge.czt.circus.visitor.Action2Visitor;
import net.sourceforge.czt.circus.visitor.ActionDVisitor;
import net.sourceforge.czt.circus.visitor.AssignmentCommandVisitor;
import net.sourceforge.czt.circus.visitor.BasicActionVisitor;
import net.sourceforge.czt.circus.visitor.CallActionVisitor;
import net.sourceforge.czt.circus.visitor.GuardedActionVisitor;
import net.sourceforge.czt.circus.visitor.IfGuardedCommandVisitor;
import net.sourceforge.czt.circus.visitor.LetVarActionVisitor;
import net.sourceforge.czt.circus.visitor.MuActionVisitor;
import net.sourceforge.czt.circus.visitor.PrefixingActionVisitor;
import net.sourceforge.czt.circus.visitor.SchExprActionVisitor;
import net.sourceforge.czt.circus.visitor.SpecStmtCommandVisitor;
import net.sourceforge.czt.circus.visitor.SubstitutionActionVisitor;
import net.sourceforge.czt.circus.visitor.VarDeclCommandVisitor;
import net.sourceforge.czt.circuspatt.ast.JokerAction;
import net.sourceforge.czt.circuspatt.visitor.JokerActionVisitor;
import net.sourceforge.czt.z.ast.AndPred;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.MemPred;
import net.sourceforge.czt.z.ast.Name;
import net.sourceforge.czt.z.ast.NewOldPair;
import net.sourceforge.czt.z.ast.NextStroke;
import net.sourceforge.czt.z.ast.Pred;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.SchExpr;
import net.sourceforge.czt.z.ast.SchExpr2;
import net.sourceforge.czt.z.ast.SetExpr;
import net.sourceforge.czt.z.ast.Stroke;
import net.sourceforge.czt.z.ast.StrokeList;
import net.sourceforge.czt.z.ast.ZDeclList;
import net.sourceforge.czt.z.ast.ZName;
import net.sourceforge.czt.z.ast.ZNameList;
import net.sourceforge.czt.z.ast.ZSchText;
import net.sourceforge.czt.z.ast.ZStrokeList;
import net.sourceforge.czt.z.visitor.AndPredVisitor;
import net.sourceforge.czt.z.visitor.MemPredVisitor;
import net.sourceforge.czt.z.visitor.PredVisitor;
import net.sourceforge.czt.z.visitor.SchExpr2Visitor;
import net.sourceforge.czt.z.visitor.SchExprVisitor;
import circusRefine.core.crules.utils.NormalizationApplicationException;
import circusRefine.core.crules.utils.NormalizeApplier;
import circusRefine.core.opsdischarge.OPsDischargeUtils;
import circusRefine.core.opsdischarge.SyntacticFunctionsUtils;
import circusRefine.util.Pair;

/**
 * Classe utilizada na aplica��o da fun��o wrtV (written variables)
 * 
 * @author Cristiano Castro
 */
public class WrittenVariablesApplier 
extends SyntacticFunctionApplier<CircusAction, Expr> {

	/**
	 * Testa se duas refer�ncia � da forma x e x' ou x' e x.
	 * 
	 * @param lhs primeira refer�ncia analisada
	 * @param rhs segunda refer�ncia analisada
	 * @return um par contendo um booleano <code>true</code> caso 
	 *  as vari�veis passadas sejam do tipo descrito acima, 
	 *  <code>false</code> caso contr�rio. Caso o booleano seja
	 *  <code>true</code>, ent�o a segunda vari�vel do par 
	 *  representa cont�m o nome com menos strokes.
	 * @see #isXDashEqualXType(RefExpr, RefExpr)
	 */
	public static Pair<Boolean, ZName> isXDashEqualXTypeAux(RefExpr lhs, 
			RefExpr rhs) {
		Pair<Boolean, Pair<ZName, ZName>> result = 
			WrittenVariablesApplier.isXDashEqualXType(lhs, rhs);
		return new Pair<Boolean, ZName>(result.getFirst(), 
				result.getSecond().getFirst());
	}
	
	/**
	 * Testa se duas refer�ncia � da forma x e x' ou x' e x.
	 * 
	 * @param lhs primeira refer�ncia analisada
	 * @param rhs segunda refer�ncia analisada
	 * @return um par contendo um booleano <code>true</code> caso 
	 *  as vari�veis passadas sejam do tipo descrito acima, 
	 *  <code>false</code> caso contr�rio. Caso o booleano seja
	 *  <code>true</code>, ent�o o segundo elemento do par � outro par
	 *  contendo como primeiro elemento o nome com menos strokes e 
	 *  como segundo elemento o nome com mais strokes
	 */
	public static Pair<Boolean, Pair<ZName, ZName>> 
		isXDashEqualXType(RefExpr lhs, RefExpr rhs) {
		Pair<Boolean, Pair<ZName, ZName>> result = 
			new Pair<Boolean, Pair<ZName, ZName>>();
		try {
			ZName nomeLhs = lhs.getZName();
			ZName nomeRhs = rhs.getZName();
			if (nomeLhs.getWord().equals(nomeRhs.getWord())) {
				
				/* Mesmo nome de vari�vel: testa strokes */
				int numStrokesLhs = 
					WrittenVariablesApplier.getNumberOfStrokes(nomeLhs, 
							NextStroke.class);
				int numStrokesRhs = 
					WrittenVariablesApplier.getNumberOfStrokes(nomeRhs, 
							NextStroke.class);
				
				if (Math.abs(numStrokesLhs - numStrokesRhs) == 1) {
					
					/* 
					 * Diferen�a nos strokes � s� de 1. Vari�vel 
					 * n�o modificada!! 
					 */
					result.setFirst(true);
					if (numStrokesLhs < numStrokesRhs) {
						result.setSecond(new Pair<ZName, ZName>(nomeLhs, 
								nomeRhs));
					} else {
						result.setSecond(new Pair<ZName, ZName>(nomeRhs, 
								nomeLhs));
					}
				} else {
					result.setFirst(false);
				}
			}
		} catch (UnsupportedAstClassException e) {
			
			/* N�o � desse tipo */
			result.setFirst(false);
		}
		
		return result;
	}
	
	/**
	 * Verifica o n�mero de um determinado tipo de stroke em um 
	 * nome
	 * 
	 * @param nome o nome a ser testado
	 * @param tipoStroke o tipo do stroke a ser contabilizado
	 * @return o n�mero de strokes do nome do tipo determinado
	 */
	private static int getNumberOfStrokes(ZName nome, 
			Class<? extends Stroke> tipoStroke) {
		int result = 0;

		StrokeList listaStrokes = nome.getStrokeList();
		if (listaStrokes instanceof ZStrokeList) {
			ZStrokeList listaStrokesZ = (ZStrokeList) listaStrokes;
			
			for (Stroke stk : listaStrokesZ) {
				if (tipoStroke.isInstance(stk)) {
					
					/* 
					 * stk � uma inst�ncia do tipo de stroke 
					 * analisado 
					 */
					result++;
				}
			}
		}
		
		return result;
	}
	
	/** Nomes expandidos pelo passo anterior da resolu��o OPs */
	private Set<String> nomesExpandidos; 
	
	/**
	 * Inicia o objeto informando os nomes que foram expandidos no 
	 * passo anterior da aplica��o de OPs
	 * 
	 * @param nomesExpandidos os nomes que 
	 */
	public WrittenVariablesApplier(Set<String> nomesExpandidos) {
		this.setNomesExpandidos(nomesExpandidos);
	}

	/**
	 * @return the nomesExpandidos
	 */
	private Set<String> getNomesExpandidos() {
		return nomesExpandidos;
	}

	/**
	 * @param nomesExpandidos the nomesExpandidos to set
	 */
	private void setNomesExpandidos(Set<String> nomesExpandidos) {
		this.nomesExpandidos = nomesExpandidos;
	}

	/**
	 * Aplica a fun��o wrtV no formato da AST de Circus
	 * 
	 * @param args a a��o a ser analisada
	 * @return o conjunto de express�es contendo os nomes das 
	 *  vari�veis escritas na a��o
	 * @throws CannotEvaluateException caso a a��o n�o possa ser 
	 *  analisada
	 */
	public Expr apply(CircusAction args) throws CannotEvaluateException {
		return this.transformarRepresentacaoConjuntoNomes(this.getWrtV(args));
	}
	
	/**
	 * Aplica a fun��o wrtV a uma a��o retornando o conjunto de nomes
	 * no formato JAVA das vari�veis escritas
	 * 
	 * @param acao a a��o a ser analisada
	 * @return o conjunto de nomes resultante da aplica��o da fun��o
	 * @throws 
	 */
	public HashSet<ZName> getWrtV(CircusAction acao) 
	throws CannotEvaluateException {
		WrittenVariablesVisitor visitor = 
			new WrittenVariablesVisitor(this.getNomesExpandidos());
		try {
			HashSet<ZName> partial = acao.accept(visitor);
			return OPsDischargeUtils.removeStrokesOfNames(partial, 
					NextStroke.class);
		} catch (CannotEvaluateRunTimeException e) {
			//e.printStackTrace();
			throw new CannotEvaluateException(e);
		}
	}
	
	/**
	 * <p>Dado um predicado da forma P1 \land P2 \land ..., esse 
	 * m�todo captura as vari�veis x tais que x' = x � um dos 
	 * predicados Pn</p>
	 * 
	 * @param predicado o predicado da forma especificada acima a ser
	 *  analisado
	 * @return o conjunto de nomes que n�o � modificado pelo predicado
 	 */
	public HashSet<ZName> getUnmodifiedVariables(Pred predicado) {
		UnmodifiedVariablesVisitor visitor = new UnmodifiedVariablesVisitor();
		return predicado.accept(visitor);
	}

	/**
	 * Classe que implementa o visitor respons�vel pela aplica��o da 
	 * fun��o wrtV
	 * 
	 * @author Cristiano Castro
	 */
	protected class WrittenVariablesVisitor 
	implements ActionDVisitor<HashSet<ZName>>, Action2Visitor<HashSet<ZName>>,
	AssignmentCommandVisitor<HashSet<ZName>>, 
	BasicActionVisitor<HashSet<ZName>>, CallActionVisitor<HashSet<ZName>>,
	IfGuardedCommandVisitor<HashSet<ZName>>, 
	GuardedActionVisitor<HashSet<ZName>>,
	Action1Visitor<HashSet<ZName>>, JokerActionVisitor<HashSet<ZName>>,
	LetVarActionVisitor<HashSet<ZName>>, MuActionVisitor<HashSet<ZName>>,
	PrefixingActionVisitor<HashSet<ZName>>, 
	SchExprActionVisitor<HashSet<ZName>>, 
	SpecStmtCommandVisitor<HashSet<ZName>>, 
	SubstitutionActionVisitor<HashSet<ZName>>, 
	VarDeclCommandVisitor<HashSet<ZName>>, SchExpr2Visitor<HashSet<ZName>>, 
	TermVisitor<HashSet<ZName>>, SchExprVisitor<HashSet<ZName>> {

		/** 
		 * Conjunto utilizado para habilitar uma visita e uma 
		 * refer�ncia de a��o 
		 */
		private Set<ZName> referenciasAcoesAceitas;

		private DashedFreeVariablesApplier applier;

		/**
		 * @return the applier
		 */
		private DashedFreeVariablesApplier getApplier() {
			return applier;
		}

		/**
		 * @param applier the applier to set
		 */
		private void setApplier(DashedFreeVariablesApplier applier) {
			this.applier = applier;
		}

		/**
		 * @param referenciasAcoesAceitas
		 */
		public WrittenVariablesVisitor(Set<String> nomesExpandidos) {
			this.setReferenciasAcoesAceitas(new HashSet<ZName>());
			this.setApplier(new DashedFreeVariablesApplier(nomesExpandidos));
		}

		/**
		 * @return the referenciasAcoesAceitas
		 */
		private Set<ZName> getReferenciasAcoesAceitas() {
			return referenciasAcoesAceitas;
		}

		/**
		 * @param referenciasAcoesAceitas the referenciasAcoesAceitas to set
		 */
		private void setReferenciasAcoesAceitas(Set<ZName> referenciasAcoesAceitas) {
			this.referenciasAcoesAceitas = referenciasAcoesAceitas;
		}
		
		/**
		 * Visita um termo gen�rico. Dispara uma excess�o pois um termo 
		 * gen�rico n�o pode ser avaliado
		 * 
		 * @param arg0 o termo gen�rico a ser visitado
		 * @throws CannotEvaluateRunTimeException pois um termo gen�rico n�o pode 
		 *  ser avaliado
		 */
		public HashSet<ZName> visitTerm(Term arg0) {
			throw new CannotEvaluateRunTimeException("WrtV: N�o pode avaliar o " +
					"termo gen�rico " + arg0);
		}
		
		/**
		 * Visita uma a��o com uma lista de declara��es. Retira as 
		 * vari�veis declaradas do conjunto de vari�veis utilizadas
		 * 
		 * @param arg0 a a��o com a lista de declara��es
		 * @return o conjunto de vari�veis livres com as declara��es 
		 *  retiradas.
		 */
		public HashSet<ZName> visitActionD(ActionD arg0) {
			HashSet<ZName> result = arg0.getCircusAction().accept(this);
			Set<ZName> variaveisDeclaradas = 
				SyntacticFunctionsUtils.variaveisDeclaradas(arg0.getZDeclList());
			result.removeAll(variaveisDeclaradas);
			return result;
		}

		/**
		 * Visita uma a��o bin�ria. As vari�veis usadas s�o a uni�o das
		 * vari�veis das duas a��es na a��o bin�ria
		 * 
		 * @param arg0 a a��o bin�ria a ser visitada
		 * @return a uni�o do conjunto de vari�veis livres das duas
		 *  a��es da a��o bin�ria
		 */
		public HashSet<ZName> visitAction2(Action2 arg0) {
			HashSet<ZName> result = arg0.getLeftAction().accept(this);
			result.addAll(arg0.getRightAction().accept(this));
			return result;
		}

		/**
		 * Visita um comando de atribui��o. Pega as vari�veis do lado 
		 * esquerdo da atribui��o e as v�ri�veis livres de cada express�o
		 * 
		 * @param arg0 o comando de atribui��o a ser visitado
		 * @return o conjunto de vari�veis usadas do canal
		 */
		public HashSet<ZName> visitAssignmentCommand(AssignmentCommand arg0) {

			AssignmentPairs atribuicao = arg0.getAssignmentPairs();
			HashSet<ZName> result = new HashSet<ZName>();

			for (Name nome : atribuicao.getZLHS()) {
				if (nome instanceof ZName) {
					result.add((ZName) nome);

				} else {
					throw new CannotEvaluateRunTimeException("H� um nome no " +
							"lado esquerdo da atribui��o que n�o � inst�ncia " +
					"de de ZName");
				}
			}

			return result;
		}

		/**
		 * Visita uma call Action. 
		 * 
		 * @param arg0 o Call Action a ser visitado
		 * @return um conjunto vazio
		 */
		public HashSet<ZName> visitCallAction(CallAction arg0) {
			arg0.getExprList();
			return new HashSet<ZName>();
		}
		
		/**
		 * Visita uma a��o b�sica. Uma a��o b�sica n�o tem vari�veis 
		 * escritas
		 * 
		 * @param arg0 a a��o b�sica a ser visitada
		 * @return um conjunto vazio
		 */
		public HashSet<ZName> visitBasicAction(BasicAction arg0) {
			return new HashSet<ZName>();
		}

		/**
		 * Uma refer�ncia a uma a��o n�o pode ser avaliada
		 * 
		 * @param arg0 a refer�ncia a a��o a ser visitada
		 * @throws CannotEvaluateRunTimeException pois a a��o n�o pode
		 *  ser avaliada
		 */
	/*	public HashSet<ZName> visitCallAction(CallAction arg0) {
			if (this.getReferenciasAcoesAceitas().contains(arg0.getZName())) {

				 A CallAction � aceita 
				return new HashSet<ZName>();
			} else {

				 O nome da fun��o n�o � aceito 
				String nomeAcao = "__";

				if (arg0.getName() instanceof ZName) {
					nomeAcao = arg0.getZName().getWord();
				}

				throw new CannotEvaluateRunTimeException("A CallAction " +
						nomeAcao + "n�o pode ser avaliada na aplica��o de " +
				"usedV");
			}
		}
*/
		/**
		 * Uma {@link IfGuardedCommand} ter� seu predicado e sua a��o 
		 * analisados
		 * 
		 * @param arg0 o {@link IfGuardedCommand} a ser visitado
		 * @return a uni�o das vari�veis livres do predicado com as 
		 *  vari�veis usadas da a��o do comando
		 */
		public HashSet<ZName> visitIfGuardedCommand(IfGuardedCommand arg0) {
			HashSet<ZName> result = new HashSet<ZName>();
			for (CircusAction acao : arg0.getGuardedActionList()) {
				result.addAll(acao.accept(this));
			}
			return result;
		}

		/**
		 * Visita uma a��o guardada por um predicado. As vari�veis 
		 * escritas da a��o s�o as vari�veis escritas do predicado 
		 * unidas com as vari�v�is usadas da a��o guardada
		 * 
		 * @param arg0 a a��o guardada por um predicado
		 * @return o conjunto de vari�veis escritas da a��o guardada
		 *  unida com as vari�veis escritas do predicado
		 */
		public HashSet<ZName> visitGuardedAction(GuardedAction arg0) {
			HashSet<ZName> result = arg0.getCircusAction().accept(this);
			try {
				result.addAll(getApplier().getDFV(arg0.getPred()));
			} catch (CannotEvaluateException e) {
				throw new CannotEvaluateRunTimeException(e);
			}
			return result;
		}

		/**
		 * Visita a a��o encapsulada pela a��o un�ria
		 * 
		 * @param arg0 a a��o un�ria a ser analisada
		 * @return o conjunto de vari�veis livres da a��o encapsulada 
		 *  pela a��o un�ria
		 */
		public HashSet<ZName> visitAction1(Action1 arg0) {
			return arg0.getCircusAction().accept(this);
		}

		/**
		 * Visita um joker. Dispara uma excess�o pois um joker n�o
		 * pode ser analisado
		 * 
		 * @param arg0 o joker a ser visitado
		 * @throws CannotEvaluateRunTimeException pois o joker n�o 
		 *  pode ser analisado
		 */
		public HashSet<ZName> visitJokerAction(JokerAction arg0) {
			throw new CannotEvaluateRunTimeException("UsedV: O joker n�o pode" +
			" ser analisado");
		}

		/**
		 * Visita uma a��o que cont�m declara��es de vari�veis.
		 * 
		 * @param arg0 a a��o com declara��es de vari�veis a ser 
		 * visitada
		 * @return o conjunto de nomes da a��o subtra�do das vari�veis 
		 *  declaradas
		 */
		public HashSet<ZName> visitLetVarAction(LetVarAction arg0) {
			HashSet<ZName> result = arg0.getCircusAction().accept(this);
			Set<ZName> variaveisDeclaradas = 
				SyntacticFunctionsUtils.variaveisDeclaradas(arg0.getZDeclList());
			result.removeAll(variaveisDeclaradas);
			return result;
		}

		/**
		 * Visita uma a��o recursiva
		 * 
		 * @param arg0 a {@link MuAction} a ser visitada
		 * @return o conjunto de vari�veis livres da a��o menos
		 * @throws UnsupportedAstClassException se o nome do ponto de
		 *  recurs�o n�o for uma inst�ncia de {@link ZName}
		 */
		public HashSet<ZName> visitMuAction(MuAction arg0) {
			this.getReferenciasAcoesAceitas().add(arg0.getZName());
			HashSet<ZName> result = arg0.getCircusAction().accept(this);
			result.remove(arg0.getZName());
			this.getReferenciasAcoesAceitas().remove(arg0.getZName());
			return result;
		}

		/**
		 * Visita uma a��o prefixada por um canal
		 * 
		 * @param arg0 a a��o prefixada pelo canal
		 * @return o conjunto de vari�veis usadas da a��o menos o 
		 *  conjunto de vari�veis declaradas no canal
		 * @throws CannotEvaluateRunTimeException caso o canal n�o 
		 *  possa ser analisado
		 */
		public HashSet<ZName> visitPrefixingAction(PrefixingAction arg0) {
			try {
				InputVarsApplier ivApplier = new InputVarsApplier();
				HashSet<ZName> result = arg0.getCircusAction().accept(this);
				result.removeAll(ivApplier.apply(arg0.getCommunication()));
				return result;
			} catch (CannotEvaluateException e) {
				throw new CannotEvaluateRunTimeException(e);
			}
		}

		/**
		 * Visita uma combina��o bin�ria qualquer de dois esquemas: pode
		 * ser conjun��o, disjun��o, implica��o, se e somente se, etc. O
		 * resultado s�o as vari�veis livres da esquerda unido com as
		 * vari�veis livres do esquema da direita
		 * 
		 * @param arg a opera��o sobre esquemas a ser visitado
		 * @return o conjunto de vari�veis escritas do esquema da direita
		 *  unido com o conjunto de vari�veis escritas do esquema da 
		 *  esquerda
		 */
		public HashSet<ZName> visitSchExpr2(SchExpr2 arg0) {
				SchExpr normal;
				try {
					normal = NormalizeApplier.getInstance().applyRewrite(arg0);
				} catch (NormalizationApplicationException e) {
					throw new CannotEvaluateRunTimeException(e);
				}
				return normal.accept(this);
		}
		
		/**
		 * Visita um esquema. As vari�veis escritas s�o as vari�veis 
		 * as dashed-free variables do esquema, retirando-se as vari�vieis
		 * x tal que o predicado � da forma <code>... \land x' = x \land 
		 * ...</code>
		 * 
		 * @param arg0 o esquema a ser visitado
		 * @return o conjunto de vari�vies marcadas definidos menos as 
		 *  vari�veis definidas segundo a defini��o da vari�vel. 
		 */
		public HashSet<ZName> visitSchExpr(SchExpr arg0) {
			HashSet<ZName> result;
			try {
				result = getApplier().getDFV(arg0);
				result = OPsDischargeUtils.removeStrokesOfNames(result, 
						NextStroke.class);
				
				ZSchText esq = arg0.getZSchText();
				HashSet<ZName> unmodified = 
					getUnmodifiedVariables(esq.getPred());
				
				result = SyntacticFunctionsUtils.subtracaoConjuntos(result, 
						unmodified);
			} catch (CannotEvaluateException e) {
				throw new CannotEvaluateRunTimeException(e);
			}
			return result;
		}
		
		/**
		 * Pega as vari�veis livres da express�o encapsulada pelo 
		 * {@link SchExprAction}
		 * 
		 * @param arg0 a a��o representada por um esquema
		 * @return o conjunto de vari�veis livres do esquema
		 */
		public HashSet<ZName> visitSchExprAction(SchExprAction arg0) {
			return arg0.getExpr().accept(this);
		}

		/**
		 * Visita uma especifica��o no estilo Carrol Morgan. As 
		 * vari�veis escritas s�o as vari�veis do quadro
		 * 
		 * @param arg0 a especifica��o a ser testada
		 * @return o conjunto de nomes livres do quadro mais o 
		 *  conjunto de nomes livres da pr� e p�s condi��o
		 * @throws CannotEvaluateRunTimeException caso o quador da 
		 *  especifica��o n�o seja uma {@link ZNameList}
		 * @throws CannotEvaluateRunTimeException caso algum nome da 
		 *  lista de nomes do quadro n�o for um {@link ZName}
		 */
		public HashSet<ZName> visitSpecStmtCommand(SpecStmtCommand arg0) {
			try {
				HashSet<ZName> result = new HashSet<ZName>();
				for (Name nome : arg0.getZFrame()) {
					result.add((ZName) nome);
				}
				return result;
			} catch (UnsupportedAstClassException e) {
				throw new CannotEvaluateRunTimeException(e);
			} catch (ClassCastException e) {
				throw new CannotEvaluateRunTimeException(e);
			}
		}

		/**
		 * Visita uma express�o de substitui��o. Renomeia as vari�veis 
		 * livres de acordo com a lista de vari�veis renomeadas
		 * 
		 * @param arg0 a a��o de substitui��o
		 * @return o nome das vari�veis depois da substitui��o de 
		 *  nomes
		 * @throws CannotEvaluateRunTimeException se alguma opera��o
		 *  sobre a AST disparar um 
		 *  {@link UnsupportedAstClassException}
		 */
		public HashSet<ZName> visitSubstitutionAction(SubstitutionAction arg0) {
			HashSet<ZName> result = arg0.getCircusAction().accept(this);

			try {
				for (NewOldPair renomeacao : arg0.getZRenameList()) {
					if (result.remove(renomeacao.getZRefName())) {
						result.add(renomeacao.getZDeclName());
					}
				}
			} catch (UnsupportedAstClassException e) {
				throw new CannotEvaluateRunTimeException(e);
			}

			return result;
		}

		/**
		 * Visita uma comando de declara��o de vari�vel no estilo
		 * Carrol Morgan. O conjunto de vari�veis livres da a��o � 
		 * subtra�do dos nomes declarados na lista de declara��s de
		 * vari�veis.
		 * 
		 * @param arg0 o comando de declara��o de vari�vel 
		 * @return o conjunto de vari�veis usadas na a��o menos as 
		 *  vari�veis declaradas
		 * @throws CannotEvaluateRunTimeException caso a lista de 
		 *  nomes da declara��o de vari�veis n�o for uma 
		 *  {@link ZDeclList} 
		 */
		public HashSet<ZName> visitVarDeclCommand(VarDeclCommand arg0) {
			HashSet<ZName> result = arg0.getCircusAction().accept(this);
			try {
				result.removeAll(SyntacticFunctionsUtils.variaveisDeclaradas(
						arg0.getZDeclList()));
			} catch (UnsupportedAstClassException e) {
				throw new CannotEvaluateRunTimeException(e);
			}
			return result;
		}
	}
	
	/**
	 * <p>Classe utilizada para capturar as vari�veis n�o modificadas
	 * de um predicado. Dado um predicado P e uma vari�vel x, x � 
	 * considerada n�o modificado se P � da forma P0 \land P1 
	 * \land ... \land Pn e x' = x � um Pm tal que 0 <= m <= n</p>
	 *  
	 * @author Cristiano Castro
	 */
	private class UnmodifiedVariablesVisitor implements 
		AndPredVisitor<HashSet<ZName>>, MemPredVisitor<HashSet<ZName>>, 
		PredVisitor<HashSet<ZName>> {
		
		/**
		 * Visita os dois predicados da conjun��o.
		 * 
		 * @param arg0 a conjun��o a ser visitada
		 * @return a uni�o das vari�veis n�o modificadas dos dois 
		 *  lados da conjun��o
		 */
		public HashSet<ZName> visitAndPred(AndPred arg0) {
			HashSet<ZName> result = arg0.getLeftPred().accept(this);
			result.addAll(arg0.getRightPred().accept(this));
			return result;
		}
		
		/**
		 * Um predicado gen�rico n�o cont�m vari�veis n�o modificadas
		 * segundo sua defini��o
		 * 
		 * @param arg0 o predicado gen�rico
		 * @return um conjunto de vari�veis vazio
		 */
		public HashSet<ZName> visitPred(Pred arg0) {
			return new HashSet<ZName>();
		}

		/**
		 * Testa se o argumento � da forma x = x' ou x' = x e caso 
		 * positivo adiciona x ao conjunto de vari�veis n�o 
		 * modificadas do predicado
		 * 
		 * @param arg0 a rela��o a ser testada
		 * @return o conjunto unit�rio contendo a vari�vel livre caso
		 *  a rela��o seja da forma descrita acima, ou um conjunto 
		 *  vazio caso contr�rio
		 */
		public HashSet<ZName> visitMemPred(MemPred arg0) {
			HashSet<ZName> result;
			if (arg0.getMixfix() && arg0.getRightExpr() instanceof SetExpr) {
				
				/* � uma rela��o de igualdade */
				Expr lhs = arg0.getLeftExpr();
				Expr rhs = 
					((SetExpr) arg0.getRightExpr()).getZExprList().get(0);
				
				if (lhs instanceof RefExpr && rhs instanceof RefExpr) {
					
					Pair<Boolean, ZName> verificacao = 
						WrittenVariablesApplier.isXDashEqualXTypeAux((RefExpr) lhs,
								(RefExpr) rhs);
					
					if (verificacao.getFirst()) {
						
						/* Representa uma unmodified dashed variable */
						ZName naoMod = verificacao.getSecond();
						result = 
							new HashSet<ZName>(Collections.singleton(naoMod));
					} else {
						
						/* N�o � uma vari�vel n�o modificada */
						result = new HashSet<ZName>();
					}
					
				} else {
					
					/* N�o � uma igualdade entre vari�veis */
					result = new HashSet<ZName>();
				}
				
			} else {
				
				/* N�o � uma rela��o de igualdade */
				result = new HashSet<ZName>();
			}
			
			return result;
		}
		
	}
	
}
