/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.opsdischarge.syntacticfunctions;

import java.util.HashSet;
import java.util.Set;

import net.sourceforge.czt.base.util.UnsupportedAstClassException;
import net.sourceforge.czt.circus.ast.Action1;
import net.sourceforge.czt.circus.ast.Action2;
import net.sourceforge.czt.circus.ast.ActionD;
import net.sourceforge.czt.circus.ast.AssignmentCommand;
import net.sourceforge.czt.circus.ast.AssignmentPairs;
import net.sourceforge.czt.circus.ast.BasicAction;
import net.sourceforge.czt.circus.ast.CallAction;
import net.sourceforge.czt.circus.ast.CircusAction;
import net.sourceforge.czt.circus.ast.GuardedAction;
import net.sourceforge.czt.circus.ast.IfGuardedCommand;
import net.sourceforge.czt.circus.ast.LetVarAction;
import net.sourceforge.czt.circus.ast.MuAction;
import net.sourceforge.czt.circus.ast.PrefixingAction;
import net.sourceforge.czt.circus.ast.SchExprAction;
import net.sourceforge.czt.circus.ast.SpecStmtCommand;
import net.sourceforge.czt.circus.ast.SubstitutionAction;
import net.sourceforge.czt.circus.ast.VarDeclCommand;
import net.sourceforge.czt.circus.visitor.Action1Visitor;
import net.sourceforge.czt.circus.visitor.Action2Visitor;
import net.sourceforge.czt.circus.visitor.ActionDVisitor;
import net.sourceforge.czt.circus.visitor.AssignmentCommandVisitor;
import net.sourceforge.czt.circus.visitor.BasicActionVisitor;
import net.sourceforge.czt.circus.visitor.CallActionVisitor;
import net.sourceforge.czt.circus.visitor.GuardedActionVisitor;
import net.sourceforge.czt.circus.visitor.IfGuardedCommandVisitor;
import net.sourceforge.czt.circus.visitor.LetVarActionVisitor;
import net.sourceforge.czt.circus.visitor.MuActionVisitor;
import net.sourceforge.czt.circus.visitor.PrefixingActionVisitor;
import net.sourceforge.czt.circus.visitor.SchExprActionVisitor;
import net.sourceforge.czt.circus.visitor.SpecStmtCommandVisitor;
import net.sourceforge.czt.circus.visitor.SubstitutionActionVisitor;
import net.sourceforge.czt.circus.visitor.VarDeclCommandVisitor;
import net.sourceforge.czt.circuspatt.ast.JokerAction;
import net.sourceforge.czt.circuspatt.visitor.JokerActionVisitor;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.Name;
import net.sourceforge.czt.z.ast.NewOldPair;
import net.sourceforge.czt.z.ast.NextStroke;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.ZDeclList;
import net.sourceforge.czt.z.ast.ZName;
import net.sourceforge.czt.z.ast.ZNameList;
import circusRefine.core.opsdischarge.OPsDischargeUtils;
import circusRefine.core.opsdischarge.SyntacticFunctionsUtils;

/**
 * Classe respons�vel pela aplica��o da fun��o FreeVariables
 * 
 * @author Cristiano Castro
 */
public class UsedVariablesApplier 
extends SyntacticFunctionApplier<CircusAction, Expr> {

	/** O aplicador de FV_Expr */
	private Set<String> nomesExpandidos;

	/**
	 * Inicia o aplicador informando o conjunto de nomes que foram 
	 * alvo de expans�o no passo anterior
	 * 
	 * @param nomeExpandidos o conjunto de nomes expandidos no passo
	 *  anterior da resolu��o de OPs
	 */
	public UsedVariablesApplier(Set<String> nomeExpandidos) {
		this.setNomesExpandidos(nomeExpandidos);
	}

	/**
	 * Aplica a fun��o usedV de acordo com a AST de circus
	 * 
	 * @param args a a��o a ser visitada
	 * @return o conjunto de nomes {@link RefExpr} que representam 
	 *  vari�veis utilizadas
	 */
	public Expr apply(CircusAction args) throws CannotEvaluateException {
		HashSet<ZName> uv = this.getUsedVariables(args);
		return this.transformarRepresentacaoConjuntoNomes(uv);
	}

	/**
	 * Retorna as vari�veis usadas em uma a��o. Aplica a fun��o usedV
	 * 
	 * @param acao a a��o a ser analisada
	 * @return o conjunto de nomes usados na a��o
	 * @throws CannotEvaluateException caso a fun��o n�o possa ser 
	 *  avaliada.
	 */
	public HashSet<ZName> getUsedVariables(CircusAction acao) 
	throws CannotEvaluateException {
		UsedVariablesVisitor visitor = 
			new UsedVariablesVisitor(this.getNomesExpandidos());
		try {
			HashSet<ZName> result = acao.accept(visitor);
  			return OPsDischargeUtils.removeStrokesOfNames(result, 
  					NextStroke.class);
		} catch (CannotEvaluateRunTimeException e) {
			throw new CannotEvaluateException(e);
		}
	}

	/**
	 * @return the nomesExpandidos
	 */
	private Set<String> getNomesExpandidos() {
		return nomesExpandidos;
	}

	/**
	 * @param nomesExpandidos the nomesExpandidos to set
	 */
	private void setNomesExpandidos(Set<String> nomesExpandidos) {
		this.nomesExpandidos = nomesExpandidos;
	}
	
	/**
	 * Classe que visita os termos para a aplica��o da fun��o 
	 * <i>usedV</i>
	 * 
	 * @author Cristiano Castro
	 */
	protected class UsedVariablesVisitor 
	implements ActionDVisitor<HashSet<ZName>>, Action2Visitor<HashSet<ZName>>,
	AssignmentCommandVisitor<HashSet<ZName>>, 
	BasicActionVisitor<HashSet<ZName>>, 
	CallActionVisitor<HashSet<ZName>>,
	IfGuardedCommandVisitor<HashSet<ZName>>, 
	GuardedActionVisitor<HashSet<ZName>>,
	Action1Visitor<HashSet<ZName>>, JokerActionVisitor<HashSet<ZName>>,
	LetVarActionVisitor<HashSet<ZName>>, MuActionVisitor<HashSet<ZName>>,
	PrefixingActionVisitor<HashSet<ZName>>, 
	SchExprActionVisitor<HashSet<ZName>>, 
	SpecStmtCommandVisitor<HashSet<ZName>>, 
	SubstitutionActionVisitor<HashSet<ZName>>, 
	VarDeclCommandVisitor<HashSet<ZName>> {
		
		/** 
		 * Conjunto utilizado para habilitar uma visita e uma 
		 * refer�ncia de a��o 
		 */
		private Set<ZName> referenciasAcoesAceitas;
		
		private FreeVariablesApplier applier;
		
		/**
		 * @return the applier
		 */
		private FreeVariablesApplier getApplier() {
			return applier;
		}

		/**
		 * @param applier the applier to set
		 */
		private void setApplier(FreeVariablesApplier applier) {
			this.applier = applier;
		}

		/**
		 * @param referenciasAcoesAceitas
		 */
		public UsedVariablesVisitor(Set<String> nomesExpandidos) {
			//Deve ser as ações que já foram visitadas
			this.setReferenciasAcoesAceitas(new HashSet<ZName>());
			this.setApplier(new FreeVariablesApplier(nomesExpandidos));
		}

		/**
		 * @return the referenciasAcoesAceitas
		 */
		private Set<ZName> getReferenciasAcoesAceitas() {
			return referenciasAcoesAceitas;
		}

		/**
		 * @param referenciasAcoesAceitas the referenciasAcoesAceitas to set
		 */
		private void setReferenciasAcoesAceitas(Set<ZName> referenciasAcoesAceitas) {
			this.referenciasAcoesAceitas = referenciasAcoesAceitas;
		}

		/**
		 * Visita uma a��o com uma lista de declara��es. Retira as 
		 * vari�veis declaradas do conjunto de vari�veis utilizadas
		 * 
		 * @param arg0 a a��o com a lista de declara��es
		 * @return o conjunto de vari�veis livres com as declara��es 
		 *  retiradas.
		 */
		public HashSet<ZName> visitActionD(ActionD arg0) {
			HashSet<ZName> result = arg0.getCircusAction().accept(this);
			Set<ZName> variaveisDeclaradas = 
				SyntacticFunctionsUtils.variaveisDeclaradas(
						arg0.getZDeclList());
			result.removeAll(variaveisDeclaradas);
			return result;
		}

		/**
		 * Visita uma a��o bin�ria. As vari�veis usadas s�o a uni�o das
		 * vari�veis das duas a��es na a��o bin�ria
		 * 
		 * @param arg0 a a��o bin�ria a ser visitada
		 * @return a uni�o do conjunto de vari�veis livres das duas
		 *  a��es da a��o bin�ria
		 */
		public HashSet<ZName> visitAction2(Action2 arg0) {
			HashSet<ZName> result = arg0.getLeftAction().accept(this);
			result.addAll(arg0.getRightAction().accept(this));
			return result;
		}

		/**
		 * Visita um comando de atribui��o. Pega as vari�veis do lado 
		 * esquerdo da atribui��o e as v�ri�veis livres de cada express�o
		 * 
		 * @param arg0 o comando de atribui��o a ser visitado
		 * @return o conjunto de vari�veis usadas do canal
		 * @throws CannotEvaluateRunTimeException caso haja algum nome da 
		 * lista de atribui��o que n�o seja inst�ncia de {@link ZName}
		 * @throws CannotEvaluateRunTimeException caso n�o se possa 
		 * avaliar as express�es do lado direito das atribui��es
		 */
		public HashSet<ZName> visitAssignmentCommand(AssignmentCommand arg0) {

			AssignmentPairs atribuicao = arg0.getAssignmentPairs();
			HashSet<ZName> result = new HashSet<ZName>();

			for (Name nome : atribuicao.getZLHS()) {
				if (nome instanceof ZName) {
					result.add((ZName) nome);

				} else {
					throw new CannotEvaluateRunTimeException("H� um nome no " +
							"lado esquerdo da atribui��o que n�o � inst�ncia " +
					"de de ZName");
				}
			}

			for (Expr expr : atribuicao.getZRHS()) {
				try {
					result.addAll(getApplier().getFV(expr));
				} catch (CannotEvaluateException e) {
					throw new CannotEvaluateRunTimeException(e);
				}
			}

			return result;
		}

		/**
		 * Visita uma a��o b�sica. Uma a��o b�sica n�o tem vari�veis 
		 * usadas
		 * 
		 * @param arg0 a a��o b�sica a ser visitada
		 * @return um conjunto vazio
		 */
		public HashSet<ZName> visitBasicAction(BasicAction arg0) {
			return new HashSet<ZName>();
		}

		/**
		 * Uma refer�ncia a uma a��o n�o pode ser avaliada
		 * 
		 * @param arg0 a refer�ncia a a��o a ser visitada
		 * @throws CannotEvaluateRunTimeException pois a a��o n�o pode
		 *  ser avaliada
		 */
		public HashSet<ZName> visitCallAction(CallAction arg0) {
		//	if (this.getReferenciasAcoesAceitas().contains(arg0.getZName())) {

				// A CallAction � aceita 
				return new HashSet<ZName>();
		/*	} else {

				 O nome da fun��o n�o � aceito 
				String nomeAcao = "__";

				if (arg0.getName() instanceof ZName) {
					nomeAcao = arg0.getZName().getWord();
				}

				throw new CannotEvaluateRunTimeException("A CallAction " +
						nomeAcao + "n�o pode ser avaliada na aplica��o de " +
				"usedV");
			}*/
		}

		/**
		 * Uma {@link IfGuardedCommand} ter� seu predicado e sua a��o 
		 * analisada
		 * 
		 * @param arg0 o {@link IfGuardedCommand} a ser visitado
		 * @return a uni�o das vari�veis livres do predicado com as 
		 *  vari�veis usadas da a��o do comando
		 */
		public HashSet<ZName> visitIfGuardedCommand(IfGuardedCommand arg0) {
			HashSet<ZName> result = new HashSet<ZName>();
			for (CircusAction acao : arg0.getGuardedActionList()) {
				result.addAll(acao.accept(this));
			}
			return result;
		}

		/**
		 * Visita uma a��o guardada por um predicado. As vari�veis 
		 * usadas da a��o s�o as vari�veis livres do predicado unidas 
		 * com as vari�v�is usadas da a��o guardada
		 * 
		 * @param arg0 a a��o guardada por um predicado
		 * @return o conjunto de vari�veis usadas da a��o guardada
		 *  unida com as vari�veis livres do predicado
		 * @throws CannotEvaluateRunTimeException caso o predicado n�o
		 *  possa ser avaliado
		 */
		public HashSet<ZName> visitGuardedAction(GuardedAction arg0) {
			HashSet<ZName> result = arg0.getCircusAction().accept(this);
			try {
				result.addAll(getApplier().getFV(arg0.getPred()));
			} catch (CannotEvaluateException e) {
				throw new CannotEvaluateRunTimeException(e);
			}
			return result;
		}

		/**
		 * Visita a a��o encapsulada pela a��o un�ria
		 * 
		 * @param arg0 a a��o un�ria a ser analisada
		 * @return o conjunto de vari�veis livres da a��o encapsulada 
		 *  pela a��o un�ria
		 */
		public HashSet<ZName> visitAction1(Action1 arg0) {
			return arg0.getCircusAction().accept(this);
		}

		/**
		 * Visita um joker. Dispara uma excess�o pois um joker n�o
		 * pode ser analisado
		 * 
		 * @param arg0 o joker a ser visitado
		 * @throws CannotEvaluateRunTimeException pois o joker n�o 
		 *  pode ser analisado
		 */
		public HashSet<ZName> visitJokerAction(JokerAction arg0) {
			throw new CannotEvaluateRunTimeException("UsedV: O joker n�o pode" +
			" ser analisado");
		}

		/**
		 * Visita uma a��o que cont�m declara��es de vari�veis.
		 * 
		 * @param arg0 a a��o com declara��es de vari�veis a ser 
		 * visitada
		 * @return o conjunto de nomes da a��o subtra�do das vari�veis 
		 *  declaradas
		 */
		public HashSet<ZName> visitLetVarAction(LetVarAction arg0) {
			HashSet<ZName> result = arg0.getCircusAction().accept(this);
			ZDeclList lista = arg0.getZDeclList();
			Set<ZName> variaveisDeclaradas = 
				SyntacticFunctionsUtils.variaveisDeclaradas(lista);
			result.removeAll(variaveisDeclaradas);
			return result;
		}

		/**
		 * Visita uma a��o recursiva
		 * 
		 * @param arg0 a {@link MuAction} a ser visitada
		 * @return o conjunto de vari�veis livres da a��o menos
		 * @throws UnsupportedAstClassException se o nome do ponto de
		 *  recurs�o n�o for uma inst�ncia de {@link ZName}
		 */
		public HashSet<ZName> visitMuAction(MuAction arg0) {
			this.getReferenciasAcoesAceitas().add(arg0.getZName());
			HashSet<ZName> result = arg0.getCircusAction().accept(this);
			result.remove(arg0.getZName());
			this.getReferenciasAcoesAceitas().remove(arg0.getZName());
			return result;
		}

		/**
		 * Visita uma a��o prefixada por um canal
		 * 
		 * @param arg0 a a��o prefixada pelo canal
		 * @return o conjunto de vari�veis usadas da a��o menos o 
		 *  conjunto de vari�veis declaradas no canal
		 * @throws CannotEvaluateRunTimeException caso o canal n�o 
		 *  possa ser analisado
		 */
		public HashSet<ZName> visitPrefixingAction(PrefixingAction arg0) {
			try {
				InputVarsApplier ivApplier = new InputVarsApplier();
				HashSet<ZName> result = arg0.getCircusAction().accept(this);
				result.removeAll(ivApplier.apply(arg0.getCommunication()));
				
				/* 
				 * adiciona as vari�veis livres da express�o de saida do 
				 * canal 
				 */
				HashSet<Expr> expressoes =  
					SyntacticFunctionsUtils.expressoesSaidasCanal(
							arg0.getCommunication());
				for (Expr saida : expressoes) {
					result.addAll(this.getApplier().getFV(saida));
				}
				
				return result;
			} catch (CannotEvaluateException e) {
				throw new CannotEvaluateRunTimeException(e);
			}
		}

		/**
		 * Pega as vari�veis livres da express�o encapsulada pelo 
		 * {@link SchExprAction}
		 * 
		 * @param arg0 a a��o representada por um esquema
		 * @return o conjunto de vari�veis livres do esquema
		 * @throws CannotEvaluateRunTimeException se o esquema n�o puder
		 *  ser analisado
		 */
		public HashSet<ZName> visitSchExprAction(SchExprAction arg0) {
			try {
				return getApplier().getFV(arg0.getExpr());
			} catch (CannotEvaluateException e) {
				throw new CannotEvaluateRunTimeException(e);
			}
		}

		/**
		 * Visita uma especifica��o no estilo Carrol Morgan. As 
		 * vari�veis livres s�o as vari�veis do quadro mais as 
		 * vari�veis livres da pr� e p�s-condi��o
		 * 
		 * @param arg0 a especifica��o a ser testada
		 * @return o conjunto de nomes livres do quadro mais o 
		 *  conjunto de nomes livres da pr� e p�s condi��o
		 * @throws CannotEvaluateRunTimeException caso o quadro da 
		 *  especifica��o n�o seja uma {@link ZNameList}
		 * @throws CannotEvaluateRunTimeException caso algum nome da 
		 *  lista de nomes do quadro n�o for um {@link ZName}
		 * @throws CannotEvaluateRunTimeException caso os predicados da
		 *  especifica��o n�o puderem ser analisados
		 */
		public HashSet<ZName> visitSpecStmtCommand(SpecStmtCommand arg0) {
			try {
				HashSet<ZName> result = new HashSet<ZName>();
				result.addAll(getApplier().getFV(arg0.getPre()));
				result.addAll(getApplier().getFV(arg0.getPost()));
				for (Name nome : arg0.getZFrame()) {
					result.add((ZName) nome);
				}
				return result;
			} catch (UnsupportedAstClassException e) {
				throw new CannotEvaluateRunTimeException(e);
			} catch (ClassCastException e) {
				throw new CannotEvaluateRunTimeException(e);
			} catch (CannotEvaluateException e) {
				throw new CannotEvaluateRunTimeException(e);
			}
			
		}

		/**
		 * Visita uma express�o de substitui��o. Renomeia as vari�veis 
		 * livres de acordo com a lista de vari�veis renomeadas
		 * 
		 * @param arg0 a a��o de substitui��o
		 * @return o nome das vari�veis depois da substitui��o de 
		 *  nomes
		 * @throws CannotEvaluateRunTimeException se alguma opera��o
		 *  sobre a AST disparar um 
		 *  {@link UnsupportedAstClassException}
		 */
		public HashSet<ZName> visitSubstitutionAction(SubstitutionAction arg0) {
			HashSet<ZName> result = arg0.getCircusAction().accept(this);
			
			try {
				for (NewOldPair renomeacao : arg0.getZRenameList()) {
					if (result.remove(renomeacao.getZRefName())) {
						result.add(renomeacao.getZDeclName());
					}
				}
			} catch (UnsupportedAstClassException e) {
				throw new CannotEvaluateRunTimeException(e);
			}
			
			return result;
		}

		/**
		 * Visita uma comando de declara��o de vari�vel no estilo
		 * Carrol Morgan. O conjunto de vari�veis livres da a��o � 
		 * subtra�do dos nomes declarados na lista de declara��s de
		 * vari�veis.
		 * 
		 * @param arg0 o comando de declara��o de vari�vel 
		 * @return o conjunto de vari�veis usadas na a��o menos as 
		 *  vari�veis declaradas
		 * @throws CannotEvaluateRunTimeException caso a lista de 
		 *  nomes da declara��o de vari�veis n�o for uma 
		 *  {@link ZDeclList} 
		 */
		public HashSet<ZName> visitVarDeclCommand(VarDeclCommand arg0) {
			HashSet<ZName> result = arg0.getCircusAction().accept(this);
			try {
				result.removeAll(SyntacticFunctionsUtils.variaveisDeclaradas(
						arg0.getZDeclList()));
			} catch (UnsupportedAstClassException e) {
				throw new CannotEvaluateRunTimeException(e);
			}
			return result;
		}
	}


}
