/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.opsdischarge.syntacticfunctions;

import java.util.HashSet;
import java.util.Set;
import java.util.TreeSet;

import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.util.UnsupportedAstClassException;
import net.sourceforge.czt.circus.ast.BasicChannelSetExpr;
import net.sourceforge.czt.circus.ast.CircusCommunicationList;
import net.sourceforge.czt.circus.ast.Communication;
import net.sourceforge.czt.circuspatt.ast.CircusPatternFactory;
import net.sourceforge.czt.circuspatt.impl.CircusPatternFactoryImpl;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.Pred;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.SetExpr;
import net.sourceforge.czt.z.ast.ZName;
import circusRefine.core.opsdischarge.OPsDischargeUtils;

/**
 * Interface implementada por qualquer aplicador de funo sinttica.
 * Essa interface indentifica o tipo da funo
 * 
 * @param <A> o tipo dos argumentos da funo
 * @param <V> o valor retornado pela funo
 * @author Cristiano Castro
 */
public abstract class SyntacticFunctionApplier<A extends Term, V extends Term> {

	/**
	 * Verifica se um predicado 
	 * 
	 * @param dashedVar
	 * @param predicado
	 * @return
	 */
	public static HashSet<Expr> verificaModificacaoVariavel(ZName dashedVar, 
			Pred predicado) {
		return null;
	}
	
	/** Fbrica para as leis a serem aplicadas */
	CircusPatternFactory factory = new CircusPatternFactoryImpl();
	
	/**
	 * Mtodo que aplica a funo sinttica
	 * 
	 * @param args o argumento da funo sinttica
	 * @return o resultado da funo sinttica
	 */
	public abstract V apply(A args) throws CannotEvaluateException;
	
	/**
	 * Tranforma uma representao de conjunto de nomes de um 
	 * {@link Set} para um {@link SetExpr} e assim ficar de acordo com
	 * a AST de circus 
	 * 
	 * @param conj o conjunto de nomes a ser mudado
	 * @return o {@link SetExpr} representando o conjunto de nomes
	 * @see #transformarRepresentacaoConjuntoNomes(SetExpr)
	 */
	protected Expr transformarRepresentacaoConjuntoNomes(Set<ZName> conj) {
		Expr aRetornar;
		if (!conj.isEmpty()) {
			SetExpr result = 
				this.factory.createSetExpr(factory.createZExprList());

			/* Monta a expresso resultado */
			for (ZName nome : conj) {
				RefExpr toAdd = this.factory.createRefExpr(nome, 
						this.factory.createZExprList(), false, false);
				result.getZExprList().add(toAdd);
			}

			aRetornar = result;
		} else {
			aRetornar = OPsDischargeUtils.criarConjuntoVazio();
		}
		return aRetornar;
	}

	/**
	 * Monta, a partir de um conjunto de canais 
	 * {@link BasicChannelSetExpr}, o conjunto de nomes do canal
	 * {@link Set}
	 * 
	 * @param cs o conjunto de canais no estilo AST Circus
	 * @return o conjunto de nomes de canais no estilo JAVA
	 * @throws UnsupportedAstClassException caso a lista de canais do
	 *  conjunto no for uma instncia de 
	 *  {@link CircusCommunicationList}
	 * @see {@link #getNameOfACommunication(Communication)}
	 */
	protected Set<ZName> capturarNomesCanais(BasicChannelSetExpr cs) {
		Set<ZName> result = new TreeSet<ZName>();
		
		for (Communication comm : cs.getCircusCommunicationList()) {
			result.add(this.getNameOfACommunication(comm));
		}
		
		return result;
	}
	
	/**
	 * Acha e monta o conjunto de canais utilizados pela ao com
	 * prefixo
	 * 
	 * @param c a comunicao
	 * @return o {@link ZName} representando o nome da comunicao
	 * @throws UnsupportedAstClassException se o nome do canal no
	 *  for uma instncia de um {@link ZName}.
	 */
	protected ZName getNameOfACommunication(Communication c) {
		RefExpr ref = c.getChannelExpr();
		ZName nomeCanal = ref.getZName();
		return nomeCanal;
	}
	
}
