/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.opsdischarge.syntacticfunctions;

import java.util.Arrays;
import java.util.HashSet;

import net.sourceforge.czt.circus.ast.Action2;
import net.sourceforge.czt.circus.ast.ActionIte;
import net.sourceforge.czt.circus.ast.CircusAction;
import net.sourceforge.czt.circus.ast.ExtChoiceAction;
import net.sourceforge.czt.circus.ast.IntChoiceAction;
import net.sourceforge.czt.circus.ast.ParAction;
import net.sourceforge.czt.circus.ast.PrefixingAction;
import net.sourceforge.czt.circus.visitor.ActionIteVisitor;
import net.sourceforge.czt.circus.visitor.CircusActionVisitor;
import net.sourceforge.czt.circus.visitor.ExtChoiceActionVisitor;
import net.sourceforge.czt.circus.visitor.IntChoiceActionVisitor;
import net.sourceforge.czt.circus.visitor.ParActionVisitor;
import net.sourceforge.czt.circus.visitor.PrefixingActionVisitor;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.ZName;

/**
 * Aplicador da funo initials: Action
 * 
 * @author Cristiano Castro
 */
public class InitialsApplier extends SyntacticFunctionApplier<CircusAction, 
	Expr> {

	/**
	 * Aplica a funo initials a uma ao 
	 * 
	 * @param args a ao a ser analisada
	 * @return o conjunto de canais no formato da AST de 
	 *  {@link CircusAction}
	 */
	public Expr apply(CircusAction args) throws CannotEvaluateException {
		return this.transformarRepresentacaoConjuntoNomes(
				this.getInitials(args));
	}

	/**
	 * Retorna os canais iniciais de uma ao.
	 * 
	 * @param acao a ao a ser analisada
	 * @return o conjunto de nomes de canais
	 * @throws CannotEvaluateException caso a funo no possa ser 
	 *  aplicada  ao
	 */
	public HashSet<ZName> getInitials(CircusAction acao) 
	throws CannotEvaluateException {
		try {
			InitialsVisitor visitor = new InitialsVisitor();
			return acao.accept(visitor);
		} catch (CannotEvaluateRunTimeException e) {
			throw new CannotEvaluateException(e);
		}
	}

	/**
	 * Visitor para extrair os canais presentes em <i>initials(A)</i>
	 * 
	 * @author Cristiano Castro
	 */
	protected class InitialsVisitor 
	implements CircusActionVisitor<HashSet<ZName>>, 
	PrefixingActionVisitor<HashSet<ZName>>, 
	ExtChoiceActionVisitor<HashSet<ZName>>,
	IntChoiceActionVisitor<HashSet<ZName>>, ParActionVisitor<HashSet<ZName>>,
	ActionIteVisitor<HashSet<ZName>> {

		/**
		 * Visita uma ao com prefixo (canal).
		 * 
		 * @param arg0 a ao prefixada.
		 * @return retorna o conjunto unitrio com o nome do canal.
		 */
		public HashSet<ZName> visitPrefixingAction(PrefixingAction arg0) {
			ZName nomeCanal = getNameOfACommunication(arg0.getCommunication());
			return new HashSet<ZName>(Arrays.asList(nomeCanal));
		}

		/**
		 * Uma ao genrica tem initials vazio
		 * 
		 * @param arg0 a ao genrica a ser visitada
		 * @return um conjunto de nomes de canal vazio
		 */
		public HashSet<ZName> visitCircusAction(CircusAction arg0) {
			return new HashSet<ZName>();
		}

		/**
		 * Pega os canais iniciais dos dois lados da escolha externa
		 * 
		 * @param a escolha externa a ser visitada
		 * @return os canais iniciais do LHS da escolha externa unido
		 *  com os canais do RHS da escolha externa
		 * @see #unirCanais(Action2)
		 */
		public HashSet<ZName> visitExtChoiceAction(ExtChoiceAction arg0) {
			return this.unirCanais(arg0);
		}

		/**
		 * O conjunto de canais initials de uma escolha interna  a
		 * unio do conjunto de canais iniciais dos dois lados da 
		 * escolha.
		 * 
		 * @param arg0 a escolha interna a ser visitada
		 * @return o conjunto de nomes de canais iniciais, isto , a 
		 *  unio dos canais iniciais das duas aes da escolha interna
		 * @see #unirCanais(Action2)
		 */
		public HashSet<ZName> visitIntChoiceAction(IntChoiceAction arg0) {
			return this.unirCanais(arg0);
		}

		/**
		 * Visita uma ao paralela abstrata. O conjunto de canais 
		 * iniciais  o conjunto de canais iniciais da ao da 
		 * esquerda mais o conjunto de canais iniciais da ao da
		 * direita
		 * 
		 * @param arg0 a ao abstrata paralela a ser analisada
		 * @return o conjunto de canais iniciais da ao da direita 
		 *  unido com os canais iniciais da ao da esquerda 
		 */
		public HashSet<ZName> visitParAction(ParAction arg0) {
			return this.unirCanais(arg0);
		}

		/**
		 * O conjunto de canais de uma ao iterada no pode ser 
		 * avaliada.
		 * 
		 * @param arg0 a ao iterada a ser analisada
		 * @throws CannotEvaluateRunTimeException
		 */
		public HashSet<ZName> visitActionIte(ActionIte arg0) {
			throw new CannotEvaluateRunTimeException("No consegue analisar " +
			"os canais iniciais de uma ao iterada");
		}

		/**
		 * Une os initials dos dois lados de uma ao binria.
		 * 
		 * @param arg a ao binria a ter os canais analisados.
		 * @return o conjunto de canais do LHS da ao binria unido
		 *  com o conjunto de canais do RHS da ao binria.
		 */
		private HashSet<ZName> unirCanais(Action2 arg) {
			HashSet<ZName> result = arg.getLeftAction().accept(this);
			result.addAll(arg.getRightAction().accept(this));
			return result;
		}
	}


}
