/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.opsdischarge.syntacticfunctions;

import java.util.HashSet;
import java.util.List;

import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.util.UnsupportedAstClassException;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.circus.ast.ActionD;
import net.sourceforge.czt.circus.ast.ActionPara;
import net.sourceforge.czt.circus.ast.BasicProcess;
import net.sourceforge.czt.circus.ast.CallAction;
import net.sourceforge.czt.circus.ast.CircusAction;
import net.sourceforge.czt.circus.ast.CircusProcess;
import net.sourceforge.czt.circus.ast.IntChoiceActionIte;
import net.sourceforge.czt.circus.ast.LetVarAction;
import net.sourceforge.czt.circus.ast.ParallelActionIte;
import net.sourceforge.czt.circus.ast.PrefixingAction;
import net.sourceforge.czt.circus.ast.Process1;
import net.sourceforge.czt.circus.ast.Process2;
import net.sourceforge.czt.circus.ast.ProcessD;
import net.sourceforge.czt.circus.ast.SchExprAction;
import net.sourceforge.czt.circus.ast.SubstitutionAction;
import net.sourceforge.czt.circus.ast.VarDeclCommand;
import net.sourceforge.czt.circus.visitor.ActionDVisitor;
import net.sourceforge.czt.circus.visitor.BasicProcessVisitor;
import net.sourceforge.czt.circus.visitor.CircusProcessVisitor;
import net.sourceforge.czt.circus.visitor.LetVarActionVisitor;
import net.sourceforge.czt.circus.visitor.PrefixingActionVisitor;
import net.sourceforge.czt.circus.visitor.Process1Visitor;
import net.sourceforge.czt.circus.visitor.Process2Visitor;
import net.sourceforge.czt.circus.visitor.ProcessDVisitor;
import net.sourceforge.czt.circus.visitor.SubstitutionActionVisitor;
import net.sourceforge.czt.circus.visitor.VarDeclCommandVisitor;
import net.sourceforge.czt.z.ast.AxPara;
import net.sourceforge.czt.z.ast.ConstDecl;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.NewOldPair;
import net.sourceforge.czt.z.ast.Para;
import net.sourceforge.czt.z.ast.SchExpr;
import net.sourceforge.czt.z.ast.ZDeclList;
import net.sourceforge.czt.z.ast.ZName;
import net.sourceforge.czt.z.ast.ZParaList;
import net.sourceforge.czt.z.ast.ZSchText;
import circusRefine.core.crules.utils.NormalizationApplicationException;
import circusRefine.core.crules.utils.NormalizeApplier;
import circusRefine.core.opsdischarge.SyntacticFunctionsUtils;
import circusRefine.core.util.ChildrenTermExtractor;
import circusRefine.core.util.ParaNameGetter;

/**
 * <p>Classe respons�vel pela aplica��o da fun��o 
 * FV_P: CircusProcess \> \power Name.</p>
 * 
 * <p>A fun��o FV_P recebe um BasicProcess como par�metro e retorna 
 * um conjunto de nomes definido da seguinte forma:<br>
 * <br>
 * - Todos as vari�veis declaradas no estado s�o vari�veis livres do 
 * processo<br>
 * - Todos os nomes de par�grafos s�o vari�veis livres do processo<br>
 * - Todas as vari�veis declaradas na a��o principal do processo s�o
 * vari�veis livres do processo
 * <br>
 * </p>
 * 
 * @author Cristiano Castro
 */
public class FreeVariablesProcessApplier extends 
SyntacticFunctionApplier<CircusProcess, Expr> {

	/**
	 * <p>Aplica a função FV_p recebendo o resultado no formato aceito
	 * pela árvore sintática de circus</p>
	 * 
	 * @param args o processo argumento da fun��o FV
	 * @return o conjunto de nomes no formato da AST de Circus que 
	 *  representam as vari�veis livres do processo
	 */
	public Expr apply(CircusProcess args) throws CannotEvaluateException {
		HashSet<ZName> conjunto = this.aplicarFuncao(args);
		return this.transformarRepresentacaoConjuntoNomes(conjunto);
	}

	/**
	 * Aplica a fun��o FV_p tal como � definida na descri��o da 
	 * classe. O conjunto de nomes � retornado no estilo JAVA.
	 * 
	 * @param arg o processo argumento da fun��o FV
	 * @return o conjunto de nomes resultante da aplica��o de FV ao
	 *  processo argumento
	 */
	public HashSet<ZName> aplicarFuncao(CircusProcess arg) 
	throws CannotEvaluateException {
		HashSet<ZName> result = new HashSet<ZName>();
		try {

			/* O processo � avaliado */
			FreeVariablesProcessVisitor visitor = 
				new FreeVariablesProcessVisitor();
			result = arg.accept(visitor);
		} catch (CannotEvaluateRunTimeException e) {

			/* N�o p�de avaliar o processo*/
			throw new CannotEvaluateException(e);
		}

		/* O conjunto de vari�veis livres com seus termos normalizados */
		return result;
	}

	/**
	 * Captura as vari�veis declaradas em uma a��o
	 * 
	 * @param acao a a��o na qual a fun��o ser� aplicada
	 * @return o conjunto de vari�veis declaradas na a��o
	 */
	protected HashSet<ZName> variaveisDeclaradas(CircusAction acao) {
		DeclaredNamesGetter visitor = new DeclaredNamesGetter();
		return acao.accept(visitor);
	}

	/**
	 * <p>Classe que efetivamente aplica a fun��o FV atrav�s da 
	 * visita��o dos componentes do processo</p>
	 * 
	 * @author Cristiano Castro
	 */
	private class FreeVariablesProcessVisitor implements 
	CircusProcessVisitor<HashSet<ZName>>, 
	BasicProcessVisitor<HashSet<ZName>>, Process2Visitor<HashSet<ZName>>,
	Process1Visitor<HashSet<ZName>>, ProcessDVisitor<HashSet<ZName>> {

		/**
		 * Visita um processo gen�rico, que n�o pode ser avaliado.
		 * 
		 * @param arg0 o processo gen�rico a ser avaliado
		 * @throws CannotEvaluateRunTimeException pois um processo 
		 *  gen�rico n�o pode ser avaliado
		 */
		public HashSet<ZName> visitCircusProcess(CircusProcess arg0) {
			throw new CannotEvaluateRunTimeException("N�o pode avaliar o " +
					"processo gen�rico " + arg0);
		}

		/**
		 * As vari�veis livres do processo interno s�o as vari�veis 
		 * livres do processo interno
		 * 
		 * @param arg0 o processo un�rio a ser visitado
		 * @return o conjunto de vari�veis livres do processo interno
		 */
		public HashSet<ZName> visitProcess1(Process1 arg0) {
			return arg0.getCircusProcess().accept(this);
		}

		/**
		 * Visita um processo bin�rio. Captuta as vari�veis livres dos
		 * dois processos que comp�em o processo bin�rio
		 * 
		 * @param arg0 o processo bin�rio a ser visitado
		 * @return a uni�o das vari�veis dos dois processos
		 */
		public HashSet<ZName> visitProcess2(Process2 arg0) {
			HashSet<ZName> result = arg0.getLeftProcess().accept(this);
			result.addAll(arg0.getRightProcess().accept(this));
			return result;
		}

		/**
		 * Visita um processo com uma lista de declara��es. As 
		 * vari�veis declaradas s�o removidas das vari�veis livres do 
		 * processo
		 * 
		 * @param arg0 o processo com a lista de declara��es a ser 
		 * visitado
		 * @return o conjunto de vari�veis livres do processo interno
		 *  subtra�do das vari�veis declaradas na lista de declara��es
		 * @throws CannotEvaluateRunTimeException caso a lista de 
		 *  declara��oes do processo n�o seja uma {@link ZDeclList}
		 */
		public HashSet<ZName> visitProcessD(ProcessD arg0) {
			try {
				HashSet<ZName> result = arg0.getCircusProcess().accept(this);
				ZDeclList listaDecl = arg0.getZDeclList();
				HashSet<ZName> decl = 
					SyntacticFunctionsUtils.variaveisDeclaradas(listaDecl);
				result.removeAll(decl);
				return result;
			} catch (UnsupportedAstClassException e) {
				throw new CannotEvaluateRunTimeException(e);
			}
		}

		/**
		 * Visita um processo b�sico, captura os nomes de vari�veis 
		 * livres do processo segundo a defini��o da classe
		 * 
		 * @param arg0 o processo b�sico a ser visitado
		 * @return o conjunto de nomes representando as vari�veis 
		 * livres do processo
		 */
		public HashSet<ZName> visitBasicProcess(BasicProcess arg0) {
			HashSet<ZName> result;

			/* As vari�veis do estado do esquema s�o capturadas */
			Para estado = arg0.getStatePara();
			result = capturarNomesEstado(estado);

			/* Os nomes de par�grafos s�o capturados */
			try {
				result.addAll(this.pegarNomesParagrafos(arg0.getZParaList()));
			} catch (UnsupportedAstClassException e) {

				/* 
				 * Caso o conjunto de par�grafos do processo n�o for 
				 * um ZParaList 
				 */
				throw new CannotEvaluateRunTimeException(e);
			}

			/* adiciona as vari�veis declaradas na a��o principal */
			result.addAll(variaveisDeclaradas(arg0.getMainAction()));

			return result;
		}

		/**
		 * Captura as vari�veis declaradas em um estado par�grafo do
		 * processo, dado o estado desse processo
		 * 
		 * @param statePara o par�grafo de estado do processo
		 * @return o conjunto de nomes declarados no par�grafo de estado
		 */
		private HashSet<ZName> capturarNomesEstado(Para statePara) {

			/* Pega a express�o do estado */
			if (statePara instanceof AxPara) {
				AxPara stPara = (AxPara) statePara;
				ZSchText para = stPara.getZSchText();
				ZDeclList listaDecl = para.getZDeclList();
				ConstDecl declaracaoEsquema = (ConstDecl) listaDecl.get(0);
				Expr state = declaracaoEsquema.getExpr(); 

				/* Normaliza a express�o */
				SchExpr normalState;
				try {
					normalState = 
						NormalizeApplier.getInstance().applyRewrite(state);
				} catch (NormalizationApplicationException e) {
					throw new CannotEvaluateRunTimeException(e);
				}
				ZSchText esqSt = normalState.getZSchText();

				/* captura as vari�veis declaradas */
				return SyntacticFunctionsUtils.variaveisDeclaradas(esqSt);
			}
			else if (statePara instanceof ActionPara) {
				
				ActionPara actionP = (ActionPara)statePara;
				if (actionP.getCircusAction() instanceof SchExprAction) {
					SchExprAction schema = (SchExprAction) actionP.getCircusAction();
					
					/* Normaliza a express�o */
					SchExpr normalState;
					try {
						normalState = 
							NormalizeApplier.getInstance().applyRewrite(schema.getExpr());
					} catch (NormalizationApplicationException e) {
						throw new CannotEvaluateRunTimeException(e);
					}
					ZSchText esqSt = normalState.getZSchText();

					/* captura as vari�veis declaradas */
					return SyntacticFunctionsUtils.variaveisDeclaradas(esqSt);
					
				}
				else {
					//TODO se lembrar de depois tratar casos A \land A. Estado a partir de conjunções de schemas.
					return null;
				}

				//TODO se lembrar de depois tratar casos A \land A. Estado a partir de conjunções de schemas.
			} else {
				System.err.println("ERROR FreeVariablesProcessApplier");
				return null;
			}




		}

		/**
		 * Captura os nomes dos par�grafos para montar o conjunto de 
		 * vari�veis livres de um processo
		 * 
		 * @param listaParagrafos a lista de par�grafos de um 
		 *  {@link BasicProcess}
		 * @return o conjunto de nomes dos par�grafos do processo
		 */
		private HashSet<ZName> pegarNomesParagrafos(ZParaList listaParagrafos) {
			HashSet<ZName> result = new HashSet<ZName>();
			try {
				for (Para paragrafo : listaParagrafos) {
					result.add(ParaNameGetter.getNameList(paragrafo));
				}
			} catch (UnsupportedAstClassException e) {
				throw new CannotEvaluateRunTimeException(e);
			} catch (ClassCastException e) {
				throw new CannotEvaluateRunTimeException(e);
			}
			return result;
		}



	}

	/**
	 * Classe utilizada para se obter as vari�veis declaradas em uma
	 * a��o de Circus. Utilizada no c�lculo de vari�veis livres do 
	 * processo para se obter as vari�veis declaradas na a��o 
	 * principal do processo
	 * 
	 * @author Cristiano Castro
	 */
	private class DeclaredNamesGetter 
	implements ActionDVisitor<HashSet<ZName>>, 
	LetVarActionVisitor<HashSet<ZName>>, TermVisitor<HashSet<ZName>>, 
	PrefixingActionVisitor<HashSet<ZName>>, 
	SubstitutionActionVisitor<HashSet<ZName>>, 
	VarDeclCommandVisitor<HashSet<ZName>> {

		/**
		 * Visita um termo gen�rico. Retorna a uni�o das vari�veis
		 * declaradas dos filhos
		 * 
		 * @param arg0 o termo gen�rico a ser visitado
		 * @return o conjunto de nomes de vari�veis declaradas nos 
		 *  filhos do termo
		 */
		public HashSet<ZName> visitTerm(Term arg0) {
			List<Term> filhos = ChildrenTermExtractor.extrairFilhos(arg0);
			HashSet<ZName> result = new HashSet<ZName>();

			/* Percorre os filhos */
			for (Term filho : filhos) {
				result.addAll(filho.accept(this));
			}
			return result;
		}

		/**
		 * Inclui as vari�veis declaradas em a��es como 
		 * {@link IntChoiceActionIte} ou {@link ParallelActionIte}
		 * 
		 * @param arg0 a a��o a ser visitada
		 * @return o conjunto de vari�veis declaradas na lista de 
		 *  declara��es unido com as vari�veis declaradas na a��o 
		 *  interna
		 */
		public HashSet<ZName> visitActionD(ActionD arg0) {
			try {

				/* Percorre a �rvore */
				HashSet<ZName> result = arg0.getCircusAction().accept(this);

				/* Adiciona as vari�veis declaradas */
				ZDeclList lista = arg0.getZDeclList();
				HashSet<ZName> declaradas = 
					SyntacticFunctionsUtils.variaveisDeclaradas(lista);
				result.addAll(declaradas);

				return result;
			} catch (UnsupportedAstClassException e) {

				/* Caso a ActionD n�o possua uma inst�ncia de ZDeclList */
				throw new CannotEvaluateRunTimeException(e);
			}
		}

		/**
		 * Adiciona as declara��es de vari�veis
		 * 
		 * @param arg0 a a��o com declara��o de vari�veis
		 * @return as declara��es de vari�veis da lista de declara��es
		 *  unido com as vari�vies declaradas na a��o interna
		 */
		public HashSet<ZName> visitLetVarAction(LetVarAction arg0) {
			HashSet<ZName> result = arg0.getCircusAction().accept(this);

			/* Adiciona as vari�veis declaradas */
			try {
				ZDeclList lista = arg0.getZDeclList();
				HashSet<ZName> vd = 
					SyntacticFunctionsUtils.variaveisDeclaradas(lista);
				result.addAll(vd);
				return result;
			} catch (UnsupportedAstClassException e) {

				/* 
				 * Caso a lista de vari�veis declaradas n�o seja uma 
				 * ZDeclList 
				 */
				throw new CannotEvaluateRunTimeException(e);
			}
		}

		/**
		 * Adiciona as vari�veis declaradas para um input do canal. 
		 * Exemplo: c?x
		 * 
		 * @param arg0 a a��o prefixada por uma comunica��o a ser 
		 *  visitada
		 * @return o conjunto de nomes declarados no canal unido com a
		 *  a��o interna. 
		 */
		public HashSet<ZName> visitPrefixingAction(PrefixingAction arg0) {
			try {
				InputVarsApplier ivApplier = new InputVarsApplier();
				HashSet<ZName> result = arg0.getCircusAction().accept(this);
				result.addAll(ivApplier.apply(arg0.getCommunication()));
				return result;
			} catch (CannotEvaluateException e) {
				throw new CannotEvaluateRunTimeException(e);
			}
		}

		/**
		 * Visita uma a��o para substitui��o de nomes. Troca os nomes
		 * antigos pelos novos no conjunto de vari�veis declaradas na 
		 * a��o
		 * 
		 * @param arg0 a a��o com substitui��o de nomes
		 * @return o conjunto com os nomes trocados (caso necess�rio)
		 */
		public HashSet<ZName> visitSubstitutionAction(SubstitutionAction arg0) {
			HashSet<ZName> result = arg0.getCircusAction().accept(this);

			try {
				for (NewOldPair renomeacao : arg0.getZRenameList()) {
					if (result.remove(renomeacao.getZRefName())) {
						result.add(renomeacao.getZDeclName());
					}
				}
			} catch (UnsupportedAstClassException e) {
				throw new CannotEvaluateRunTimeException(e);
			}

			return result;
		}


		/**
		 * Visita uma comando de declara��o de vari�vel no estilo
		 * Carrol Morgan. Ao conjunto de vari�veis declaradas na a��o
		 * s�o adicionadas as vari�veis declaradas
		 * 
		 * @param arg0 o comando de declara��o de vari�vel 
		 * @return o conjunto de vari�veis usadas na a��o menos as 
		 *  vari�veis declaradas
		 * @throws CannotEvaluateRunTimeException caso a lista de 
		 *  nomes da declara��o de vari�veis n�o for uma 
		 *  {@link ZDeclList} 
		 */
		public HashSet<ZName> visitVarDeclCommand(VarDeclCommand arg0) {
			HashSet<ZName> result = arg0.getCircusAction().accept(this);
			try {

				/* Vari�veis declaradas */
				result.addAll(SyntacticFunctionsUtils.variaveisDeclaradas(
						arg0.getZDeclList()));
			} catch (UnsupportedAstClassException e) {
				throw new CannotEvaluateRunTimeException(e);
			}

			return result;
		}

	}

}
