/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.opsdischarge.syntacticfunctions;

import java.util.HashSet;
import java.util.Set;

import net.sourceforge.czt.base.util.UnsupportedAstClassException;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.NextStroke;
import net.sourceforge.czt.z.ast.Pred;
import net.sourceforge.czt.z.ast.Stroke;
import net.sourceforge.czt.z.ast.ZName;

/**
 * Classe responsvel pela aplicao da Funo DFV_Expr, ou <i>dashed 
 * free-variables</i>.
 * 
 * @author Cristiano Castro
 */
public class DashedFreeVariablesApplier extends 
SyntacticFunctionApplier<Expr, Expr>{

	/** Aplicados da funo FV_Expr */
	private FreeVariablesApplier fvApplier;
	
	/**
	 * @return the fvApplier
	 */
	private FreeVariablesApplier getFvApplier() {
		return fvApplier;
	}

	/**
	 * @param fvApplier the fvApplier to set
	 */
	private void setFvApplier(FreeVariablesApplier fvApplier) {
		this.fvApplier = fvApplier;
	}

	/**
	 * Inicia a classe informando termos que j foram expandidos pelo
	 * passo anterior
	 *   
	 * @param termosExpandidos o conjunto de nomes que foram 
	 *  expandidos
	 */ 
	public DashedFreeVariablesApplier(Set<String> termosExpandidos) {
		this.setFvApplier(new FreeVariablesApplier(termosExpandidos));
	}

	/**
	 * Aplica a funo DFV_Expr a uma expresso
	 * 
	 * @param args a expresso argumento da 
	 */
	public Expr apply(Expr args) throws CannotEvaluateException {
		return this.transformarRepresentacaoConjuntoNomes(this.getDFV(args));
	}

	/**
	 * Aplica a funo Dashed free variables a uma expresso
	 * 
	 * @param args a expresso a ser avaliada
	 * @return o conjunto de nomes com as variveis do tipo marcadas
	 * @throws CannotEvaluateException caso a expresso no possa ser
	 *  avaliada
	 */
	public HashSet<ZName> getDFV(Expr args) throws CannotEvaluateException {
		HashSet<ZName> fv = this.getFvApplier().getFV(args);
		HashSet<ZName> dfv = this.filtrarNomesMarcados(fv);
		return dfv;
	}

	/**
	 * Aplica a funo Dashed free variables a uma expresso
	 * 
	 * @param args a expresso a ser avaliada
	 * @return o conjunto de nomes com as variveis do tipo marcadas
	 * @throws CannotEvaluateException caso a expresso no possa ser
	 *  avaliada
	 */
	public HashSet<ZName> getDFV(Pred args) throws CannotEvaluateException {
		HashSet<ZName> fv = this.getFvApplier().getFV(args);
		HashSet<ZName> dfv = this.filtrarNomesMarcados(fv);
		return dfv;
	}

	/**
	 * Filtra os nomes marcados com {@link NextStroke} de um conjunto
	 * de nomes
	 * 
	 * @param aFiltrar o conjunto de nomes a ser filtrado
	 * @return o conjunto contendo somente os nomes que contm a marca 
	 *  de um {@link NextStroke}
	 * @throws UnsupportedAstClassException se alguns dos mtodos 
	 *  utilitrios no puderm ser utilizados
	 */
	protected HashSet<ZName> filtrarNomesMarcados(HashSet<ZName> aFiltrar) {
		HashSet<ZName> result = new HashSet<ZName>();

		/* Percorre a lista de nomes procurando os nomes com marcas */
		for (ZName nome : aFiltrar) {

			/* Percorre os strokes do nome e v se tem uma marcao */
			boolean adiciona = false;
			for (Stroke stk : nome.getZStrokeList()) {
				if (stk instanceof NextStroke) {

					/* Pode adicionar a lista de DFV_Expr */
					adiciona = true;
				}
			}

			if (adiciona) {
				result.add(nome);
			}
		}

		/* Retorna o conjunto contendo os nomes marcados */
		return result;

	}

}
