/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.opsdischarge.syntacticfunctions;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import net.sourceforge.czt.base.util.UnsupportedAstClassException;
import net.sourceforge.czt.z.ast.ConstDecl;
import net.sourceforge.czt.z.ast.Decl;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.InclDecl;
import net.sourceforge.czt.z.ast.Name;
import net.sourceforge.czt.z.ast.SchExpr;
import net.sourceforge.czt.z.ast.VarDecl;
import net.sourceforge.czt.z.ast.ZDeclList;
import net.sourceforge.czt.z.ast.ZName;
import net.sourceforge.czt.z.ast.ZSchText;
import net.sourceforge.czt.z.visitor.ConstDeclVisitor;
import net.sourceforge.czt.z.visitor.InclDeclVisitor;
import net.sourceforge.czt.z.visitor.SchExprVisitor;
import net.sourceforge.czt.z.visitor.VarDeclVisitor;
import net.sourceforge.czt.z.visitor.ZDeclListVisitor;
import net.sourceforge.czt.z.visitor.ZSchTextVisitor;
import circusRefine.core.crules.utils.NormalizationApplicationException;
import circusRefine.core.crules.utils.NormalizeApplier;

/**
 * Classe que aplica efetivamente a funo \alpha: 
 *  SchExpr -> \power Name
 * 
 * @author Cristiano Castro
 */
public class AlphaApplier extends SyntacticFunctionApplier<SchExpr, Expr> 
{

	/**
	 * Aplica a funo \alpha. Captura os nomes declarados do esquema
	 * e depois muda a representao para estar de acordo com a AST
	 * de circus
	 * 
	 * @param args o esquema a ser visitado
	 * @return o conjunto de nomes declarados no esquema
	 * @throws CannotEvaluateException caso a operao de capturar 
	 *  nomes de esquemas dispare alguma excesso
	 * @see #getDeclaredNames(SchExpr)
	 */
	public Expr apply(SchExpr args) throws CannotEvaluateException {
		Set<ZName> conjunto = this.getDeclaredNames(args);
		return this.transformarRepresentacaoConjuntoNomes(conjunto);
	}

	/**
	 * Mtodo utilizado para retornar os nomes declarados em um esquema
	 * 
	 * @param expr o {@link SchExpr} a ter suas variveis declaradas 
	 *  visitadas
	 * @return o conjunto dos nomes das variveis declaradas no 
	 *  esquema
	 * @throws se alguma excesso do tipo {@link ClassCastException}, 
	 * {@link UnsupportedAstClassException} ou 
	 * {@link CannotEvaluateRunTimeException} for capturada na 
	 *  operao de captura de variveis declaradas em esquemas.
	 * @see #apply(SchExpr)
	 */
	public Set<ZName> getDeclaredNames(SchExpr expr) 
	throws CannotEvaluateException {
		try {
			AlphaVisitor visitor = new AlphaVisitor();
			return expr.accept(visitor);
		} catch (ClassCastException e) {
			throw new CannotEvaluateException(e);
		} catch (UnsupportedAstClassException e) {
			throw new CannotEvaluateException(e);
		} catch (CannotEvaluateRunTimeException e) {
			throw new CannotEvaluateException(e);
		}
	}

	/**
	 * Classe utilizada para aplicar a funo 
	 * 
	 * @author crisgc
	 *
	 */
	private class AlphaVisitor implements SchExprVisitor<HashSet<ZName>>, 
	ZDeclListVisitor<HashSet<ZName>>, ConstDeclVisitor<HashSet<ZName>>, 
	VarDeclVisitor<HashSet<ZName>>, InclDeclVisitor<HashSet<ZName>>, 
	ZSchTextVisitor<HashSet<ZName>> {

		/**
		 * Visita um esquema, normaliza o esquema e depois coleta as 
		 * variveis declaradas
		 * 
		 * @param arg0 o schExpr a ser visitado para a aplicao de 
		 *  \alpha
		 * @return o conjutno de variveis declaradas no esquema
		 */
		public HashSet<ZName> visitSchExpr(SchExpr arg0) {
				SchExpr normalizado;
				try {
					normalizado = 
						NormalizeApplier.getInstance().applyRewrite(arg0);
				} catch (NormalizationApplicationException e) {
					throw new CannotEvaluateRunTimeException(e);
				}
				return normalizado.getSchText().accept(this);
		}
		
		/**
		 * Faz com que se visite a lista de declaraes do 
		 * {@link ZSchText}
		 * 
		 * @param arg0 o esquema a ser visitado
		 * @return o conjunto de nomes declarados no esquema
		 */
		public HashSet<ZName> visitZSchText(ZSchText arg0) {
			return arg0.getDeclList().accept(this);
		}
		
		/**
		 * Percorre a lista de declaraes de variveis.
		 * 
		 * @param arg0 a lista de declaraes de variveis
		 * @return os nomes declarados na lista de variveis
		 */
		public HashSet<ZName> visitZDeclList(ZDeclList arg0) {
			HashSet<ZName> result = new HashSet<ZName>();
			for (Decl decl : arg0) {
				result.addAll(decl.accept(this));
			}
			return result;
		}

		/**
		 * Captura o nome declarado em uma declarao constante
		 * 
		 * @param arg0 a declarao constante
		 * @return o nome da declarao constante
		 * @throws UnsupportedAstClassException se o nome do argumento
		 *  no for uma instncia de {@link ZName}
		 */
		public HashSet<ZName> visitConstDecl(ConstDecl arg0) {
			return new HashSet<ZName>(Arrays.asList(arg0.getZName()));
		}

		/**
		 * Uma declarao de incluso no possui variveis que possam
		 * ser tidas como declaradas em esquemas. Pois os termos 
		 * presentes em uma declarao de incluso j foram expandidos
		 * no passo anterior da resoluo de OPs
		 * 
		 * @param arg0 a declarao de incluso a ser visitada
		 * @return um conjunto de nomes vazio
		 */
		public HashSet<ZName> visitInclDecl(InclDecl arg0) {
			return new HashSet<ZName>();
		}

		/**
		 * Visita uma declarao de variveis. Inclui o nome das 
		 * variveis declaradas no conjunto de nomes declarados no
		 * esquema
		 * 
		 * @param arg0 a declarao de varivel
		 * @return o conjunto de nomes declarados na declarao de
		 *  varivel
		 */
		public HashSet<ZName> visitVarDecl(VarDecl arg0) {
			HashSet<ZName> result = new HashSet<ZName>();

			/* Insere os nomes da lista de declarao no conjunto de
			 * variveis declaradas */
			for (Name nome : arg0.getZNameList()) {

				/* 
				 * Inclui o nome no conjunto de declaraes de 
				 * variveis 
				 */
				result.add((ZName) nome);
			}

			return result;
		}

	}

}
