/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.opsdischarge.setoperators;

import net.sourceforge.czt.base.util.UnsupportedAstClassException;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.SetExpr;
import net.sourceforge.czt.z.ast.ZExprList;
import circusRefine.core.opsdischarge.NormalASTForComparisonGenerator;
import circusRefine.core.opsdischarge.syntacticfunctions.CannotEvaluateException;

/**
 * Aplicador do operador igualdade.
 * 
 * @author Cristiano Castro
 */
public class EqualityApplier {

	/**
	 * Verfica a igualdade de uma expresso com a expresso contida 
	 * em um conjunto unitrio. Esse mtodo est definido dessa forma
	 * para facilitar a aplicao de acordo com a AST de Circus que
	 * define a igualdade dessa forma
	 * 
	 * @param e1 a primeira expresso a ser comparada
	 * @param c1 o conjunto unitrio contendo a segunda expresso a 
	 *  ser comparada
	 * @return <code>true</code> caso as expresses sejam iguais. A 
	 *  comparao  feita utilizado do mtodo <i>equals</i> das 
	 *  expresses a serem comparadas
	 * @throws CannotEvaluateException caso a lista de expresses do
	 *  conjunto unitrio no for uma instncia de {@link ZExprList}
	 * @throws CannotEvaluateException caso a lista de expresses no
	 *  tenha um elemento. No caso o conjunto passado  invlido
	 */
	public Boolean apply(Expr e1, SetExpr c1) throws CannotEvaluateException {
		try {
			Expr e2 = c1.getZExprList().get(0);
			
			/* Normaliza os argumentos */
			e1 = (Expr) NormalASTForComparisonGenerator.gerarFormaNormal(e1);
			e2 = (Expr) NormalASTForComparisonGenerator.gerarFormaNormal(e2);
			
			Boolean result = e1.equals(e2);
			
			/* 
			 * Se os termos no forem estritamente iguais, ento no 
			 * h como avaliar se so diferentes sem ter acesso aos 
			 * valores que cada expresso representa. Portanto, se a
			 * comparao por igualdade de termos no deu certo, no
			 * h como saber se os termos so iguais ou no.
			 * 
			 * Por exemplo: x = x ==> true
			 * 				mas x = y ==> indefinido, no sei quem  
			 * 							  x nem y
			 */
			/*if (result != true) {
			 * TODO verificar a colocada desse comentario com cristiano
				throw new CannotEvaluateException();
			}*/
			
			return result;
		} catch (UnsupportedAstClassException e) {
			throw new CannotEvaluateException(e);
		} catch (IndexOutOfBoundsException e) {
			throw new CannotEvaluateException(e);
		}
	}
	
	/**
	 * Aplica o operador igualdade. Esse mtodo recebe um parmetro
	 * do operador igualdade e um conjunto unitrio contendo o outro
	 * parmetro da igualdade. Esse mtodo foi definido dessa forma
	 * para facilitar sua aplicao de acordo com a AST de Circus
	 * 
	 * @param e1 o primeiro elemento a ser comparado
	 * @param e2 o conjunto unitrio contendo o segundo elemento a ser
	 *  comparado
	 * @return <code>true</code> caso os dois elementos comparados 
	 *  sejam iguais segundo o mtodo <i>equals</i>
	 * @throws CannotEvaluateException caso a segunda expresso no 
	 *  for um conjunto unitrio
	 * @see #apply(Expr, SetExpr)
	 */
	public Boolean apply(Expr e1, Expr e2) throws CannotEvaluateException {
		try {
			return this.apply(e1, (SetExpr) e2);
		} catch (ClassCastException e) {
			e.printStackTrace();
			throw new CannotEvaluateException(e);
		}
	}
	
}
