/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.opsdischarge;

import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.util.UnsupportedAstClassException;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.circus.ast.BasicChannelSetExpr;
import net.sourceforge.czt.circuspatt.ast.CircusPatternFactory;
import net.sourceforge.czt.circuspatt.impl.CircusPatternFactoryImpl;
import net.sourceforge.czt.session.Markup;
import net.sourceforge.czt.z.ast.AndPred;
import net.sourceforge.czt.z.ast.FalsePred;
import net.sourceforge.czt.z.ast.IffPred;
import net.sourceforge.czt.z.ast.ImpliesPred;
import net.sourceforge.czt.z.ast.MemPred;
import net.sourceforge.czt.z.ast.NegPred;
import net.sourceforge.czt.z.ast.OrPred;
import net.sourceforge.czt.z.ast.Pred;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.SetExpr;
import net.sourceforge.czt.z.ast.TruePred;
import net.sourceforge.czt.z.ast.ZName;
import net.sourceforge.czt.z.visitor.AndPredVisitor;
import net.sourceforge.czt.z.visitor.FalsePredVisitor;
import net.sourceforge.czt.z.visitor.IffPredVisitor;
import net.sourceforge.czt.z.visitor.ImpliesPredVisitor;
import net.sourceforge.czt.z.visitor.MemPredVisitor;
import net.sourceforge.czt.z.visitor.NegPredVisitor;
import net.sourceforge.czt.z.visitor.OrPredVisitor;
import net.sourceforge.czt.z.visitor.TruePredVisitor;
import circusRefine.core.opsdischarge.setoperators.EqualityApplier;
import circusRefine.core.opsdischarge.setoperators.MembershipOperatorApplier;
import circusRefine.core.opsdischarge.setoperators.SubseteqApplier;
import circusRefine.core.opsdischarge.syntacticfunctions.CannotEvaluateException;
import circusRefine.core.print.Printer;
import circusRefine.util.OPTipos;

/**
 * Classe que ir tentar solucionar uma obrigao de prova
 * 
 * @author Cristiano Castro
 */
public class Provador implements NegPredVisitor<Boolean>, 
AndPredVisitor<Boolean>, OrPredVisitor<Boolean>, ImpliesPredVisitor<Boolean>, 
IffPredVisitor<Boolean>, MemPredVisitor<Boolean>, TermVisitor<Boolean>, TruePredVisitor<Boolean>,
FalsePredVisitor<Boolean>{

	/**
	 * Tenta provar uma obrigao de prova.
	 * 
	 * @param op a OP a tentar ser provada
	 * @return o tipo da avaliao de acordo com o resultado do provador.
	 */
	public static OPTipos prove(Pred op) {
		OPTipos result;
		try {

			if (Provador.proveAux(op)) {

				/* Provador validou a OP */
				result = OPTipos.OP_CHECADA_TRUE;
			} else {

				/* Provador no validou a OP */
				result = OPTipos.OP_CHECADA_FALSE;
			}
		} catch (CannotProveException e) {
			/* No consegui provar */
			result = OPTipos.OP_CHECADA_NRECONHECIDA;
		}
		return result;
	}

	/**
	 * Tenta provar uma Obrigao de Prova. 
	 * 
	 * @param op a Obrigao de prova a ser testada
	 * @return <code>true</code> caso a OP seja vlida ou 
	 *  <code>false</code> caso contrrio
	 * @throws CannotProveException caso a obrigao de prova no possa 
	 *  ser analisada
	 */
	public static boolean proveAux(Pred op) throws CannotProveException {
		try {
			Provador visitor = new Provador();
			Boolean result = op.accept(visitor);
			return result;
		} catch (CannotProveRuntimeException e) {
			throw new CannotProveException(e);
		}
	}

	CircusPatternFactory factory = new CircusPatternFactoryImpl();

	/**
	 * Visita um termo genrico. Dispara uma excesso pois um termo
	 * genrico no pode ser avaliado
	 * 
	 * @param arg0 o termo a ser avaliado
	 * @throws CannotProveRuntimeException pois o termo no pode ser
	 *  avaliado
	 */
	public Boolean visitTerm(Term arg0) {
		throw new CannotProveRuntimeException("o termo " + arg0 + 
		" no pode ser avaliado");
	}

	/**
	 * Avalia uma negao
	 * 
	 * @param arg0 uma negao
	 * @return o resultado oposto ao avaliado
	 */
	public Boolean visitNegPred(NegPred arg0) {
		return !arg0.getPred().accept(this);
	}

	/**
	 * Avalia uma conjuno
	 * 
	 * @param arg0 a conjuno a ser analisada
	 * @return um booleano correspondendo ao valor da avaliao da 
	 *  conjuno
	 */
	public Boolean visitAndPred(AndPred arg0) {
		return arg0.getLeftPred().accept(this) && 
		arg0.getRightPred().accept(this);
	}

	/**
	 * Avalia uma disjuno
	 * 
	 * @param arg0 a disjuno a ser avaliada para testar a validade 
	 * do predicado
	 * @return a avaliao da disjuno
	 */
	public Boolean visitOrPred(OrPred arg0) {
		return arg0.getLeftPred().accept(this) || 
		arg0.getRightPred().accept(this);
	}

	/**
	 * Visita uma implicao
	 * 
	 * @param arg0 a implicao a ser testada
	 * @return a avaliao da implicao
	 */
	public Boolean visitImpliesPred(ImpliesPred arg0) {
		return (!arg0.getLeftPred().accept(this) || 
		arg0.getRightPred().accept(this));
	}

	/**
	 * Visita um se e somente se.
	 * 
	 * @param o predicado
	 * @return a avaliao da condio que o predicado representa
	 */
	public Boolean visitIffPred(IffPred arg0) {
		return arg0.getLeftPred().accept(this).equals(
				arg0.getRightPred().accept(this));
	}

	/**
	 * Visita um predicado tipo pertinncia. ltimo estgo para 
	 * resolver obrigaes de prova
	 * 
	 * @param arg0 o predicado tipo pertinncia a ser analisado
	 * @return a resposta ao predicado
	 */
	public Boolean visitMemPred(MemPred arg0) {
		Boolean result;
		try {
			if (!arg0.getMixfix()) {

				/* Pertinncia!! */
				MembershipOperatorApplier applier = 
					new MembershipOperatorApplier();
				
				result = applier.apply(arg0.getLeftExpr(), arg0.getRightExpr());	

			} else if (arg0.getRightExpr() instanceof SetExpr) {

				/* Igualdade!! */
				EqualityApplier applier = new EqualityApplier();
				result = applier.apply(arg0.getLeftExpr(), arg0.getRightExpr());
			} else {

				/* Outro operador!! */
				try {
					RefExpr refOperador = (RefExpr) arg0.getRightExpr();
					ZName nomeOperador = refOperador.getZName();
					String strOperador = 
						OPsDischargeUtils.getNomeReal(nomeOperador);

					if (strOperador.equals(OPsDischargeUtils.getNomeReal(
							OPsDischargeUtils.SUBSETEQ))) {

						SubseteqApplier applier = new SubseteqApplier();
						result = applier.apply(arg0.getLeftExpr());
					} else {
						throw new CannotProveRuntimeException("A funo " + 
								strOperador + " no pode ser avaliada");
					}

				} catch (ClassCastException e) {
					throw new CannotProveRuntimeException(e);
				} catch (UnsupportedAstClassException e) {
					throw new CannotProveRuntimeException(e);
				}
			}
		} catch (CannotEvaluateException e) {
			throw new CannotProveRuntimeException(e);
		}
		return result;
	}

	public Boolean visitTruePred(TruePred arg0) {
		return true;
	}

	public Boolean visitFalsePred(FalsePred arg0) {
		return false;
	}
}
