/*
 * Projeto: Circus Refine
 */
package circusRefine.core.opsdischarge;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;

import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.util.UnsupportedAstClassException;
import net.sourceforge.czt.circuspatt.ast.CircusPatternFactory;
import net.sourceforge.czt.circuspatt.impl.CircusPatternFactoryImpl;
import net.sourceforge.czt.session.Markup;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.Pred;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.SetExpr;
import net.sourceforge.czt.z.ast.Spec;
import net.sourceforge.czt.z.ast.Stroke;
import net.sourceforge.czt.z.ast.ZExprList;
import net.sourceforge.czt.z.ast.ZName;
import net.sourceforge.czt.z.ast.ZStrokeList;
import net.sourceforge.czt.z.util.ZString;
import net.sourceforge.czt.zpatt.ast.Sequent;
import circusRefine.core.InternalManager;
import circusRefine.core.print.Printer;
import circusRefine.util.Pair;

/**
 * Classe utilit�ria do pacote para tratar as Obriga��es de Prova
 * 
 * @author Cristiano Castro
 */
public final class OPsDischargeUtils {

	/** Nome da fun��o FV_Expr */
	public final static String FV_Expr = "FV-E" + ZString.ARG_TOK;

	/** Nome da fun��o DFV_Expr */
	public final static String DFV_Expr = "DFV-E" + ZString.ARG_TOK;

	/** Nome da fun��o UDFV */
	public final static String UDFV_Expr = "UDFV-E" + ZString.ARG_TOK;

	/** Nome da fun��o wrtV */
	public final static String WRT_V = "wrtV" + ZString.ARG_TOK;
	
	/** Nome da fun��o usedV */
	public final static String USED_V = "usedV" + ZString.ARG_TOK;

	/** Nome da fun��o usedV */
	public final static String FV_Action = "FV-A" + ZString.ARG_TOK;
	
	/** Fun��o FV_P */
	public final static String FV_Process = "FV-P" + ZString.ARG_TOK;

	/** Nome da fun��o inputVars */
	public final static String INPUT_VARS = "inputVars" + ZString.ARG_TOK;

	/** Nome da fun��o \alpha */
	public final static String ALPHA = "\\alpha" + ZString.ARG_TOK;

	/** Nome da fun��o usedC */
	public final static String USED_C = "usedC" + ZString.ARG_TOK;

	/** Nome da fun��o initials */
	public final static String INITIALS = "initials" + ZString.ARG_TOK;

	/** Nome da fun��o pre */
	public final static String PRE = "pre" + ZString.ARG_TOK;

	/** Nome da fun��o State */
	public final static String STATE = "State" + ZString.ARG_TOK;

	/** Nome da fun��o \cup */
	public final static String CUP = ZString.ARG_TOK + ZString.CUP + 
	ZString.ARG_TOK;

	/** Nome da fun��o \cap */
	public final static String CAP = ZString.ARG_TOK + ZString.CAP + 
	ZString.ARG_TOK;

	/** Nome da fun��o _ .. _ */
	public final static String UPTO = ZString.ARG_TOK + ".." + 
	ZString.ARG_TOK;

	/** Nome da fun��o _ \subseteq _ */
	public final static String SUBSETEQ = ZString.ARG_TOK + ZString.SUBSETEQ + 
	ZString.ARG_TOK;
	
	/** Fun��o <i>divergence-free</i> */
	public final static String DIVERGENCE_FREE = "divergence-free" + 
	ZString.ARG_TOK;
	
	/** Fun��o <i>deterministic</i> */
	public final static String DETERMINISTIC = "deterministic" + 
	ZString.ARG_TOK;

	/**
	 * A partir de nomes de fun��o onde � especificado a posi��o dos
	 * argumentos e os operadores, retorna a String que especifica o
	 * operador em si.
	 * 
	 * @param nomeOperador o ZName com o nome do operador, do tipo
	 *  "_ > _".
	 * @return a {@link String} com o operador em si, no caso ">"
	 * @see #getNomeReal(String)
	 */
	public static String getNomeReal(ZName nomeOperador) {
		return OPsDischargeUtils.getNomeReal(nomeOperador.getWord());
	}

	/**
	 * A partir de nomes de fun��o onde � especificado a posi��o dos
	 * argumentos e os operadores, retorna a String que especifica o
	 * operador em si.
	 * 
	 * @param nomeOperador a String com o nome do operador, do tipo
	 *  "_ > _".
	 * @return a {@link String} com o operador em si, no caso ">"
	 * @see #getNomeReal(ZName)
	 */
	public static String getNomeReal(String nomeOperador) {
		StringTokenizer tokenizer = new StringTokenizer(nomeOperador, 
				ZString.ARG_TOK + " \t\n\r\f");
		return tokenizer.nextToken();
	}

	/**
	 * M�todo para retornar os nomes das fun��es trat�veis pelo
	 * gerenciador de Obriga��es de Prova do CRefine.
	 * 
	 * Exs.: 
	 * \subseteq --> Est� contido ou � igual
	 * \subset --> 	Est� contido
	 * 
	 * @return a lista com os operadores trat�veis
	 */
	public static List<String> operadoresTrataveis() {
		LinkedList<String> result = new LinkedList<String>();

		result.add(ZString.SUBSETEQ);
		result.add(ZString.SUBSET);

		return result;
	}

	/**
	 * <p>M�todo para retornar nomes de fun��es que recebem express�es
	 * como argumentos.</p>
	 * 
	 * <p>Ex.:
	 * FV_Expr: Expr --> \power Name (Free-variables of an expression)
	 * DFV_Expr: Expr --> \power Name (dashed free-variables of an 
	 * 	expression)
	 * UDFV: Expr --> \power Name (undashed free-variables of an 
	 * 	expression)
	 * </p>
	 * 
	 * @return a lista com as fun��es trat�veis pelo 
	 */
	public static List<String> funcoesDeExpressoes() {
		LinkedList<String> result = new LinkedList<String>();

		result.add(OPsDischargeUtils.FV_Expr);
		result.add(OPsDischargeUtils.DFV_Expr);
		result.add(OPsDischargeUtils.UDFV_Expr);

		return result;
	}

	/**
	 * <p>Cria a lista de fun��es que recebem a��es como 
	 * argumentos.</p>
	 * 
	 * <p>
	 * Ex.:
	 * wrtV: Action --> \power Name (written variables of an action) 
	 * usedV: Action --> \power Name (used variables of an action)
	 * usedC: Action --> \power Name (used channels of an action)
	 * initials: Action --> \power Name (initials channels of an 
	 * 	action)</p> 
	 * 
	 * @return
	 */
	public static List<String> funcoesDeAcoes() {
		LinkedList<String> result = new LinkedList<String>();

		result.add(OPsDischargeUtils.WRT_V);
		result.add(OPsDischargeUtils.FV_Action);
		result.add(OPsDischargeUtils.USED_C);
		result.add(OPsDischargeUtils.INITIALS);

		return result;
	}

	/**
	 * <p>Cria as fun��es que recebem uma Communication como 
	 * par�metro</p>
	 * 
	 * <p>
	 * Ex: inputVars: Communication --> seq Name
	 * </p>
	 * 
	 * @return a lista de fun��es
	 */
	public static List<String> funcoesDeComunicacoes() {
		LinkedList<String> result = new LinkedList<String>();

		result.add(OPsDischargeUtils.INPUT_VARS);

		return result;
	}

	/**
	 * <p>Cria as fun��es que recebem um Schema como 
	 * par�metro</p>
	 * 
	 * <p>Ex.: \alpha: SchExpr --> \power Name</p>
	 * 
	 * @return a lista de fun��es
	 */
	public static List<String> funcoesDeEsquemas() {
		List<String> result = new LinkedList<String>();

		result.add(OPsDischargeUtils.ALPHA);
		result.add(OPsDischargeUtils.PRE);

		return result;
	}

	/**
	 * <p>Cria a fun��o que recebe um processo como par�metro</p>
	 * 
	 * <p>Ex.: State: CircusProcess --> SchExpr</p>
	 * 
	 * @return a lista de fun��es
	 */
	public static List<String> funcoesDeProcessos() {
		LinkedList<String> result = new LinkedList<String>();

		result.add(OPsDischargeUtils.STATE);
		result.add(OPsDischargeUtils.FV_Process);

		return result;
	}

	/**
	 * <p>Cria as fun��es que recebem dois n�meros como argumentos</p>
	 * 
	 * <p>Ex.: \\upto --> \\nat x \nat --> set \nat</p>
	 * 
	 * @return 
	 */
	public static List<String> funcoesDeNumeros() {
		List<String> result = new LinkedList<String>();

		//result.add(OPsDischargeUtils.UPTO);

		return result;
	}

	/**
	 * <p>Cria fun��es bin�rias de conjuntos</p>
	 * 
	 * <p>Ex.: 
	 * \cap: SetExpr X SetExpr --> SetExpr
	 * \cup: SetExpr X SetExpr --> SetExpr</p>
	 * @return
	 */
	public static List<String> funcoesConjuntos2() {
		List<String> result = new LinkedList<String>();

		result.add(OPsDischargeUtils.CAP);
		result.add(OPsDischargeUtils.CUP);

		return result;
	}

	/**
	 * Remove os ZString.ARG_TOK de uma lista de nomes de fun��es,
	 * restando somente o identificador da fun��o em si
	 * 
	 * @param originais a lista de nomes de fun��es
	 * @return a lista dos nomes das fun��es sem o ARG_TOK
	 */
	public static List<String> funcoesSemArgTok(List<String> originais) {
		List<String> result = new LinkedList<String>();
		for (String nomeFun : originais) {
			result.add(OPsDischargeUtils.getNomeReal(nomeFun));
		}
		return result;
	}

	/**
	 * M�todo utilizado para retornar o n�mero de argumentos de uma 
	 * fun��o com base na quantidade de ARG_TOK
	 * 
	 * @param funcao o nome da fun��o a ser testada
	 * @return o n�mero de argumentos da fun��o
	 */
	public static int getNumberOfArguments(String funcao) {
		int result = 0;
		int k = 0;

		/* Procura por toda a ocorr�ncia */
		while (k != -1) {
			k = funcao.indexOf(ZString.ARG_TOK, k);
			if (k != -1) {

				/* Achou uma ocorr�ncia e Continua a busca */
				result++;
				k += ZString.ARG_TOK.length();
			} 
		}

		/* Caso n�o seja encontrado nenhum  */
		return (result == 0) ? 1 : result;

	}

	/**
	 * Retorna uma lista de todas as fun��es que podem ser tratadas 
	 * pelo manipulador de OPs do CRefine
	 *  
	 * @return uma lista de nomes das fun��es. Esses nomes podem 
	 *  conter ZString.ARG_TOK.
	 */
	public static List<String> retornarFuncoesTrataveis() {
		List<String> result = new LinkedList<String>();

		/* Insere todas as fun��es no resultado */
		result.addAll(OPsDischargeUtils.funcoesConjuntos2());
		result.addAll(OPsDischargeUtils.funcoesDeAcoes());
		result.addAll(OPsDischargeUtils.funcoesDeComunicacoes());
		result.addAll(OPsDischargeUtils.funcoesDeEsquemas());
		result.addAll(OPsDischargeUtils.funcoesDeExpressoes());
		result.addAll(OPsDischargeUtils.funcoesDeProcessos());
		result.addAll(OPsDischargeUtils.funcoesDeNumeros());

		return result;
	}

	/**
	 * Utilizado para retornar uma refer�ncia � alguma fun��o
	 * 
	 * @param strFuncao a fun��o que se deseja montar a refer�ncia
	 * @return a refer�ncia � fun��o
	 */
	public static RefExpr refFuncao(String strFuncao) {
		CircusPatternFactory factory = new CircusPatternFactoryImpl();
		ZName nomeFuncao = factory.createZName(strFuncao, 
				factory.createZStrokeList(), null);
		return factory.createRefExpr(nomeFuncao, factory.createZExprList(), 
				false, false);
	}

	/**
	 * Retorna a vers�osimplificada da obriga��o de prova. N�ose
	 * trata de um provador de teoremas (ainda) mas apenas algumas OPs
	 * simplificadas s�otratadas 
	 * 
	 * @param op o {@link Sequent} gerado por uma obriga��o de prova
	 * @param antigaAST a AST presente antes da aplica��o da lei
	 * @param alvoAplicacao o termo no qual a lei de refinamento que
	 *  gerou a obriga��o de prova foi aplicada
	 * @param progResultante o programa resultante depois da aplica��o
	 *  da lei
	 * @return o predicado simplificado, ou <code>null</code> caso 
	 *  essa simplifica��o n�oseja poss�vel
	 */
	public static Pred expandirOP(Sequent op, Term antigaAST, 
			Term alvoAplicacao, Term progResultante, InternalManager gerInterno) {
		Pred novaOP;

		/* OP � compat�vel */
		Pair<Term, Set<String>> resultadoExpansao = 
			ExpandedProofObligationGenerator.expandirOPs(op.getPred(), 
					antigaAST, alvoAplicacao, progResultante, gerInterno);

		Pred expandido = (Pred) resultadoExpansao.getFirst();
		

		try {

			/* Aplicar as fun��es sint�ticas */
			novaOP = (Pred) 
			SyntacticFunctionsApplier.aplicarFuncoesSintaticas(
					resultadoExpansao.getSecond(), expandido);
		} catch (CannotExpandException e) {
			/* N�o pode expandir as fun��es sint�ticas */
			novaOP = null;
			//e.printStackTrace();
		}

		
		return novaOP;
	}
	
	/**
	 * Cria uma refer�ncia ao conjunto vazio
	 * 
	 * @return o uma refer�ncia ao conjunto vazio
	 */
	public static RefExpr criarConjuntoVazio() {
		CircusPatternFactory factory = new CircusPatternFactoryImpl();
		ZName nome = factory.createZName(ZString.EMPTYSET, 
				factory.createZStrokeList(), null);
		return factory.createRefExpr(nome, factory.createZExprList(), false, 
				false);
	}
	
	/**
	 * Transforma se necess�rio a refer�ncia de um conjunto
	 * 
	 * @param expr o conjunto a ser analisado
	 * @return o pr�prio conjunto se n�ofor vazio ou uma referncia a 
	 *  emptyset se o conjunto for vazio
	 * @throws UnsupportedAstClassException caso a lista de 
	 *  express�es do conjunto n�ofor uma inst�ncia de 
	 *  {@link ZExprList}
	 */
	public static Expr criarReferenciaEmptysetCasoNecessario(SetExpr expr) {
		Expr result;
		if (expr.getZExprList().isEmpty()) {
			result = OPsDischargeUtils.criarConjuntoVazio();
		} else {
			result = expr;
		}
		return result;
	}
	
	/**
	 * Remove um determinado tipo de stroke de todos os nomes de um 
	 * conjunto de nomes
	 * 
	 * @param nomes o conjunto de nomes a terem um determinado tipo
	 *  de stroke removido
	 * @param tipoARemover o tipo de Stroke a ser removido
	 * @return  o conjunto de nomes com os str
	 * @see #removeStrokes(ZName, Class)
	 */
	public static HashSet<ZName> removeStrokesOfNames(Set<ZName> nomes, 
			Class<? extends Stroke> tipoARemover) {
		HashSet<ZName> result = new HashSet<ZName>();
		for (ZName nome : nomes) {
			OPsDischargeUtils.removeStrokes(nome, tipoARemover);
			result.add(nome);
		}
		return result;
	}
	
	/**
	 * Remove um determinado tipo de {@link Stroke} de um nome
	 * 
	 * @param nome o nome a ter os strokes removidos
	 * @param tipoARemover o tipo do stroke a remover
	 * @throws UnsupportedAstClassException caso a lista de strokes 
	 *  n�ofor uma {@link ZStrokeList}
	 */
	public static void removeStrokes(ZName nome, 
			Class<? extends Stroke> tipoARemover) {
		CircusPatternFactory factory = new CircusPatternFactoryImpl();
		ZStrokeList stkList = nome.getZStrokeList();
		ZStrokeList nova = factory.createZStrokeList();

		/* Monta a nova lista de strokes do nome */
		for (Stroke stk : stkList) {
			if (!tipoARemover.isInstance(stk)) {
				
				/* N�o� uma inst�ncia do tipo a remover */
				nova.add(stk);
			}
		}
		
		/* Muda a lista de strokes do nome */
		nome.setStrokeList(nova);
	}
	
}
