/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.crules.utils;

import java.util.Arrays;
import java.util.Collections;

import net.sourceforge.czt.base.util.UnsupportedAstClassException;
import net.sourceforge.czt.circus.ast.ActionPara;
import net.sourceforge.czt.circus.ast.BasicProcess;
import net.sourceforge.czt.circus.ast.CircusAction;
import net.sourceforge.czt.circus.ast.CircusStateAnn;
import net.sourceforge.czt.circus.ast.NameSetPara;
import net.sourceforge.czt.circus.ast.ProcessPara;
import net.sourceforge.czt.circus.ast.SchExprAction;
import net.sourceforge.czt.circus.util.CircusUtils;
import net.sourceforge.czt.circus.visitor.ActionParaVisitor;
import net.sourceforge.czt.circus.visitor.BasicProcessVisitor;
import net.sourceforge.czt.circus.visitor.NameSetParaVisitor;
import net.sourceforge.czt.circuspatt.ast.CircusPatternFactory;
import net.sourceforge.czt.circuspatt.impl.CircusPatternFactoryImpl;
import net.sourceforge.czt.z.ast.AndExpr;
import net.sourceforge.czt.z.ast.ApplExpr;
import net.sourceforge.czt.z.ast.AxPara;
import net.sourceforge.czt.z.ast.ConstDecl;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.MemPred;
import net.sourceforge.czt.z.ast.Para;
import net.sourceforge.czt.z.ast.Pred;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.SetExpr;
import net.sourceforge.czt.z.ast.Spec;
import net.sourceforge.czt.z.ast.TupleExpr;
import net.sourceforge.czt.z.ast.ZExprList;
import net.sourceforge.czt.z.ast.ZName;
import net.sourceforge.czt.z.ast.ZParaList;
import net.sourceforge.czt.z.visitor.AxParaVisitor;
import net.sourceforge.czt.z.visitor.ZParaListVisitor;
import net.sourceforge.czt.zpatt.ast.Sequent;
import circusRefine.core.InternalManager;
import circusRefine.core.astmodifiers.ActionArgumentAnn;
import circusRefine.core.opsdischarge.CannotProveException;
import circusRefine.core.opsdischarge.OPsDischargeUtils;
import circusRefine.core.opsdischarge.Provador;
import circusRefine.core.util.ClonerVisitor;
import circusRefine.core.util.ParaNameGetter;
import circusRefine.util.Pair;

/**
 * Visitor usado na aplica��o da lei C-146 Process splitting.
 * 
 * @author Cristiano Castro
 */
public class ProcessesMounter implements BasicProcessVisitor<Void>, 
ZParaListVisitor<Void>, ActionParaVisitor<Void>, AxParaVisitor<Void>, 
NameSetParaVisitor<Void> {

	/**
	 * Monta os processos da lei C-146
	 * 
	 * @param nomeP1 o nome do 1o. processo a ser montado
	 * @param nomeP2 o nome do 2o. processo a ser montado
	 * @param acaoPrincipal1 a a��o principal do 1o.processo a ser 
	 * 	montado
	 * @param acaoPrincipal2 a a��o principal do 2o. processo a ser 
	 * 	montado
	 * @param processoAnterior o processo original
	 * @return o dois par�grafos de processo montados
	 */
	public static Pair<Pair<ProcessPara, ProcessPara>, Pred> 
		montarProcessos(ZName nomeP1, ZName nomeP2, 
			CircusAction acaoPrincipal1, CircusAction acaoPrincipal2, 
			BasicProcess processoAnterior, Spec ast, InternalManager gerInt) {

		/* Efetua a montagem do processo */
		ProcessesMounter pm = new ProcessesMounter(nomeP1, nomeP2, 				
				acaoPrincipal1, acaoPrincipal2, ast, processoAnterior, gerInt);
		processoAnterior.accept(pm);
		return pm.montaProcesso();
		
	}

	/** F�brica */
	private CircusPatternFactory factory = 
		new CircusPatternFactoryImpl();

	/** Lista de par�grafos do processo 1 */
	private ZParaList process1;

	/** Lista de par�grafos do processo 2 */
	private ZParaList process2;

	/** nome do Processo1 */
	private ZName nomeProcesso1;

	/** Nome do processo2 */
	private ZName nomeProcesso2;
	
	/** Conjunto de vari�veis de estado do processo P1 */
	private AxPara stateP1;
	
	/** Vari�veis de estado do processo P2 */
	private AxPara stateP2;
	
	/** A��o principal do primeiro processo */
	private CircusAction acaoPrincipalProcesso1;

	/** A��o principal do segundo processo */
	private CircusAction acaoPrincipalProcesso2;
	
	/** AST no qual as defini��es podem ser buscadas */
	private Spec ast;
	
	/** Referencia para o gerenciador Interno*/
	private InternalManager gerInterno;
	
	/** Processo original a ser quebrado em 2 */
//	private BasicProcess original;

	/**
	 * Construtor que recebe como par�metro os nomes dos processos 
	 * montados e as a��es principais dos processos
	 * 
	 * @param nomeP1 o nome do primeiro processo a ser montado
	 * @param nomeP2 o nome do segundo processo a ser montado
	 * @param acaoPrincipal1 a a��o principal do primeiro processo
	 * @param acaoPrincipal2 a a��o principal do segundo processo
	 */
	public ProcessesMounter(ZName nomeP1, ZName nomeP2, 
			CircusAction acaoPrincipal1, CircusAction acaoPrincipal2, Spec ast, 
			BasicProcess original, InternalManager gerInt) {
		this.setNomeProcesso1(nomeP1);
		this.setNomeProcesso2(nomeP2);
		this.setProcess1(this.factory.createZParaList());
		this.setProcess2(this.factory.createZParaList());
		this.setAcaoPrincipalProcesso1(acaoPrincipal1);
		this.setAcaoPrincipalProcesso2(acaoPrincipal2);
		this.setAst(ast);
		gerInterno = gerInt;
//		this.setOriginal(original);
	}
	
	private ZParaList getProcess1() {
		return process1;
	}

	private void setProcess1(ZParaList process1) {
		this.process1 = process1;
	}

	private ZParaList getProcess2() {
		return process2;
	}

	private void setProcess2(ZParaList process2) {
		this.process2 = process2;
	}

	private CircusAction getAcaoPrincipalProcesso1() {
		return acaoPrincipalProcesso1;
	}

	private void setAcaoPrincipalProcesso1(CircusAction acaoPrincipalProcesso1) {
		this.acaoPrincipalProcesso1 = acaoPrincipalProcesso1;
	}

	private CircusAction getAcaoPrincipalProcesso2() {
		return acaoPrincipalProcesso2;
	}

	private void setAcaoPrincipalProcesso2(CircusAction acaoPrincipalProcesso2) {
		this.acaoPrincipalProcesso2 = acaoPrincipalProcesso2;
	}

	private ZName getNomeProcesso1() {
		return nomeProcesso1;
	}

	private void setNomeProcesso1(ZName nomeProcesso1) {
		this.nomeProcesso1 = nomeProcesso1;
	}

	private ZName getNomeProcesso2() {
		return nomeProcesso2;
	}

	private void setNomeProcesso2(ZName nomeProcesso2) {
		this.nomeProcesso2 = nomeProcesso2;
	}

	private AxPara getStateP1() {
		return stateP1;
	}

	private void setStateP1(AxPara stateP1) {
		this.stateP1 = stateP1;
	}

	private AxPara getStateP2() {
		return stateP2;
	}

	private void setStateP2(AxPara stateP2) {
		this.stateP2 = stateP2;
	}
	
	/**
	 * @return the ast
	 */
	private Spec getAst() {
		return ast;
	}

	/**
	 * @param ast the ast to set
	 */
	private void setAst(Spec ast) {
		this.ast = ast;
	}

	/**
	 * @return the original
	 */
//	private BasicProcess getOriginal() {
//		return original;
//	}

	/**
	 * @param original the original to set
	 */
//	private void setOriginal(BasicProcess original) {
//		this.original = original;
//	}

	/**
	 * Inclui as main actions aos processos, tamb�m inclui os estados 
	 * aos processos
	 * 
	 * @return os processos montados
	 */
	public Pair<Pair<ProcessPara, ProcessPara>, Pred> montaProcesso() {

		/* Montando as a��es principais */
		String nome = CircusUtils.DEFAULT_MAIN_ACTION_NAME;
		ZName nomeAcao1 = this.factory.createZName(nome, 
				this.factory.createZStrokeList(), null);
		ActionPara acaoPrincipal1 = 
			this.factory.createActionPara(nomeAcao1, 
					this.getAcaoPrincipalProcesso1());
		ZName nomeAcao2 = 
			(ZName)ClonerVisitor.cloneTermRemovingRelationsStack(nomeAcao1);
		ActionPara acaoPrincipal2 = 
			this.factory.createActionPara(nomeAcao2, 
					this.getAcaoPrincipalProcesso2());

		this.getProcess1().add(acaoPrincipal1);
		this.getProcess2().add(acaoPrincipal2);
		
		/* Adiciona os estados aos processos */
		AxPara estado1 = 
			ClonerVisitor.cloneTermRemovingRelationsStack(this.getStateP1());
		AxPara estado2 = 
			ClonerVisitor.cloneTermRemovingRelationsStack(this.getStateP2());
		estado1.getAnns().add(this.factory.createCircusStateAnn());
		estado2.getAnns().add(this.factory.createCircusStateAnn());
		this.getProcess1().add(0, estado1);
		this.getProcess2().add(0, estado2);

		/* Montando os par�grafos, baseado nas listas */
		BasicProcess basicProcess1 = 
			this.factory.createBasicProcess(this.getProcess1());
		BasicProcess basicProcess2 = 
			this.factory.createBasicProcess(this.getProcess2());

		ProcessPara p1 = 
			this.factory.createProcessPara(this.getNomeProcesso1(),
					this.factory.createZNameList(), basicProcess1);
		ProcessPara p2 = 
			this.factory.createProcessPara(this.getNomeProcesso2(),
					this.factory.createZNameList(), basicProcess2);

		/* Monta as OPs a retornar */
		Pred op = this.montarOP();
		
		Pair<ProcessPara, ProcessPara> paras = 
			new Pair<ProcessPara, ProcessPara>(p1, p2);
		return new Pair<Pair<ProcessPara,ProcessPara>, Pred>(paras, op);
	}
	
	/**
	 * Monta a fun��o alpha da express�o de um par�grafo de estados. 
	 * Utilizado para montar a obriga��o de prova.
	 * 
	 * @param state o par�grafo da defini��o de estado
	 * @return
	 */
	public ApplExpr montarFuncaoAlpha(AxPara state) {
		ConstDecl esq = (ConstDecl) state.getZSchText().getZDeclList().get(0);
		Expr esquema = esq.getExpr();
		RefExpr funAlpha = OPsDischargeUtils.refFuncao(OPsDischargeUtils.ALPHA);
		ApplExpr appFun = 
			this.factory.createApplExpr(Arrays.asList(funAlpha, esquema), 
					false);
		return appFun;
	}
	
	/**
	 * Monta a OP da lei de Process Splitting: 
	 * \alpha(S1) \cap \alpha(S2) = \emptyset
	 * 
	 * @return a OP da lei de Process Splitting
	 */
	public Pred montarOP() {
		
		/* alpha(S1) */
		ApplExpr alphaS1 = this.montarFuncaoAlpha(this.getStateP1());
		
		/* alpha(S2) */
		ApplExpr alphaS2 = this.montarFuncaoAlpha(this.getStateP2());
		
		/* alpha(S1) \cap alpha(S2) */
		ZExprList listaArgs = 
			this.factory.createZExprList(Arrays.asList(alphaS1, alphaS2));
		TupleExpr alphaS1AlphaS2 = this.factory.createTupleExpr(listaArgs);
		RefExpr cap = OPsDischargeUtils.refFuncao(OPsDischargeUtils.CAP);
		ApplExpr capAlphaS1AlphaS2 = 
			this.factory.createApplExpr(Arrays.asList(cap, alphaS1AlphaS2), 
					true);
		
		/* alpha(S1) \cap alpha(S2) = \emptyset */
		RefExpr emptyset = OPsDischargeUtils.criarConjuntoVazio();
		ZExprList conjUnitario = 
			this.factory.createZExprList(Collections.singletonList(emptyset));
		SetExpr conjUnitEmptySet = this.factory.createSetExpr(conjUnitario);
		MemPred op = this.factory.createMemPred(Arrays.asList(capAlphaS1AlphaS2, 
				conjUnitEmptySet), true);
		return op;
	}

	/**
	 * Visita o processo original para montar os outros. Apenas chama 
	 * o m�todo accept da sua lista de par�grafos
	 * 
	 * @param arg0 o processo b�sico a ser visitado
	 */
	public Void visitBasicProcess(BasicProcess arg0) {
		
		/* Captura o estado do processo original */
		AxPara estadoOriginal = (AxPara) arg0.getStatePara();
		
		ConstDecl decl;
		try {
			decl = (ConstDecl) 
				estadoOriginal.getZSchText().getZDeclList().get(0);
		} catch (UnsupportedAstClassException e) {
			
			/* Par�grafo de estado inv�lido */
			throw new MounterException("COD0619");
		}
		
		/* Busca pelos estados dos processos a serem montados */
		Expr exprEstado = decl.getExpr();
		
		/* Testa se o estado � do tipo X \land Y */
		if (exprEstado instanceof AndExpr) {
			AndExpr and = (AndExpr) exprEstado;
			
			/* Testa se o estado � do tipo S1 \land S2 */
			if (and.getLeftExpr() instanceof RefExpr && 
					and.getRightExpr() instanceof RefExpr) {
				
				try {
					
					/* Nomes dos estados */
					ZName nomeS1 = ((RefExpr) and.getLeftExpr()).getZName();
					ZName nomeS2 = ((RefExpr) and.getRightExpr()).getZName();
					
					/* 
					 * Procura as defini��es dos nomes dos par�grafos de 
					 * estado 
					 */
					for (Para para : arg0.getZParaList()) {
						if (para instanceof AxPara) {
							
							AxPara axPara = (AxPara) para;
							
							ConstDecl temp = (ConstDecl) 
								axPara.getZSchText().getZDeclList().get(0);
							if (temp.getZName().equals(nomeS1)) {
								
								/* Achou o 1o. par�grafo */
								this.setStateP1((AxPara) para);
							} else if (temp.getZName().equals(nomeS2)) {
								
								/* Achou o estado do 2o. par�grafo */
								this.setStateP2((AxPara) para);
							}
						}
					}
				} catch (UnsupportedAstClassException e) {
					
					/* Par�grafo de estado inv�lido */
					throw new MounterException("COD0619");
				}
				
			} else {

				/* Par�grafo de estado inv�lido */
				throw new MounterException("COD0619");
			}
			
			
		} else {
			
			/* Par�grafo de estado inv�lido */
			throw new MounterException("COD0619");
		}
		
		
		
		arg0.getZParaList().accept(this);
		return null;
	}
	
	/**
	 * Visita a lista de par�grafos do processo. Visita cada um dos 
	 * par�grafos da lista para encaix�-lo em algum dos processos
	 * 
	 * @param arg0 a lista de par�grafos a ser visitada
	 */
	public Void visitZParaList(ZParaList arg0) {
		for (Para para : arg0) {
			para.accept(this);
		}
		return null;
	}
	
	/**
	 * Visita um par�grafo que declara um conjunto de nomes. Dispara 
	 * uma excess�o, pois a lei n�o pode ser aplicada
	 */
	public Void visitNameSetPara(NameSetPara arg0) {
		throw new MounterException("COD0618");
	}

	/**
	 * Visita um par�grafo de declara��o de a��o. S� o trata se n�o 
	 * for uma a��o de declara��o de a��o principal. Verifica se pertence 
	 * ao primeiro ou ao segundo processo
	 */
	public Void visitActionPara(ActionPara arg0) {

		if (!arg0.getZName().getWord().contains(
				CircusUtils.DEFAULT_MAIN_ACTION_NAME)) {

			ActionPara aAdicionar = (ActionPara)
				ClonerVisitor.cloneTermRemovingRelationsStack(arg0);
			int result = this.whichProcess(aAdicionar);
			/* N�o � a a��o principal */
			if (result == 1) {

				/* Par�grafo pertence ao primeiro processo */
				this.getProcess1().add(aAdicionar);
			} else if (result == 2) {

				/* Par�grafo pertence ao segundo processo */
				this.getProcess2().add(aAdicionar);
			} else {
				
				throw new MounterException("COD0620");
			}
		}

		return null;
	}

	/**
	 * Visita os par�grafos de defini��o de esquemas de um processo
	 * 
	 * @param arg0 o par�grafo a ser visitado 
	 */
	public Void visitAxPara(AxPara arg0) {

		/* V� se � um par�grafo v�lido */
		AxPara aAdicionar = (AxPara)
		ClonerVisitor.cloneTermRemovingRelationsStack(arg0);
		
		ZName nomePara = ParaNameGetter.getNameList(arg0);
		ZName nomeS1 = ParaNameGetter.getNameList(this.getStateP1());
		ZName nomeS2 = ParaNameGetter.getNameList(this.getStateP2());
		
		/* N�o � o par�grafo de estado do processo */
		if (arg0.getAnn(CircusStateAnn.class) == null && 
				!nomePara.equals(nomeS1) && !nomePara.equals(nomeS2)) {
			
			int numeroProcesso = this.whichProcess(aAdicionar);

			/* Adiciona */
			/* v� em qual processo vai ser adicionado */
			if (numeroProcesso == 1) {

				/* Adiciona ao primeiro processo */
				this.getProcess1().add(aAdicionar);
			} else if (numeroProcesso == 2) {

				/* Adiciona ao segundo processo */
				this.getProcess2().add(aAdicionar);
			} else {
				throw new MounterException("COD0620");
			}

		}
		return null;
	}
	
	/**
	 * M�todo para saber se um par�grafo deve ser colocado no primeiro
	 * ou no segundo processo.
	 * 
	 * @param para o par�grafo a verificar
	 * @return 0 caso o par�grafo n�o seja classificado, 1 caso o 
	 *  par�grafo perten�a ao 1o. processo, 2 caso o par�grafo 
	 *  perten�a ao 2o. processo
	 */
	private int whichProcess(ActionPara para) {
		int result;
		
		try {
			
			/* V� se o paragrafo est� no primeiro processo */
			ConstDecl decl1 = (ConstDecl) 
			this.getStateP1().getZSchText().getZDeclList().get(0);
			Expr estado1 = decl1.getExpr();

			ConstDecl decl2 = (ConstDecl) 
			this.getStateP2().getZSchText().getZDeclList().get(0);
			Expr estado2 = decl2.getExpr();

			CircusAction acao = para.getCircusAction();
			
			if (this.isInProcess(acao, estado1)) {
				
				/* � do processo 1 */
				result = 1;
			} else if (this.isInProcess(acao, estado2)) {
				
				/* � do processo 2 */
				result = 2;
			} else {
				
				/* n�o conseguiu ser classificado */
				result = 0;
			}
			
		} catch (UnsupportedAstClassException e) {
			
			/* N�o conseguiu classificar algum dos par�grafos */
			throw new MounterException("COD0620", e);
		}
			

		/* N�o achou da lista */
		return result;
	}
	
	
	/**
	 * M�todo para saber se um par�grafo deve ser colocado no primeiro
	 * ou no segundo processo.
	 * 
	 * @param para o par�grafo a verificar
	 * @return 0 caso o par�grafo n�o seja classificado, 1 caso o 
	 *  par�grafo perten�a ao 1o. processo, 2 caso o par�grafo 
	 *  perten�a ao 2o. processo
	 */
	private int whichProcess(AxPara para) {
		int result;
		
		try {
			
			/* V� se o paragrafo est� no primeiro processo */
			ConstDecl decl1 = (ConstDecl) 
			this.getStateP1().getZSchText().getZDeclList().get(0);
			Expr estado1 = decl1.getExpr();

			ConstDecl decl2 = (ConstDecl) 
			this.getStateP2().getZSchText().getZDeclList().get(0);
			Expr estado2 = decl2.getExpr();

			Expr expr = ((ConstDecl) 
					para.getZSchText().getZDeclList().get(0)).getExpr();
			
			if (this.isInProcess(expr, estado1)) {
				
				/* � do processo 1 */
				result = 1;
			} else if (this.isInProcess(expr, estado2)) {
				
				/* � do processo 2 */
				result = 2;
			} else {
				
				/* n�o conseguiu ser classificado */
				result = 0;
			}
			
		} catch (UnsupportedAstClassException e) {
			
			/* N�o conseguiu classificar algum dos par�grafos */
			throw new MounterException("COD0620", e);
		}
			

		/* N�o achou da lista */
		return result;
	}
	
	/**
	 * V� se uma determinada a��o se refere aos elememtos de um 
	 * estado.
	 * 
	 * @param acao a a��o a ser analisada
	 * @param estado1 o estado do processo a ser deliberado
	 * @return o resultado da avalia��o de 
	 *  wrtV(acao) \subseteq \alpha(estado)
	 */ 
	private boolean isInProcess(CircusAction acao, Expr estado) {
		
		/* usedV(A) */
		RefExpr usedV = OPsDischargeUtils.refFuncao(OPsDischargeUtils.USED_V);
		RefExpr argAcao = this.factory.createRefExpr();
		ActionArgumentAnn ann = new ActionArgumentAnn(acao);
		argAcao.getAnns().add(ann);
		ApplExpr usedVAcao = this.factory.createApplExpr(Arrays.asList(usedV, 
				argAcao), true);
		
		/* alpha(S) */
		RefExpr alpha = OPsDischargeUtils.refFuncao(OPsDischargeUtils.ALPHA);
		ApplExpr alphaEstado = this.factory.createApplExpr(Arrays.asList(alpha, 
				estado), true);
		
		/* wrtV(A) \subseteq alpha(S) */
		RefExpr subseteq = 
			OPsDischargeUtils.refFuncao(OPsDischargeUtils.SUBSETEQ);
		ZExprList lista = this.factory.createZExprList(Arrays.asList(usedVAcao, 
				alphaEstado));
		TupleExpr args = this.factory.createTupleExpr(lista);
		MemPred subseteqArgs = 
			this.factory.createMemPred(Arrays.asList(args, subseteq), true);
		
  		Sequent op = this.factory.createSequent(null, subseteqArgs);
  		Pred opExtendida = OPsDischargeUtils.expandirOP(op, this.getAst(), acao, 
  				acao, gerInterno);
  		
  		boolean validOP;
		try {
			validOP = Provador.proveAux(opExtendida);
		} catch (CannotProveException e) {
			throw new MounterException("COD0620", e);
		}
		
		return validOP;
	}
	
	/**
	 * V� se um determinado esquema usa as vari�veis de um determinado 
	 * estado
	 * 
	 * @param esquema o esquema a ser testado
	 * @param estado o estado base do teste
	 */
	private boolean isInProcess(Expr esquema, Expr estado) {
		SchExprAction acao = this.factory.createSchExprAction(esquema);
		return this.isInProcess(acao, estado);
	}
	
	/**
	 * Excess�o quando o processo n�o pode ser montado
	 * 
	 * @author Cristiano Castro
	 */
	protected class MounterException extends RuntimeException {

		/** N�mero para serializa��o */
		private static final long serialVersionUID = 6553553209830289315L;

		public MounterException() {
			super();
		}

		public MounterException(String message, Throwable cause) {
			super(message, cause);
		}

		public MounterException(String message) {
			super(message);
		}

		public MounterException(Throwable cause) {
			super(cause);
		}
		
	}

}
