/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc@consiste.dimap.ufrn.br>
 */
package circusRefine.core.crules.utils;

import java.util.List;

import circusRefine.core.util.ChildrenTermExtractor;

import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.circus.ast.BasicProcess;
import net.sourceforge.czt.circus.ast.ProcessPara;
import net.sourceforge.czt.circus.visitor.ProcessParaVisitor;
import net.sourceforge.czt.z.ast.ParaList;

/**
 * Classe para retornar o nome do processo a partir da sua definio 
 * 
 * @author Cristiano Castro
 */
public class ProcessParaFromAParaListGetter implements TermVisitor<ProcessPara>, 
	ProcessParaVisitor<ProcessPara> {

	/**
	 * Retorna um nome a partir de uma Lista de pargrafos de um 
	 * processo
	 * 
	 * @param pl a lista de pargrafos
	 * @param ast a rvore no qual ser efetuada a busca
	 * @return o nome do pargrafo no qual est definido o 
	 *  {@link BasicProcess}, ou <code>null</code> caso este no 
	 *  exista.
	 */
	public static ProcessPara getName(ParaList pl, Term ast) {
		ProcessParaFromAParaListGetter visitor = 
			new ProcessParaFromAParaListGetter(pl);
		return ast.accept(visitor);
	}
	
	/** Pargrafos de um processo */
	private ParaList paragrafos;
	
	/**
	 * Construtor que recebe a {@link ParaList} com os pargrafos do
	 * processo a ter o nome buscado
	 * 
	 * @param a a lista de pargrafos
	 */
	public ProcessParaFromAParaListGetter(ParaList a) {
		this.setParagrafos(a);
	}

	private ParaList getParagrafos() {
		return paragrafos;
	}
	
	private void setParagrafos(ParaList paragrafos) {
		this.paragrafos = paragrafos;
	}
	
	public ProcessPara visitProcessPara(ProcessPara arg0) {
		
		/* Testa se a lista de pargrafos bate com o processo 
		 * procurado */
		if (arg0.getCircusProcess() instanceof BasicProcess) {
			BasicProcess bP = (BasicProcess)arg0.getCircusProcess();
			
			/* Testa se o processo era o que se estava procurando */
			if (bP.getParaList().equals(this.getParagrafos())) {
				return arg0;
			}
		}
		
		/* No era o processo procurado */
		return null;
	}

	public ProcessPara visitTerm(Term arg0) {
		
		/* busca dos resultados dos filhos */
		List<Term> filhos = ChildrenTermExtractor.extrairFilhos(arg0);
		
		/* Busca pelo nome nos filhos */
		for (Term filho : filhos) {
			ProcessPara result = filho.accept(this);
			if (result != null) {
				return result;
			}
		}
		
		/* No achou o nome */
		return null;
	}
	

}
