/*
 * Projeto: Circus Refine
 */
package circusRefine.core.crules.utils;

import java.util.Arrays;

import net.sourceforge.czt.circus.ast.CallProcess;
import net.sourceforge.czt.circus.ast.ChannelSet;
import net.sourceforge.czt.circus.ast.CircusAction;
import net.sourceforge.czt.circus.ast.CircusProcess;
import net.sourceforge.czt.circus.ast.ExtChoiceAction;
import net.sourceforge.czt.circus.ast.ExtChoiceActionIte;
import net.sourceforge.czt.circus.ast.HideAction;
import net.sourceforge.czt.circus.ast.IntChoiceAction;
import net.sourceforge.czt.circus.ast.IntChoiceActionIte;
import net.sourceforge.czt.circus.ast.InterleaveAction;
import net.sourceforge.czt.circus.ast.InterleaveActionIte;
import net.sourceforge.czt.circus.ast.ParallelAction;
import net.sourceforge.czt.circus.ast.ParallelActionIte;
import net.sourceforge.czt.circus.ast.SeqAction;
import net.sourceforge.czt.circus.ast.SeqActionIte;
import net.sourceforge.czt.circus.visitor.CircusActionVisitor;
import net.sourceforge.czt.circus.visitor.ExtChoiceActionIteVisitor;
import net.sourceforge.czt.circus.visitor.ExtChoiceActionVisitor;
import net.sourceforge.czt.circus.visitor.HideActionVisitor;
import net.sourceforge.czt.circus.visitor.IntChoiceActionIteVisitor;
import net.sourceforge.czt.circus.visitor.IntChoiceActionVisitor;
import net.sourceforge.czt.circus.visitor.InterleaveActionIteVisitor;
import net.sourceforge.czt.circus.visitor.InterleaveActionVisitor;
import net.sourceforge.czt.circus.visitor.ParallelActionIteVisitor;
import net.sourceforge.czt.circus.visitor.ParallelActionVisitor;
import net.sourceforge.czt.circus.visitor.SeqActionIteVisitor;
import net.sourceforge.czt.circus.visitor.SeqActionVisitor;
import net.sourceforge.czt.circuspatt.ast.CircusPatternFactory;
import net.sourceforge.czt.circuspatt.impl.CircusPatternFactoryImpl;
import net.sourceforge.czt.z.ast.DeclList;
import net.sourceforge.czt.z.ast.ZName;
import circusRefine.core.util.ClonerVisitor;

/**
 * Classe utilizada na aplica��o da lei Process Splitting. Monta a
 * nova forma do processo original de acordo com a a��o principal
 * anterior
 * 
 * @author Cristiano Castro
 */
public class ProcessFromMainActionMounter implements 
		CircusActionVisitor<CircusProcess>, SeqActionVisitor<CircusProcess>, 
		ExtChoiceActionVisitor<CircusProcess>, 
		IntChoiceActionVisitor<CircusProcess>, 
		ParallelActionVisitor<CircusProcess>, 
		InterleaveActionVisitor<CircusProcess>, 
		HideActionVisitor<CircusProcess>,
		SeqActionIteVisitor<CircusProcess>, 
		ExtChoiceActionIteVisitor<CircusProcess>,
		IntChoiceActionIteVisitor<CircusProcess>, 
		ParallelActionIteVisitor<CircusProcess>,
		InterleaveActionIteVisitor<CircusProcess> {

	/**
	 * M�todo utilizado para montar o novo processo de acordo com a 
	 * main Action do processo anterior
	 * 
	 * @param main a main action
	 * @param left a a��o principal do primeiro processo
	 * @param right a a��o principal do segundo processo
	 * @param pLeft o nome do primeiro processo
	 * @param pRight o nome do segundo processo
	 * @return o novo processo montado
	 * @throws InvalidMainAction caso a a��o principal n�o possa ser 
	 *  mapeada em um processo
	 */
	public static CircusProcess montarProcesso(CircusAction main, 
			CircusAction left, CircusAction right, ZName pLeft, ZName pRight) {
		ProcessFromMainActionMounter visitor = 
			new ProcessFromMainActionMounter(left, right, pLeft, pRight);
		CircusProcess result = main.accept(visitor);
		return result;
	}
	
	
	public static CircusProcess montarProcessoTactic(CircusAction main, 
			CircusAction left, ZName pLeft) {
		ProcessFromMainActionMounter visitor = 
			new ProcessFromMainActionMounter(left, pLeft);
		CircusProcess result = main.accept(visitor);
		return result;
	}
	
	/** A��o principal do primeiro processo criado */
	private CircusAction leftAction;
	
	/** A��o principal do segundo processo criado */
	private CircusAction rightAction;
	
	/** Nome do processo da esquerda */
	private ZName nomeLeft;
	
	/** Nome do processo da direita */
	private ZName nomeRight;
	
	/** F�brica */
	private CircusPatternFactory factory;
	
	/**
	 * Inicia o montador com as a��es principais e os nomes do 
	 * primeiro e do segundo processo
	 * 
	 * @param leftAction a a��o principal do primeiro processo
	 * @param rightAction a a��o principal do segundo processo
	 * @param nomeLeft o nome do primeiro processo
	 * @param nomeRight o nome do segundo processo
	 */
	public ProcessFromMainActionMounter(CircusAction leftAction, 
			CircusAction rightAction, ZName nomeLeft, ZName nomeRight) {
		this.setLeftAction(leftAction);
		this.setRightAction(rightAction);
		this.setNomeLeft(nomeLeft);
		this.setNomeRight(nomeRight);
		this.setFactory(new CircusPatternFactoryImpl());
	}
	
	public ProcessFromMainActionMounter(CircusAction leftAction, 
			 ZName nomeLeft) {
		this.setLeftAction(leftAction);
		this.setNomeLeft(nomeLeft);
		this.setFactory(new CircusPatternFactoryImpl());
	}

	/**
	 * @return the leftAction
	 */
	private CircusAction getLeftAction() {
		return leftAction;
	}

	/**
	 * @param leftAction the leftAction to set
	 */
	private void setLeftAction(CircusAction leftAction) {
		this.leftAction = leftAction;
	}

	/**
	 * @return the nomeLeft
	 */
	private ZName getNomeLeft() {
		return nomeLeft;
	}

	/**
	 * @param nomeLeft the nomeLeft to set
	 */
	private void setNomeLeft(ZName nomeLeft) {
		this.nomeLeft = nomeLeft;
	}

	/**
	 * @return the nomeRight
	 */
	private ZName getNomeRight() {
		return nomeRight;
	}

	/**
	 * @param nomeRight the nomeRight to set
	 */
	private void setNomeRight(ZName nomeRight) {
		this.nomeRight = nomeRight;
	}

	/**
	 * @return the rightAction
	 */
	private CircusAction getRightAction() {
		return rightAction;
	}

	/**
	 * @param rightAction the rightAction to set
	 */
	private void setRightAction(CircusAction rightAction) {
		this.rightAction = rightAction;
	}

	/**
	 * @return the factory
	 */
	private CircusPatternFactory getFactory() {
		return factory;
	}

	/**
	 * @param factory the factory to set
	 */
	private void setFactory(CircusPatternFactory factory) {
		this.factory = factory;
	}

	/**
	 * Qualquer outra a��o que n�o seja predefinida
	 */
	public CircusProcess visitCircusAction(CircusAction arg0) {
		CallProcess teste = this.testarCasoBase(arg0);
		CircusProcess result;
		
		/* Efetua o teste do caso base */
		if (teste == null) {
			
			/* A��o n�o pode ser mapeada no processo */
			throw new InvalidMainAction();
			
		} else {
			
			/* Caso base */
			result = teste;
		}
		return result;
	}

	/**
	 * Testa o caso base. Caso n�o seja, monta a composi��o sequencial
	 * dos processos.
	 * 
	 * @param arg0 a a��o de composi��o sequencial a ser visitada
	 * @return retorna o novo processo montado
	 */
	public CircusProcess visitSeqAction(SeqAction arg0) {
		CallProcess teste = this.testarCasoBase(arg0);
		CircusProcess result;
		
		/* Efetua o teste do caso base */
		if (teste == null) {
			
			/* Monta a sequencia de processo */
			CircusProcess left = arg0.getLeftAction().accept(this);
			CircusProcess right = arg0.getRightAction().accept(this);
			result = 
				this.getFactory().createSeqProcess(Arrays.asList(left, right));
		} else {
			
			/* Caso base */
			result = teste;
		}
		return result;
	}

	/**
	 * Testa o caso base. Caso n�o seja, monta a escolha externa dos 
	 * processos.
	 * 
	 * @param arg0 a a��o de escolha externa a ser visitada
	 * @return retorna o novo processo montado
	 */
	public CircusProcess visitExtChoiceAction(ExtChoiceAction arg0) {
		CallProcess teste = this.testarCasoBase(arg0);
		CircusProcess result;
		
		/* Efetua o teste do caso base */
		if (teste == null) {
			
			/* Monta a sequencia de processo */
			CircusProcess left = arg0.getLeftAction().accept(this);
			CircusProcess right = arg0.getRightAction().accept(this);
			result = 
				this.getFactory().createExtChoiceProcess(Arrays.asList(left, 
						right));
		} else {
			
			/* Caso base */
			result = teste;
		}
		return result;
	}

	/**
	 * Testa o caso base. Caso n�o seja, monta a escolha interna dos 
	 * processos.
	 * 
	 * @param arg0 a a��o de escolha interna a ser visitada
	 * @return retorna o novo processo montado
	 */
	public CircusProcess visitIntChoiceAction(IntChoiceAction arg0) {
		CallProcess teste = this.testarCasoBase(arg0);
		CircusProcess result;
		
		/* Efetua o teste do caso base */
		if (teste == null) {
			
			/* Monta a sequencia de processo */
			CircusProcess left = arg0.getLeftAction().accept(this);
			CircusProcess right = arg0.getRightAction().accept(this);
			result = 
				this.getFactory().createIntChoiceProcess(Arrays.asList(left, 
						right));
		} else {
			
			/* Caso base */
			result = teste;
		}
		return result;
	}

	/**
	 * Testa o caso base. Caso n�o seja, monta a a��o paralela dos 
	 * processos.
	 * 
	 * @param arg0 a a��o paralela a ser visitada
	 * @return retorna o novo processo montado
	 */
	public CircusProcess visitParallelAction(ParallelAction arg0) {
		CallProcess teste = this.testarCasoBase(arg0);
		CircusProcess result;
		
		/* Efetua o teste do caso base */
		if (teste == null) {
			
			/* Monta a sequencia de processo */
			CircusProcess left = arg0.getLeftAction().accept(this);
			CircusProcess right = arg0.getRightAction().accept(this);
			ChannelSet cs = (ChannelSet)
			ClonerVisitor.cloneTermRemovingRelationsStack(arg0.getChannelSet());
			result = 
				this.getFactory().createParallelProcess(Arrays.asList(left, 
						right), cs);
		} else {
			
			/* Caso base */
			result = teste;
		}
		return result;
	}

	/**
	 * Testa o caso base. Caso n�o seja, monta a a��o interleave dos 
	 * processos.
	 * 
	 * @param arg0 a a��o de interleave a ser visitada
	 * @return retorna o novo processo montado
	 */
	public CircusProcess visitInterleaveAction(InterleaveAction arg0) {
		CallProcess teste = this.testarCasoBase(arg0);
		CircusProcess result;
		
		/* Efetua o teste do caso base */
		if (teste == null) {
			
			/* Monta a sequencia de processo */
			CircusProcess left = arg0.getLeftAction().accept(this);
			CircusProcess right = arg0.getRightAction().accept(this);
			result = 
				this.getFactory().createInterleaveProcess(Arrays.asList(left, 
						right));
		} else {
			
			/* Caso base */
			result = teste;
		}
		return result;
	}

	/**
	 * Testa o caso base. Caso n�o seja, monta a a��o hide dos 
	 * processos.
	 * 
	 * @param arg0 a a��o de hide a ser visitada
	 * @return retorna o novo processo montado
	 */
	public CircusProcess visitHideAction(HideAction arg0) {
		CallProcess teste = this.testarCasoBase(arg0);
		CircusProcess result;
		
		/* Efetua o teste do caso base */
		if (teste == null) {
			
			/* Monta a sequencia de processo */
			CircusProcess processo = arg0.getCircusAction().accept(this);
			ChannelSet cs = arg0.getChannelSet();
			cs = ClonerVisitor.cloneTermRemovingRelationsStack(cs);
			result = this.getFactory().createHideProcess(processo, cs);
		} else {
			
			/* Caso base */
			result = teste;
		}
		return result;
	}

	/**
	 * Testa o caso base. Caso n�o seja, monta sequ�ncia dos 
	 * processos.
	 * 
	 * @param arg0 a sequencia de a��es a ser visitada
	 * @return retorna o novo processo montado
	 */
	public CircusProcess visitSeqActionIte(SeqActionIte arg0) {
		CallProcess teste = this.testarCasoBase(arg0);
		CircusProcess result;
		
		/* Efetua o teste do caso base */
		if (teste == null) {
			
			/* Monta a sequencia de processo */
			CircusProcess processo = arg0.getCircusAction().accept(this);
			DeclList declaracoes = arg0.getDeclList();
			declaracoes = 
				ClonerVisitor.cloneTermRemovingRelationsStack(declaracoes);
			result = 
				this.getFactory().createSeqProcessIte(processo, declaracoes);
		} else {
			
			/* Caso base */
			result = teste;
		}
		return result;
	}

	/**
	 * Testa o caso base. Caso n�o seja, monta a escolha externa 
	 * iterada dos processos
	 * 
	 * @param arg0 a a��o de escolha externa iterada a ser visitada
	 * @return retorna o novo processo montado
	 */
	public CircusProcess visitExtChoiceActionIte(ExtChoiceActionIte arg0) {
		CallProcess teste = this.testarCasoBase(arg0);
		CircusProcess result;
		
		/* Efetua o teste do caso base */
		if (teste == null) {
			
			/* Monta a sequencia de processo */
			CircusProcess processo = arg0.getCircusAction().accept(this);
			DeclList declaracoes = arg0.getDeclList();
			declaracoes = 
				ClonerVisitor.cloneTermRemovingRelationsStack(declaracoes);
			result = 
				this.getFactory().createExtChoiceProcessIte(processo, 
						declaracoes);
		} else {
			
			/* Caso base */
			result = teste;
		}
		return result;
	}

	/**
	 * Testa o caso base. Caso n�o seja, monta a escolha interna
	 * dos processos
	 * 
	 * @param arg0 a a��o de escolha interna a ser visitada
	 * @return retorna o novo processo montado
	 */
	public CircusProcess visitIntChoiceActionIte(IntChoiceActionIte arg0) {
		CallProcess teste = this.testarCasoBase(arg0);
		CircusProcess result;
		
		/* Efetua o teste do caso base */
		if (teste == null) {
			
			/* Monta a sequencia de processo */
			CircusProcess processo = arg0.getCircusAction().accept(this);
			DeclList declaracoes = arg0.getDeclList();
			declaracoes = 
				ClonerVisitor.cloneTermRemovingRelationsStack(declaracoes);
			result = 
				this.getFactory().createIntChoiceProcessIte(processo, 
						declaracoes);
		} else {
			
			/* Caso base */
			result = teste;
		}
		return result;
	}

	/**
	 * Testa o caso base. Caso n�o seja, monta o paralelismo iterado 
	 * dos processos.
	 * 
	 * @param arg0 a a��o de paralelismo iterado a ser visitada
	 * @return retorna o novo processo montado
	 */
	public CircusProcess visitParallelActionIte(ParallelActionIte arg0) {
		CallProcess teste = this.testarCasoBase(arg0);
		CircusProcess result;
		
		/* Efetua o teste do caso base */
		if (teste == null) {
			
			/* Monta a sequencia de processo */
			CircusProcess processo = arg0.getCircusAction().accept(this);
			DeclList dl = arg0.getDeclList();
			dl = ClonerVisitor.cloneTermRemovingRelationsStack(dl);
			ChannelSet cs = arg0.getChannelSet();
			cs = ClonerVisitor.cloneTermRemovingRelationsStack(cs);
			result = 
				this.getFactory().createParallelProcessIte(processo, dl, cs);
		} else {
			
			/* Caso base */
			result = teste;
		}
		return result;
	}

	/**
	 * Testa o caso base. Caso n�o seja, monta a a��o hide dos 
	 * processos.
	 * 
	 * @param arg0 a a��o de hide a ser visitada
	 * @return retorna o novo processo montado
	 */
	public CircusProcess visitInterleaveActionIte(InterleaveActionIte arg0) {
		CallProcess teste = this.testarCasoBase(arg0);
		CircusProcess result;
		
		/* Efetua o teste do caso base */
		if (teste == null) {
			
			/* Monta a sequencia de processo */
			CircusProcess processo = arg0.getCircusAction().accept(this);
			DeclList dl = arg0.getDeclList();
			dl = ClonerVisitor.cloneTermRemovingRelationsStack(dl);
			result = this.getFactory().createInterleaveProcessIte(processo, dl);
		} else {
			
			/* Caso base */
			result = teste;
		}
		return result;
	}
	
	/**
	 * Testa o caso base da recurs�o para montagem do processo a 
	 * partir da a��o principal. Compara a a��o passada como par�metro
	 * com as a��es do primeiro e do segundo processo
	 * 
	 * @param acao a a��o a ser testada
	 * @return o processo da esquerda, caso a a��o principal desse 
	 *  seja igual ao par�metro; o processo da direita, caso a a��o 
	 *  principal desse seja igual ao par�metro ou <code>null</code>
	 *  se a a��o passada n�o for igual a nenhum dos dois   
	 */
	private CallProcess testarCasoBase(CircusAction acao) {
		CallProcess result;
		if (acao.equals(this.getLeftAction())) {
			result = this.getFactory().createCallProcess(this.getNomeLeft());
		} else if (acao.equals(this.getRightAction())) {
			result = this.getFactory().createCallProcess(this.getNomeRight());
		} else {
			result = null;
		}
		return result;
	}

	/**
	 * Excess�o privada gerada quando a MainAction do processo 
	 * original n�o pode ser mapeada na defini��o do novo processo.
	 * 
	 * @author Cristiano Castro
	 */
	private class InvalidMainAction extends RuntimeException {

		private static final long serialVersionUID = -6868470686261056621L;
		
	}
	
}
