/*
 * Projeto: Circus Refine
 * 
 * Autores: Alessandro Gurgel <alessandro87@consiste.dimap.ufrn.br>
 * 			Cristiano Castro  <crisgc1@gmail.com>
 */
package circusRefine.core.crules.utils;

import java.util.Arrays;

import circusRefine.core.astmodifiers.ActionArgumentAnn;
import circusRefine.core.opsdischarge.OPsDischargeUtils;
import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.base.visitor.VisitorUtils;
import net.sourceforge.czt.circuspatt.ast.CircusPatternFactory;
import net.sourceforge.czt.circuspatt.impl.CircusPatternFactoryImpl;
import net.sourceforge.czt.z.ast.ApplExpr;
import net.sourceforge.czt.z.ast.DecorExpr;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.Name;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.Stroke;
import net.sourceforge.czt.z.ast.StrokeList;
import net.sourceforge.czt.z.ast.ZName;
import net.sourceforge.czt.z.ast.ZStrokeList;
import net.sourceforge.czt.z.util.ZString;
import net.sourceforge.czt.z.visitor.RefExprVisitor;

/**
 * Classe para gerar uma "forma normal" da AST, til para aplicao de 
 * lei em termos
 * 
 * @author Cristiano Castro
 */
public class NormalASTGenerator implements TermVisitor<Term>, 
RefExprVisitor<Term> {
	
	/** Fbrica */
	private CircusPatternFactory factory = new CircusPatternFactoryImpl();

	/**
	 * Gera uma forma normal da AST passada como parmetro. til para
	 * a resoluo de OPs
	 * 
	 * @param ast1 a AST a ter a forma normal gerada
	 */
	public static Term gerarFormaNormal(Term ast1) {
		NormalASTGenerator visitor = new NormalASTGenerator();
		return ast1.accept(visitor);
	}
	
	/**
	 * No faz alterao em um termo genrico
	 * 
	 * @param arg0 o termo a ser visitado
	 */
	public Term visitTerm(Term arg0) {
		Term result = VisitorUtils.visitTerm(this, arg0, true);
		return result;
	}
	
	/**
	 * Confere se a {@link RefExpr} tem uma anotao de argumentos de
	 * aes. Os Termos do tipo \Delta AlgumEsquema, so transformados 
	 * em aplicao de funo \Delta(AlgumEsquema), o mesmo acontece 
	 * com o \Xi. Os nomes com Strokes so transformados em DecorExpr 
	 * aninhados.
	 * 
	 * @param arg0 a referncia a ser visitada
	 * @return uma normalizao da referncia ou da ao do termo caso 
	 *  a referncia contenha uma anotao de argumento de ao 
	 */
	public Expr visitRefExpr(RefExpr arg0) {
		Expr result;
		ActionArgumentAnn ann = arg0.getAnn(ActionArgumentAnn.class);
		if (ann == null) {
			Name nomeReferenciado = arg0.getName(); 
			if (nomeReferenciado instanceof ZName) {
				ZName nomeRefZ = (ZName) nomeReferenciado;
				String strNome = nomeRefZ.getWord();
				
				
				/* V se a String identifica alguma funo */
				if (strNome.startsWith(ZString.DELTA)) {
					result = this.transfomarEmFuncao(arg0, ZString.DELTA);
				} else if (strNome.startsWith(ZString.XI)){
					result = this.transfomarEmFuncao(arg0, ZString.XI);
				} else {
					
					/* No identifica funo mas o nome tem Strokes */
					StrokeList listaStrokesNome = nomeRefZ.getStrokeList();
					
					if (listaStrokesNome instanceof ZStrokeList) {
						ZStrokeList listaStrokesZ = (ZStrokeList) 
							listaStrokesNome;
						
						if (!listaStrokesZ.isEmpty()) {
							
							/* Tira um stroke da lista */
							Stroke stk = 
								listaStrokesZ.remove(listaStrokesZ.size() - 1);
							
							/* Visita a referncia recursivamente */
							Term ref = arg0.accept(this);
							
							/* Monta a ao decorada */
							DecorExpr decor = 
								this.factory.createDecorExpr((Expr) ref, stk); 
							result = decor;
						} else {
							
							/* Lista de Strokes  vazia */
							result = VisitorUtils.visitTerm(this, arg0, true);
						}
						
					} else {
						
						/* A lista de Strokes no  vlida */
						result = VisitorUtils.visitTerm(this, arg0, true);
					}
				}
				
			} else {
				
				/* Nome no  um ZName */
				result = VisitorUtils.visitTerm(this, arg0, true);
			}
		} else {
			
			/* Contm uma anotao de ao */
			ann.setTerm(ann.getTerm().accept(this));
			result = arg0;
		}
		
		return result;
	}
	
	/**
	 * 
	 * @param referencia
	 * @param funcao
	 * @return
	 */
	private ApplExpr transfomarEmFuncao(RefExpr referencia, 
			String strNomeFuncao) {
		ZName nomeRefZ = referencia.getZName();
		String strNome = nomeRefZ.getWord();
		String novoNome = strNome.substring(ZString.DELTA.length());
		nomeRefZ.setWord(novoNome);
		
		/* Cria funo Delta e argumento da funo Delta */
		RefExpr funDelta = OPsDischargeUtils.refFuncao(ZString.DELTA);
		referencia.setName(nomeRefZ);
		
		/* Retorna a aplicao da funo */
		ApplExpr aplicacao = 
			this.factory.createApplExpr(Arrays.asList(funDelta, referencia), 
					true);
		
		/* Continua a visitao no argumento */
		aplicacao.setRightExpr((Expr) aplicacao.getRightExpr().accept(this));
		return aplicacao;
	
	}
}