/*
 * Projeto: Circus Refine
 */
package circusRefine.core.crules.utils;

import java.util.LinkedList;
import java.util.List;

import net.sourceforge.czt.base.ast.Term;
import net.sourceforge.czt.base.visitor.TermVisitor;
import net.sourceforge.czt.circus.ast.CallAction;
import net.sourceforge.czt.circus.ast.CircusChannelSet;
import net.sourceforge.czt.circus.ast.CircusNameSet;
import net.sourceforge.czt.circus.visitor.CallActionVisitor;
import net.sourceforge.czt.circus.visitor.CircusChannelSetVisitor;
import net.sourceforge.czt.circus.visitor.CircusNameSetVisitor;
import net.sourceforge.czt.z.ast.DecorExpr;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.ZName;
import net.sourceforge.czt.z.visitor.DecorExprVisitor;
import net.sourceforge.czt.z.visitor.RefExprVisitor;
import circusRefine.core.util.ChildrenTermExtractor;
import circusRefine.util.Pair;

/**
 * Classe utilizada para buscar todas as referências a nomes de um 
 * termo do CRefine, incluindo Decorated Expressions
 * 
 * @author Cristiano Castro, Alessandro
 */
public class NamesDecFinder implements TermVisitor<List<Pair<ZName, Term>>>, 
CircusChannelSetVisitor<List<Pair<ZName, Term>>>,
CallActionVisitor<List<Pair<ZName, Term>>>, 
RefExprVisitor<List<Pair<ZName, Term>>>,
CircusNameSetVisitor<List<Pair<ZName, Term>>>,
DecorExprVisitor<List<Pair<ZName, Term>>>{

	/**
	 * Busca todos os nomes cujas defini��es devem ser procuradas
	 * 
	 * @param ast a AST a ser vasculhada
	 * @return a lista de pares nome/classes com os nomes cujas
	 *  defini��es devem ser procuradas
	 */
	public static List<Pair<ZName, Term>> findAllName(Term ast) {
		NamesDecFinder visitor = new NamesDecFinder();
		return ast.accept(visitor);
	}
	
	/**
	 * Percorre os filhos do Termo para buscar por nomes
	 * 
	 * @param arg0 a sub-�rvore a ser pesquisada
	 * @return a lista com os pares nome/refer�ncia pesquisados
	 */
	public List<Pair<ZName, Term>> visitTerm(Term arg0) {
		List<Term> filhos = ChildrenTermExtractor.extrairFilhos(arg0);
		List<Pair<ZName, Term>> result = new LinkedList<Pair<ZName,Term>>();

		for (Term filho : filhos) {
			result.addAll(filho.accept(this));
		}

		/* N�o achou a defini��o */
		return result;
	}

	/**
	 * Pesquisa uma refer�ncia a um nome de conjunto de canais
	 * 
	 * @param arg0 o conjunto de canais a ser visitado
	 * @return um par nome/classe caso o conjunto de canais fa�a 
	 *  refer�ncia a um nome declarado anteriormente
	 */
	public List<Pair<ZName,Term>> visitCircusChannelSet(CircusChannelSet arg0) {
		Expr cs = arg0.getExpr();
		List<Pair<ZName, Term>> result = new LinkedList<Pair<ZName,Term>>();

		if (cs instanceof RefExpr) {

			/* Conjunto de canais � definido pelo nome */
			if (((RefExpr)cs).getName() instanceof ZName) {
				ZName nome = ((RefExpr)cs).getZName();
				CircusChannelSet classeNome = arg0;
				Pair<ZName, Term> parNome = 
					new Pair<ZName, Term>(nome, classeNome);
				result.add(parNome);
			}
		} 

		return result;
	}

	/**
	 * Visita uma refer�ncia ao nome de uma a��o
	 * 
	 * @param arg0 a a��o a ser visitada
	 * @return a lista unit�ria com o nome e a classe da a��o
	 */
	public List<Pair<ZName, Term>> visitCallAction(CallAction arg0) {
		List<Pair<ZName, Term>> result = new LinkedList<Pair<ZName,Term>>();
		if (arg0.getName() instanceof ZName) {
			Pair<ZName, Term> parNome = 
				new Pair<ZName, Term>(arg0.getZName(), arg0);
			result.add(parNome);
		} 
		return result;
	}

	/**
	 * Retorna uma refer�ncia para ser buscada a defini��o
	 * 
	 * @param arg0 a refer�ncia visitada
	 * @return o par com o nome a ser buscado
	 */
	public List<Pair<ZName, Term>> visitRefExpr(RefExpr arg0) {
		List<Pair<ZName, Term>> result = new LinkedList<Pair<ZName,Term>>();
		if (arg0.getName() instanceof ZName) {
			result.add(new Pair<ZName, Term>(arg0.getZName(), arg0));
		}
		return result;
	}

	/**
	 * Testa se o conjunto de nomes faz refer�ncia a um conjunto de 
	 * nomes declarado anteriormente
	 * 
	 * @param arg0 o conjunto de nomes a ser testado
	 * @return uma lista com a refer�ncia ao nome caso esta existas
	 */
	public List<Pair<ZName, Term>> visitCircusNameSet(CircusNameSet arg0) {
		List<Pair<ZName, Term>> result = new LinkedList<Pair<ZName,Term>>();
		Expr cs = arg0.getExpr();
		
		if (cs instanceof RefExpr) {
			
			/* Conjunto de canais � definido pelo nome */
			RefExpr ref = (RefExpr)cs;
			
			/* Adiciona o par formado */
			if (ref.getName() instanceof ZName) {
				Pair<ZName, Term> parNome = 
					new Pair<ZName, Term>(ref.getZName(), arg0);
				result.add(parNome);
			}
		}
		
		return result;
	}

	/**
	 * Essas expressões contém nomes que podem ter 
	 */
	public List<Pair<ZName, Term>> visitDecorExpr(DecorExpr arg0) {
		List<Pair<ZName, Term>> result = new LinkedList<Pair<ZName,Term>>();
		if (arg0.getExpr() instanceof RefExpr) {
			RefExpr ref = (RefExpr)arg0.getExpr(); 
			ZName name = ref.getZName();
			name.getZStrokeList().add(arg0.getStroke());
			Pair<ZName, Term> parNome = 
				new Pair<ZName, Term>(name, arg0);
			result.add(parNome);
		}else {
			List<Term> filhos = ChildrenTermExtractor.extrairFilhos(arg0);

			for (Term filho : filhos) {
				result.addAll(filho.accept(this));
			}

		}
		return result;
	}

}
